// Global variables passed from PHP (brm_vars object is assumed to be defined by wp_localize_script)
const REST_ROOT = brm_vars.root;
const REST_NAMESPACE = brm_vars.namespace;
const REST_NONCE = brm_vars.nonce;

const APP_STATE = {
	unmatchedUrls: [],
	currentUrls: [],
	isLoading: false,
	isProcessing: false,
	filterText: '',
	totalCsvUrls: 0,
	unmatchedCount: 0,
	errors: null,
};

/**
 * Reads the uploaded CSV file content as a plain string.
 * @param {File} file
 * @returns {Promise<string>}
 */
function readCsvFile(file) {
	return new Promise((resolve, reject) => {
		const reader = new FileReader();
		reader.onload = (event) => resolve(event.target.result);
		reader.onerror = (error) => reject(error);
		reader.readAsText(file);
	});
}

/**
 * Handles the full upload and comparison process.
 * @param {Event} e
 */
async function handleUpload(e) {
	e.preventDefault();
	const fileInput = document.getElementById('csv_file');
	const file = fileInput.files[0];

	if (!file) {
		APP_STATE.errors = 'Please select a CSV file.';
		renderApp();
		return;
	}

	if (file.type !== 'text/csv' && !file.name.endsWith('.csv')) {
		 APP_STATE.errors = 'File must be a CSV format.';
		 renderApp();
		 return;
	}

	try {
		updateState({ isLoading: true, errors: null, unmatchedUrls: [], totalCsvUrls: 0, unmatchedCount: 0 });
		const csvData = await readCsvFile(file);

		const url = `${REST_ROOT}${REST_NAMESPACE}/upload-csv`;
		const response = await fetch(url, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/json',
				'X-WP-Nonce': REST_NONCE,
			},
			body: JSON.stringify({ csv_data: csvData }),
		});

		if (!response.ok) {
			const errorData = await response.json();
			throw new Error(errorData.message || `API Error: ${response.statusText}`);
		}

		const result = await response.json();
		updateState({
			unmatchedUrls: result.unmatched_urls,
			currentUrls: result.current_urls,
			totalCsvUrls: result.total_csv_urls,
			unmatchedCount: result.count,
			isLoading: false,
			errors: null
		});

		// Auto-suggest logic: Simple prefix match for current URLs
		const suggestedUrls = APP_STATE.currentUrls.filter(u => u.length > 5); // Simple filter
		APP_STATE.unmatchedUrls.forEach(item => {
			const oldPath = item.old_url.replace(/^(https?:\/\/[^\/]+)/, ''); // remove domain
			const suggestion = suggestedUrls.find(newUrl => {
				// Look for common slugs or fragments
				const oldSlug = oldPath.split('/').pop().replace('.html', '').replace('.php', '');
				return newUrl.includes(oldSlug) && oldSlug.length > 3;
			});
			item.suggested_target = suggestion || '';
		});

		renderApp();

	} catch (error) {
		console.error('Upload Error:', error);
		updateState({
			isLoading: false,
			errors: `Failed to process CSV: ${error.message}`
		});
	}
}

/**
 * Creates a single redirect entry via the REST API.
 * @param {string} sourceUrl
 * @param {string} targetUrl
 */
async function createRedirect(sourceUrl, targetUrl, index) {
	if (!sourceUrl || !targetUrl) {
		showError(`Error: Both source and target are required for redirect ${index}.`);
		return;
	}

	try {
		// Set processing state for the specific item
		APP_STATE.unmatchedUrls[index].isSaving = true;
		renderApp();

		const url = `${REST_ROOT}${REST_NAMESPACE}/create-redirect`;
		const response = await fetch(url, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/json',
				'X-WP-Nonce': REST_NONCE,
			},
			body: JSON.stringify({
				source_url: sourceUrl,
				target_url: targetUrl
			}),
		});

		if (!response.ok) {
			const errorData = await response.json();
			throw new Error(errorData.details || errorData.message || `API Error: ${response.statusText}`);
		}

		// Remove the successfully mapped URL from the list
		APP_STATE.unmatchedUrls.splice(index, 1);
		APP_STATE.unmatchedCount = APP_STATE.unmatchedUrls.length;
		showSuccess(`Redirect created for ${sourceUrl} to ${targetUrl}.`);
		renderApp();

	} catch (error) {
		// Restore state and show error
		APP_STATE.unmatchedUrls[index].isSaving = false;
		showError(`Failed to create redirect for ${sourceUrl}: ${error.message}`);
		renderApp();
	}
}

/**
 * Updates the application state and triggers a re-render.
 * @param {Object} newState
 */
function updateState(newState) {
	Object.assign(APP_STATE, newState);
	renderApp();
}

/**
 * Shows a temporary status message (success/error).
 * @param {string} message
 * @param {boolean} isSuccess
 */
function showStatus(message, isSuccess = true) {
	const statusBox = document.getElementById('brm-status-box');
	statusBox.innerHTML = `
		<div class="p-3 mb-4 rounded-lg text-sm ${isSuccess ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}">
			${message}
		</div>
	`;
	setTimeout(() => statusBox.innerHTML = '', 7000); // Clear after 7 seconds
}
const showError = (message) => showStatus(message, false);
const showSuccess = (message) => showStatus(message, true);

/**
 * Updates only the table content without re-rendering the entire UI to maintain focus
 */
function updateTableContent() {
	const tbody = document.querySelector('.scroll-container tbody');
	if (!tbody) return;

	const filteredUrls = APP_STATE.unmatchedUrls.filter(item =>
		item.old_url.toLowerCase().includes(APP_STATE.filterText.toLowerCase()) ||
		item.suggested_target.toLowerCase().includes(APP_STATE.filterText.toLowerCase())
	);

	tbody.innerHTML = filteredUrls.length > 0 ? filteredUrls.map((item, index) => `
		<tr key="${item.old_url}">
			<td class="px-4 py-3 whitespace-normal break-all text-sm font-medium text-gray-900">
				<code>${item.old_url}</code>
			</td>
			<td class="px-4 py-3">
				<input type="text"
					   id="target-url-${index}"
					   list="current-urls-list"
					   value="${item.suggested_target}"
					   placeholder="/new-page-slug-here"
					   class="datalist-input block w-full border border-gray-300 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm p-2">
			</td>
			<td class="px-4 py-3 text-sm font-medium">
				<button onclick="createRedirect('${item.old_url}', document.getElementById('target-url-${index}').value, ${index})"
						${item.isSaving ? 'disabled' : ''}
						class="w-full inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-full shadow-sm text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50 disabled:cursor-not-allowed">
					${item.isSaving ? `
						<svg class="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
							<circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
							<path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
						</svg>
						Saving...
					` : `
						Add 301 Redirect
					`}
				</button>
			</td>
		</tr>
	`).join('') : `
		<tr>
			<td colspan="3" class="text-center py-6 text-gray-500">
				${APP_STATE.isLoading ? 'Loading unmatched URLs...' : (APP_STATE.totalCsvUrls > 0 ? 'All URLs in CSV are currently redirected or no matches found for your filter.' : 'Please upload a CSV file to begin the matching process.')}
			</td>
		</tr>
	`;

	// Re-attach event listeners for target inputs
	const targetInputs = tbody.querySelectorAll('input[id^="target-url-"]');
	targetInputs.forEach(input => {
		const index = parseInt(input.id.split('-').pop());
		input.addEventListener('input', (e) => {
			APP_STATE.unmatchedUrls[index].suggested_target = e.target.value;
		});
	});
}


/**
 * Main render function that regenerates the UI based on APP_STATE.
 */
function renderApp() {
	const root = document.getElementById('brm-admin-root');
	if (!root) return;

	const filteredUrls = APP_STATE.unmatchedUrls.filter(item =>
		item.old_url.toLowerCase().includes(APP_STATE.filterText.toLowerCase()) ||
		item.suggested_target.toLowerCase().includes(APP_STATE.filterText.toLowerCase())
	);

	root.innerHTML = `
		<div class="p-6 bg-white rounded-xl shadow-lg mt-6 space-y-6">
			<header class="pb-4 border-b">
				<h1 class="text-3xl font-bold text-gray-900">Bulk Redirect Migration Manager</h1>
				<p class="text-gray-600 mt-1">Easily map old site URLs to new pages using the Redirection plugin.</p>
				<p class="text-xs text-yellow-600 mt-2">Note: This plugin requires the 'Redirection' plugin by John Godley to be active.</p>
			</header>

			<div id="brm-status-box"></div>

			<!-- 1. CSV Upload Section -->
			<section class="p-4 border border-indigo-200 bg-indigo-50 rounded-lg space-y-4 shadow-inner">
				<h2 class="text-xl font-semibold text-indigo-700 flex items-center">
					<svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
						<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12" />
					</svg>
					Upload Old URLs CSV
				</h2>

				<form id="csv-upload-form" class="flex flex-col sm:flex-row gap-4" onsubmit="handleUpload(event)">
					<input type="file" id="csv_file" name="csv_file" accept=".csv" required
							class="flex-grow file:mr-4 file:py-2 file:px-4
							file:rounded-full file:border-0
							file:text-sm file:font-semibold
							file:bg-indigo-50 file:text-indigo-700
							hover:file:bg-indigo-100
							border border-gray-300 rounded-full p-2 text-gray-700">

					<button type="submit" ${APP_STATE.isLoading ? 'disabled' : ''}
							class="w-full sm:w-auto px-6 py-2 bg-indigo-600 text-white font-semibold rounded-full shadow-md hover:bg-indigo-700 transition duration-150 disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center">
						${APP_STATE.isLoading ? `
							<svg class="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
								<circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
								<path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
							</svg>
							Comparing...
						` : `
							Compare URLs
						`}
					</button>
				</form>
				${APP_STATE.errors ? `<p class="text-red-500 text-sm mt-2">${APP_STATE.errors}</p>` : ''}
			</section>

			<!-- 2. Results & Mapping Section -->
			${APP_STATE.unmatchedCount > 0 || APP_STATE.totalCsvUrls > 0 ? `
				<section class="space-y-4">
					<div class="flex flex-col sm:flex-row justify-between items-start sm:items-center p-3 bg-gray-100 rounded-lg shadow-sm">
						<p class="text-lg font-medium text-gray-700 mb-2 sm:mb-0">
							Total URLs in CSV: <span class="font-bold text-gray-900">${APP_STATE.totalCsvUrls}</span>
							&middot; <span class="text-red-600">Unmapped (Requires Action): <span class="font-bold">${APP_STATE.unmatchedCount}</span></span>
						</p>
						<input type="text" placeholder="Filter old or suggested URLs..."
							class="px-3 py-2 border border-gray-300 rounded-full w-full sm:w-64 text-sm focus:ring-indigo-500 focus:border-indigo-500"
							id="filter-input"
							value="${APP_STATE.filterText}">
					</div>

					<!-- Data List (Sitemap for Suggestions) -->
					<datalist id="current-urls-list">
						${APP_STATE.currentUrls.map(url => `<option value="${url}">${url}</option>`).join('')}
					</datalist>

					<div class="scroll-container border border-gray-200 rounded-lg shadow-md">
						<table class="min-w-full divide-y divide-gray-200">
							<thead class="bg-gray-50 sticky top-0 z-10 shadow-sm">
								<tr>
									<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/3">Old URL (Source)</th>
									<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/3">New URL (Target Path)</th>
									<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-1/3">Action</th>
								</tr>
							</thead>
							<tbody class="bg-white divide-y divide-gray-200">
								${filteredUrls.length > 0 ? filteredUrls.map((item, index) => `
									<tr key="${item.old_url}">
										<td class="px-4 py-3 whitespace-normal break-all text-sm font-medium text-gray-900">
											<code>${item.old_url}</code>
										</td>
										<td class="px-4 py-3">
											<input type="text"
												   id="target-url-${index}"
												   list="current-urls-list"
												   value="${item.suggested_target}"
												   placeholder="/new-page-slug-here"
												   class="datalist-input block w-full border border-gray-300 rounded-lg shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm p-2">
										</td>
										<td class="px-4 py-3 text-sm font-medium">
											<button onclick="createRedirect('${item.old_url}', document.getElementById('target-url-${index}').value, ${index})"
													${item.isSaving ? 'disabled' : ''}
													class="w-full inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-full shadow-sm text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50 disabled:cursor-not-allowed">
												${item.isSaving ? `
													<svg class="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
														<circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
														<path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
													</svg>
													Saving...
												` : `
													Add 301 Redirect
												`}
											</button>
										</td>
									</tr>
								`).join('') : `
									<tr>
										<td colspan="3" class="text-center py-6 text-gray-500">
											${APP_STATE.isLoading ? 'Loading unmatched URLs...' : (APP_STATE.totalCsvUrls > 0 ? 'All URLs in CSV are currently redirected or no matches found for your filter.' : 'Please upload a CSV file to begin the matching process.')}
										</td>
									</tr>
								`}
							</tbody>
						</table>
					</div>
				</section>
			` : ''}
		</div>
	`;

	// Attach event listeners after rendering (for non-inline events like input change)
	const targetInputs = root.querySelectorAll('input[id^="target-url-"]');
	targetInputs.forEach(input => {
		const index = parseInt(input.id.split('-').pop());
		input.addEventListener('input', (e) => {
			// Update the state's suggested_target in real-time
			APP_STATE.unmatchedUrls[index].suggested_target = e.target.value;
		});
	});

	// Add filter input event listener to prevent focus loss
	const filterInput = root.querySelector('#filter-input');
	if (filterInput) {
		filterInput.addEventListener('input', (e) => {
			APP_STATE.filterText = e.target.value;
			// Update only the table content without full re-render to maintain focus
			updateTableContent();
		});
	}
}

// Initialize the application once the DOM is ready
document.addEventListener('DOMContentLoaded', function() {
	renderApp();
});