<?php
/**
 * Plugin Name: Bulk Redirect Manager for Redirection
 * Description: Extends the 'Redirection' plugin to manage bulk URL migrations via CSV upload and interactive mapping.
 * Version: 1.1.0
 * Author: makeweb
 * License: GPL2
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'BRM_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BRM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'BRM_REST_NAMESPACE', 'bulk-redirect-manager/v1' );
define( 'BRM_CAPABILITY', 'manage_options' );

// 1. Check for Redirection Plugin dependency
add_action( 'admin_init', 'brm_check_redirection_dependency' );
function brm_check_redirection_dependency() {
	if ( ! is_plugin_active( 'redirection/redirection.php' ) ) {
		add_action( 'admin_notices', 'brm_redirection_missing_notice' );
		deactivate_plugins( plugin_basename( __FILE__ ) );
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
	}
}

function brm_redirection_missing_notice() {
	?>
	<div class="notice notice-error is-dismissible">
		<p><strong>Bulk Redirect Manager</strong> requires the <strong>Redirection</strong> plugin to be installed and active. The plugin has been automatically deactivated.</p>
	</div>
	<?php
}

// 2. Setup Admin Menu and Enqueue UI Script
add_action( 'admin_menu', 'brm_add_admin_menu' );
function brm_add_admin_menu() {
	add_management_page(
		'Bulk Redirects',
		'Bulk Redirects',
		BRM_CAPABILITY,
		'bulk-redirect-manager',
		'brm_admin_page_callback'
	);
}

function brm_admin_page_callback() {
	?>
	<div id="brm-admin-root" class="wrap">
		<!-- Content will be injected by JavaScript -->
	</div>
	<?php
}

add_action( 'admin_enqueue_scripts', 'brm_enqueue_scripts' );
function brm_enqueue_scripts( $hook_suffix ) {
	// Only load our script on our specific admin page.
	if ( 'tools_page_bulk-redirect-manager' !== $hook_suffix ) {
		return;
	}

	// Enqueue Tailwind CSS from CDN
	wp_enqueue_script( 'brm-tailwind', 'https://cdn.tailwindcss.com', array(), null, false );
	
	// Enqueue our custom CSS
	wp_enqueue_style(
		'brm-admin-styles',
		BRM_PLUGIN_URL . 'assets/admin-styles.css',
		array(),
		filemtime( BRM_PLUGIN_DIR . 'assets/admin-styles.css' )
	);

	// Enqueue our JavaScript application
	wp_enqueue_script(
		'brm-admin-app',
		BRM_PLUGIN_URL . 'assets/admin-script.js',
		array(), // No jQuery dependency needed as we use native fetch
		filemtime( BRM_PLUGIN_DIR . 'assets/admin-script.js' ),
		true // Enqueue in the footer
	);

	// Localize script data for the JS app to use.
	wp_localize_script( 'brm-admin-app', 'brm_vars', array(
		'root'  => esc_url_raw( rest_url() ),
		'nonce' => wp_create_nonce( 'wp_rest' ),
		'namespace' => BRM_REST_NAMESPACE,
	) );
}

// 3. Register Custom REST API Endpoints
add_action( 'rest_api_init', 'brm_register_rest_routes' );
function brm_register_rest_routes() {
	// Endpoint for CSV Upload and Comparison
	register_rest_route( BRM_REST_NAMESPACE, '/upload-csv', array(
		'methods'             => 'POST',
		'callback'            => 'brm_handle_csv_upload',
		'permission_callback' => function () {
			return current_user_can( BRM_CAPABILITY );
		},
		'args' => array(
			'csv_data' => array(
				'required' => true,
				'type' => 'string',
				'description' => 'The CSV file content as text data.',
			),
		),
	) );

	// Endpoint for Creating a Single Redirect
	register_rest_route( BRM_REST_NAMESPACE, '/create-redirect', array(
		'methods'             => 'POST',
		'callback'            => 'brm_create_single_redirect',
		'permission_callback' => function () {
			return current_user_can( BRM_CAPABILITY );
		},
		'args' => array(
			'source_url' => array( 'required' => true, 'sanitize_callback' => 'sanitize_text_field' ),
			'target_url' => array( 'required' => true, 'sanitize_callback' => 'sanitize_text_field' ),
		),
	) );
}

// 4. REST API Callback: Handle CSV Upload and Comparison Logic
function brm_handle_csv_upload( $request ) {
	// For simplicity, we assume the CSV is provided in the request body as text data for now,
	// as direct file upload via REST API can be complex. The JS will handle reading the file.
	$csv_data = $request->get_param( 'csv_data' );
	if ( empty( $csv_data ) ) {
		return new WP_REST_Response( array( 'message' => 'CSV data is empty.' ), 400 );
	}

	$old_urls = brm_parse_csv_data( $csv_data );
	if ( empty( $old_urls ) ) {
		return new WP_REST_Response( array( 'message' => 'No valid URLs found in CSV.' ), 400 );
	}

	// Step 1: Get existing Redirection sources
	global $wpdb;
	$table_name = $wpdb->prefix . 'redirection_items';
	// Check if Redirection's table exists
	if ( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) === $table_name ) {
		$existing_redirects_sources = $wpdb->get_col( "SELECT url FROM $table_name WHERE status = 'enabled'" );
	} else {
		$existing_redirects_sources = array(); // Redirection plugin is active but table not found? Fallback.
	}

	// Step 2: Get current existing internal URLs (posts, pages, published)
	$current_internal_urls = brm_get_current_internal_urls();

	$unmatched_urls = array();

	// Step 3: Comparison logic
	foreach ( $old_urls as $old_url ) {
		$normalized_old_url = brm_normalize_url( $old_url );
		$is_matched = false;
		$match_reason = '';

		// Check against existing redirects
		foreach ($existing_redirects_sources as $source) {
			if ( brm_normalize_url( $source ) === $normalized_old_url ) {
				$is_matched = true;
				$match_reason = 'redirect';
				break;
			}
		}

		if ( $is_matched ) {
			continue; // Already redirected, skip.
		}

		// Check against current internal URLs (URLs that exist as live content)
		foreach ( $current_internal_urls as $current_url ) {
			if ( brm_normalize_url( $current_url ) === $normalized_old_url ) {
				$is_matched = true;
				$match_reason = 'existing_page';
				break;
			}
		}

		if ( $is_matched ) {
			continue; // URL exists as live content, should not redirect.
		}

		// If it's not currently redirected and doesn't exist as live content, add it to the unmatched list.
		$unmatched_urls[] = array(
			'old_url' => $old_url,
			'status' => 'unmapped',
			'suggested_target' => '' // A space for suggested target if we had a matching algorithm, for now, empty.
		);
	}

	// Store the unmatched list temporarily in the user's session or transient cache (using Transient here)
	$transient_key = 'brm_unmatched_' . get_current_user_id();
	set_transient( $transient_key, $unmatched_urls, HOUR_IN_SECONDS * 2 ); // Store for 2 hours

	return new WP_REST_Response( array(
		'message' => 'CSV processed successfully.',
		'count' => count( $unmatched_urls ),
		'total_csv_urls' => count( $old_urls ),
		'unmatched_urls' => $unmatched_urls,
		'current_urls' => $current_internal_urls // Provide current URLs for client-side suggestions
	), 200 );
}

// 5. REST API Callback: Create Single Redirect
function brm_create_single_redirect( $request ) {
	error_log( 'BRM: create_redirect called via REST API' );
	
	$source_url = $request->get_param( 'source_url' );
	$target_url = $request->get_param( 'target_url' );

	error_log( 'BRM: source_url=' . $source_url . ', target_url=' . $target_url );

	if ( empty( $source_url ) || empty( $target_url ) ) {
		error_log( 'BRM: Empty source or target URL' );
		return new WP_REST_Response( array( 'message' => 'Source and Target URLs are required.' ), 400 );
	}

	// Additional validation: Check if source URL exists as live content
	$current_internal_urls = brm_get_current_internal_urls();
	$normalized_source = brm_normalize_url( $source_url );
	
	foreach ( $current_internal_urls as $current_url ) {
		if ( brm_normalize_url( $current_url ) === $normalized_source ) {
			error_log( 'BRM: Cannot create redirect - source URL exists as live content: ' . $source_url );
			return new WP_REST_Response( array( 
				'message' => 'Cannot create redirect: Source URL exists as live content.',
				'source' => $source_url 
			), 400 );
		}
	}

	// Check if redirect already exists
	global $wpdb;
	$table_name = $wpdb->prefix . 'redirection_items';
	if ( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) === $table_name ) {
		$existing_redirect = $wpdb->get_var( $wpdb->prepare(
			"SELECT url FROM $table_name WHERE url = %s AND status = 'enabled'",
			$source_url
		) );
		
		if ( $existing_redirect ) {
			error_log( 'BRM: Cannot create redirect - redirect already exists: ' . $source_url );
			return new WP_REST_Response( array( 
				'message' => 'Cannot create redirect: Redirect already exists for this source URL.',
				'source' => $source_url 
			), 400 );
		}
	}

	// Use Redirection's REST API to create redirects
	$redirection_api_url = rest_url( 'redirection/v1/redirect' );
	
	// Get the first available group ID (usually 1)
	$groups_response = wp_remote_get( 
		rest_url( 'redirection/v1/group' ),
		array(
			'headers' => array(
				'X-WP-Nonce' => wp_create_nonce( 'wp_rest' ),
			),
		)
	);
	
	$group_id = 1; // Default fallback
	if ( ! is_wp_error( $groups_response ) && wp_remote_retrieve_response_code( $groups_response ) === 200 ) {
		$groups_data = json_decode( wp_remote_retrieve_body( $groups_response ), true );
		if ( ! empty( $groups_data['items'] ) ) {
			$group_id = $groups_data['items'][0]['id'];
		}
	}
	
	error_log( 'BRM: Using group_id=' . $group_id );
	
	// Prepare redirect data for Redirection REST API
	$redirect_data = array(
		'url' => $source_url,
		'match_type' => 'url',
		'action_type' => 'url',
		'action_data' => array( 'url' => $target_url ),
		'action_code' => 301,
		'group_id' => $group_id,
		'title' => 'Bulk Redirect: ' . $source_url,
	);
	
	error_log( 'BRM: Creating redirect via REST API with data: ' . json_encode( $redirect_data ) );
	
	// Make the REST API call to Redirection plugin
	// For internal REST API calls, we need to include proper authentication
	$current_user = wp_get_current_user();
	$cookies = array();
	
	// Include WordPress authentication cookies
	foreach ( $_COOKIE as $name => $value ) {
		if ( strpos( $name, 'wordpress_' ) === 0 || strpos( $name, 'wp-' ) === 0 ) {
			$cookies[] = new WP_Http_Cookie( array( 'name' => $name, 'value' => $value ) );
		}
	}
	
	$response = wp_remote_post( 
		$redirection_api_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'X-WP-Nonce' => wp_create_nonce( 'wp_rest' ),
			),
			'body' => json_encode( $redirect_data ),
			'timeout' => 30,
			'cookies' => $cookies,
		)
	);
	
	if ( is_wp_error( $response ) ) {
		error_log( 'BRM: HTTP error creating redirect: ' . $response->get_error_message() );
		return new WP_REST_Response( array(
			'message' => 'HTTP error creating redirect.',
			'details' => $response->get_error_message()
		), 500 );
	}
	
	$response_code = wp_remote_retrieve_response_code( $response );
	$response_body = wp_remote_retrieve_body( $response );
	
	error_log( 'BRM: Redirection API response code: ' . $response_code );
	error_log( 'BRM: Redirection API response body: ' . $response_body );
	
	if ( $response_code === 200 || $response_code === 201 ) {
		error_log( 'BRM: Redirect created successfully via REST API' );
		return new WP_REST_Response( array(
			'message' => 'Redirect created successfully.',
			'source' => $source_url,
			'target' => $target_url
		), 200 );
	} else {
		$error_data = json_decode( $response_body, true );
		$error_message = isset( $error_data['message'] ) ? $error_data['message'] : 'Unknown error';
		
		error_log( 'BRM: Failed to create redirect via REST API: ' . $error_message );
		return new WP_REST_Response( array(
			'message' => 'Failed to create redirect.',
			'details' => $error_message,
			'response_code' => $response_code
		), 500 );
	}
}

// --- Helper Functions ---

/**
 * Parses CSV text data (assuming only one column of URLs).
 *
 * @param string $csv_data The raw CSV string.
 * @return array Cleaned array of URLs.
 */
function brm_parse_csv_data( $csv_data ) {
	$urls = array();
	$lines = explode( "\n", $csv_data );
	foreach ( $lines as $line ) {
		$line = trim( $line );
		if ( empty( $line ) ) {
			continue;
		}

		// Use str_getcsv to safely handle quoted lines, taking the first column.
		$data = str_getcsv( $line );
		if ( ! empty( $data[0] ) ) {
			$urls[] = esc_url_raw( trim( $data[0] ) );
		}
	}
	return array_filter( $urls );
}

/**
 * Gets a list of all current internal published URLs (sitemap equivalent).
 *
 * @return array Array of slugs/relative paths.
 */
function brm_get_current_internal_urls() {
	$site_url = get_site_url();
	$urls = array();

	$post_types = get_post_types( array( 'public' => true ), 'names' );

	// Query for all public, published post types (posts, pages, custom types)
	$posts = get_posts( array(
		'post_type'      => $post_types,
		'post_status'    => 'publish',
		'posts_per_page' => -1, // Get all
		'fields'         => 'ids', // Get only IDs for performance
	) );

	foreach ( $posts as $post_id ) {
		$permalink = get_permalink( $post_id );
		// Use the normalize function for consistency
		$normalized_url = brm_normalize_url( $permalink );
		$urls[] = $normalized_url;
	}

	return array_unique( $urls );
}

/**
 * Normalizes a URL for comparison (removes domain, protocol, etc.)
 *
 * @param string $url The URL to normalize.
 * @return string The normalized relative path.
 */
function brm_normalize_url( $url ) {
	$site_url = get_site_url();
	$site_host = parse_url( $site_url, PHP_URL_HOST );
	
	// Parse the URL to handle it properly
	$parsed = parse_url( $url );
	
	// If it's a full URL, extract just the path
	if ( isset( $parsed['host'] ) ) {
		$normalized = isset( $parsed['path'] ) ? $parsed['path'] : '/';
	} else {
		// It's already a relative URL
		$normalized = $url;
	}
	
	// Remove any remaining domain references
	$normalized = str_replace( array( 'http://', 'https://', $site_url, $site_host ), '', $normalized );
	
	// Ensure it starts with a slash and is trimmed
	$normalized = '/' . ltrim( $normalized, '/' );
	
	// Remove query string and fragment if they weren't handled by parse_url
	$normalized = preg_replace( '/\?.*$/', '', $normalized );
	$normalized = preg_replace( '/\#.*$/', '', $normalized );
	
	// Remove trailing slash for consistent comparison (except for root)
	if ( $normalized !== '/' ) {
		$normalized = rtrim( $normalized, '/' );
	}

	return $normalized;
}

// Set up logging for debugging (optional but good practice)
// set_transient( 'redirection_Api_debug', true, HOUR_IN_SECONDS * 24 );


////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if (class_exists('makeUpdate')) {
    try {
        $updater = new makeUpdate("9547", "tkt6rkp4auz*RHA_mab", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}


?>
