/**
 * Divi-specific JavaScript for Make FAQ plugin
 */
(function($) {
    'use strict';

    // Initialize the FAQ functionality for Divi
    $(document).ready(function() {
        // Initialize accordions and toggles
        initDiviFaqDisplay();

        // Initialize search functionality
        initDiviFaqSearch();
    });

    /**
     * Initialize FAQ display (accordion, toggle, list) for Divi
     */
    function initDiviFaqDisplay() {
        // Use more general selectors to handle different Divi DOM structures
        // First try with Divi specific selectors
        var $questions = $('.et-db #et-boc .et-l .make-faq-question, .et_pb_module .make-faq-question, .make-faq-question');
        
        // Handle question click for accordion and toggle
        $questions.off('click').on('click', function() {
            var $question = $(this);
            var $item = $question.closest('.make-faq-item');
            var $container = $item.closest('.make-faq-container');
            var displayStyle = $container.data('display-style') || 'accordion'; // Default to accordion

            // Do nothing for list style (non-interactive)
            if (displayStyle === 'list') {
                return;
            }
            
            // For accordion, close other items first
            if (displayStyle === 'accordion') {
                $container.find('.make-faq-item').not($item).removeClass('active');
            }
            
            // Toggle active state
            $item.toggleClass('active');
            
            // Update ARIA attributes
            var isExpanded = $item.hasClass('active');
            $question.attr('aria-expanded', isExpanded ? 'true' : 'false');
            
            // Track view if opened
            if (isExpanded) {
                var faqId = $item.data('id');
                if (faqId) {
                    trackDiviFaqView(faqId);
                }
            }
        });
        
        // Add accessibility attributes
        // Only add button role/tabindex for interactive styles
        $questions.each(function() {
            var $q = $(this);
            var $container = $q.closest('.make-faq-container');
            var displayStyle = $container.data('display-style') || 'accordion';
            if (displayStyle === 'list') {
                $q.removeAttr('role').removeAttr('tabindex').attr('aria-expanded', 'true');
            } else {
                $q.attr('role', 'button').attr('tabindex', '0');
            }
        });
        
        // Handle keyboard navigation
        $questions.off('keydown').on('keydown', function(e) {
            var $question = $(this);
            var $container = $question.closest('.make-faq-container');
            var displayStyle = $container.data('display-style') || 'accordion';
            if (displayStyle === 'list') {
                return; // non-interactive
            }
            // Enter or Space key
            if (e.which === 13 || e.which === 32) {
                e.preventDefault();
                $(this).click();
            }
        });
        
        console.log('FAQ display initialized with', $questions.length, 'questions');
    }

    /**
     * Initialize FAQ search functionality for Divi
     */
    function initDiviFaqSearch() {
        var $inputs = $('.et-db #et-boc .et-l .make-faq-search-input, .make-faq-search-input');
        var $forms  = $('.et-db #et-boc .et-l .make-faq-search-form, .make-faq-search-form');
        var $clears = $('.et-db #et-boc .et-l .make-faq-search-clear, .make-faq-search-clear');

        function toggleHasValue($input) {
            var hasVal = $.trim($input.val()).length > 0;
            var $holder = $input.closest('.make-faq-input-holder');
            if (hasVal) {
                $holder.addClass('has-value');
            } else {
                $holder.removeClass('has-value');
            }
        }

        // Initialize state on load
        $inputs.each(function(){ toggleHasValue($(this)); });

        // Handle search input for filtering FAQs and toggle clear visibility
        $inputs.off('input').on('input', function() {
            var $input = $(this);
            var searchTerm = $input.val().toLowerCase().trim();
            toggleHasValue($input);
            filterDiviFaqs(searchTerm);
        });
    
        // Handle form submission (let it submit normally to set URL if desired)
        $forms.off('submit').on('submit', function(e) {
            // No JS prevention; filtering is handled by input event and on page load
        });
    
        // Handle clear button (X icon)
        $clears.off('click').on('click', function() {
            var $holder = $(this).closest('.make-faq-input-holder');
            var $input = $holder.find('.make-faq-search-input');

            // Clear the search input
            $input.val('');
            toggleHasValue($input);
    
            // Show all FAQs
            filterDiviFaqs('');
    
            // Remove search parameter from URL without reloading
            var url = window.location.href.split('?')[0];
            window.history.replaceState({}, document.title, url);

            // Focus back to input for convenience
            $input.focus();
        });
    
        // Check for search parameter in URL on page load
        var urlParams = new URLSearchParams(window.location.search);
        var searchParam = urlParams.get('faq_search');
    
        if (searchParam) {
            // Set the search input value
            $inputs.val(searchParam);
            $inputs.each(function(){ toggleHasValue($(this)); });
            // Filter FAQs based on the search parameter
            filterDiviFaqs(searchParam.toLowerCase());
        }
    }

    /**
     * Filter FAQs based on search term for Divi
     */
    function filterDiviFaqs(searchTerm) {
        // Get all FAQ items
        var $faqItems = $('.et-db #et-boc .et-l .make-faq-item');
        var visibleCount = 0;
    
        // If search term is empty, show all FAQs
        if (!searchTerm) {
            $faqItems.show();
            $('.et-db #et-boc .et-l .make-faq-no-results').remove();
            return;
        }
    
        // Loop through each FAQ item
        $faqItems.each(function() {
            var $item = $(this);
            var question = $item.find('.make-faq-question').text().toLowerCase();
            var answer = $item.find('.make-faq-answer').text().toLowerCase();
        
            // Check if the question or answer contains the search term
            if (question.includes(searchTerm) || answer.includes(searchTerm)) {
                $item.show();
                visibleCount++;
            } else {
                $item.hide();
            }
        });
    
        // Show no results message if no FAQs are visible
        if (visibleCount === 0) {
            // Remove existing no results message
            $('.et-db #et-boc .et-l .make-faq-no-results').remove();
        
            // Add no results message after the FAQ container
            $('.et-db #et-boc .et-l .make-faq-container').after('<div class="make-faq-no-results">' + 
                wp.i18n.__('No FAQs found matching your search.', 'make-faq') + 
                '</div>');
        } else {
            // Remove no results message if FAQs are visible
            $('.et-db #et-boc .et-l .make-faq-no-results').remove();
        }
    }

    /**
     * Track FAQ view for Divi
     */
    function trackDiviFaqView(faqId) {
        // Only track if we have an ID
        if (!faqId) {
            return;
        }
        
        // Send AJAX request to track view
        $.ajax({
            url: makeFaqPublic.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_track_view',
                nonce: makeFaqPublic.nonce,
                id: faqId
            }
        });
    }

    /**
     * Handle Divi Builder events
     */
    $(window).on('et_builder_api_ready', function(event, API) {
        // Re-initialize when Divi Builder updates the page
        API.subscribe('et.pb.division.update', function() {
            setTimeout(function() {
                initDiviFaqDisplay();
                initDiviFaqSearch();
            }, 500);
        });
    });

})(jQuery);