<?php
/**
 * Register shortcodes for the plugin.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 */

/**
 * Register shortcodes for the plugin.
 *
 * This class handles the registration and rendering of shortcodes
 * for displaying FAQs and the FAQ search functionality.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 * @author     makeWeb
 */
class Make_FAQ_Shortcodes {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the shortcodes.
     *
     * @since    1.0.0
     */
    public function register_shortcodes() {
        add_shortcode( 'makefaq', array( $this, 'faq_display_shortcode' ) );
        add_shortcode( 'makefaq_search', array( $this, 'faq_search_shortcode' ) );
    }

    /**
     * Render the FAQ Display shortcode.
     *
     * @since    1.0.0
     * @param    array    $atts    The shortcode attributes.
     * @return   string            The shortcode HTML.
     */
    public function faq_display_shortcode( $atts ) {
        // Get default display style from settings
        $options = get_option( 'make_faq_settings', array() );
        $default_display = isset( $options['display_style'] ) ? $options['display_style'] : 'accordion';
        
        // Define shortcode attributes
        $atts = shortcode_atts(
            array(
                'category'     => '',
                'tag'          => '',
                'ids'          => '',
                'display'      => $default_display,
                'orderby'      => 'date',
                'order'        => 'DESC',
                'limit'        => -1,
                'class'        => '',
            ),
            $atts,
            'makefaq'
        );

        // Convert IDs string to array if provided
        $ids = array();
        if ( ! empty( $atts['ids'] ) ) {
            $ids = array_map( 'intval', explode( ',', $atts['ids'] ) );
        }

        // Build attributes array for block renderer
        $block_attributes = array(
            'category'     => $atts['category'],
            'tag'          => $atts['tag'],
            'ids'          => $ids,
            'displayStyle' => $atts['display'],
            'orderBy'      => $atts['orderby'],
            'order'        => $atts['order'],
            'limit'        => intval( $atts['limit'] ),
            'className'    => $atts['class'],
        );

        // Use the block renderer to maintain consistency
        $blocks = new Make_FAQ_Blocks( $this->plugin_name, $this->version );
        return $blocks->render_faq_display_block( $block_attributes );
    }

    /**
     * Render the FAQ Search shortcode.
     *
     * @since    1.0.0
     * @param    array    $atts    The shortcode attributes.
     * @return   string            The shortcode HTML.
     */
    public function faq_search_shortcode( $atts ) {
        // Define shortcode attributes
        $atts = shortcode_atts(
            array(
                'placeholder' => 'Search FAQs...',
                'button'      => 'Search',
                'category'    => '',
                'tag'         => '',
                'class'       => '',
            ),
            $atts,
            'makefaq_search'
        );

        // Build attributes array for block renderer
        $block_attributes = array(
            'placeholder' => $atts['placeholder'],
            'buttonText'  => $atts['button'],
            'category'    => $atts['category'],
            'tag'         => $atts['tag'],
            'className'   => $atts['class'],
        );

        // Use the block renderer to maintain consistency
        $blocks = new Make_FAQ_Blocks( $this->plugin_name, $this->version );
        return $blocks->render_faq_search_block( $block_attributes );
    }
}