<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * the public-facing side of the site.
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/public
 * @author     makeWeb
 */
class Make_FAQ_Public {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of the plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style( $this->plugin_name, MAKE_FAQ_PLUGIN_URL . 'assets/css/make-faq-public.css', array(), $this->version, 'all' );
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        // Enqueue the WordPress i18n script for translation functions
        wp_enqueue_script( 'wp-i18n' );
        
        wp_enqueue_script( $this->plugin_name, MAKE_FAQ_PLUGIN_URL . 'assets/js/make-faq-public.js', array( 'jquery', 'wp-i18n' ), $this->version, true );
        
        // Localize the script with data
        wp_localize_script(
            $this->plugin_name,
            'makeFaqPublic',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'make_faq_public_nonce' ),
            )
        );
    }

    /**
     * Add FAQPage schema markup to the head.
     *
     * @since    1.0.0
     */
    public function add_faq_schema() {
        global $post;
        
        // Check if schema is enabled in settings
        $options = get_option( 'make_faq_settings', array() );
        $schema_enabled = isset( $options['schema_enabled'] ) ? $options['schema_enabled'] : 1;
        
        if ( ! $schema_enabled ) {
            return;
        }
        
        // Check if we're on a page with FAQs
        $has_faq_block = false;
        $has_faq_shortcode = false;
        
        if ( $post && is_singular() ) {
            // Check for Gutenberg block
            if ( has_block( 'make-faq/faq-display', $post ) ) {
                $has_faq_block = true;
            }
            
            // Check for shortcode
            if ( has_shortcode( $post->post_content, 'makefaq' ) ) {
                $has_faq_shortcode = true;
            }
        }
        
        // If no FAQs on the page, return
        if ( ! $has_faq_block && ! $has_faq_shortcode ) {
            return;
        }
        
        // Get FAQs to include in schema
        $faqs = $this->get_faqs_for_schema();
        
        if ( empty( $faqs ) ) {
            return;
        }
        
        // Build schema
        $schema = array(
            '@context'   => 'https://schema.org',
            '@type'      => 'FAQPage',
            'mainEntity' => array(),
        );
        
        foreach ( $faqs as $faq ) {
            $schema['mainEntity'][] = array(
                '@type'          => 'Question',
                'name'           => $faq->post_title,
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text'  => $faq->post_content,
                ),
            );
        }
        
        // Output schema
        echo '<script type="application/ld+json">' . wp_json_encode( $schema ) . '</script>';
    }

    /**
     * Get FAQs for schema markup.
     *
     * @since    1.0.0
     * @return   array    The FAQs.
     */
    private function get_faqs_for_schema() {
        global $post;
        
        // Get all FAQs on the page
        $faqs = array();
        
        // Check for Gutenberg block
        if ( has_block( 'make-faq/faq-display', $post ) ) {
            // Parse blocks to get attributes
            $blocks = parse_blocks( $post->post_content );
            
            foreach ( $blocks as $block ) {
                if ( $block['blockName'] === 'make-faq/faq-display' ) {
                    $attrs = $block['attrs'];
                    
                    // Build query args
                    $args = array(
                        'post_type'      => 'make_faq',
                        'posts_per_page' => isset( $attrs['limit'] ) ? intval( $attrs['limit'] ) : -1,
                        'orderby'        => isset( $attrs['orderBy'] ) ? $attrs['orderBy'] : 'date',
                        'order'          => isset( $attrs['order'] ) ? $attrs['order'] : 'DESC',
                    );
                    
                    // Add taxonomy queries if needed
                    $tax_query = array();
                    
                    if ( isset( $attrs['category'] ) && ! empty( $attrs['category'] ) ) {
                        $tax_query[] = array(
                            'taxonomy' => 'faq_category',
                            'field'    => 'slug',
                            'terms'    => explode( ',', $attrs['category'] ),
                        );
                    }
                    
                    if ( isset( $attrs['tag'] ) && ! empty( $attrs['tag'] ) ) {
                        $tax_query[] = array(
                            'taxonomy' => 'faq_tag',
                            'field'    => 'slug',
                            'terms'    => explode( ',', $attrs['tag'] ),
                        );
                    }
                    
                    if ( ! empty( $tax_query ) ) {
                        $args['tax_query'] = $tax_query;
                    }
                    
                    // Add post IDs if specified
                    if ( isset( $attrs['ids'] ) && ! empty( $attrs['ids'] ) ) {
                        $args['post__in'] = $attrs['ids'];
                        $args['orderby'] = 'post__in';
                    }
                    
                    // Get FAQs
                    $block_faqs = get_posts( $args );
                    
                    // Add to collection
                    $faqs = array_merge( $faqs, $block_faqs );
                }
            }
        }
        
        // Check for shortcode
        if ( has_shortcode( $post->post_content, 'makefaq' ) ) {
            // Extract shortcodes
            $pattern = get_shortcode_regex( array( 'makefaq' ) );
            preg_match_all( '/' . $pattern . '/s', $post->post_content, $matches );
            
            if ( ! empty( $matches[0] ) ) {
                foreach ( $matches[0] as $shortcode ) {
                    // Parse shortcode attributes
                    $atts = shortcode_parse_atts( $shortcode );
                    
                    // Build query args
                    $args = array(
                        'post_type'      => 'make_faq',
                        'posts_per_page' => isset( $atts['limit'] ) ? intval( $atts['limit'] ) : -1,
                        'orderby'        => isset( $atts['orderby'] ) ? $atts['orderby'] : 'date',
                        'order'          => isset( $atts['order'] ) ? $atts['order'] : 'DESC',
                    );
                    
                    // Add taxonomy queries if needed
                    $tax_query = array();
                    
                    if ( isset( $atts['category'] ) && ! empty( $atts['category'] ) ) {
                        $tax_query[] = array(
                            'taxonomy' => 'faq_category',
                            'field'    => 'slug',
                            'terms'    => explode( ',', $atts['category'] ),
                        );
                    }
                    
                    if ( isset( $atts['tag'] ) && ! empty( $atts['tag'] ) ) {
                        $tax_query[] = array(
                            'taxonomy' => 'faq_tag',
                            'field'    => 'slug',
                            'terms'    => explode( ',', $atts['tag'] ),
                        );
                    }
                    
                    if ( ! empty( $tax_query ) ) {
                        $args['tax_query'] = $tax_query;
                    }
                    
                    // Add post IDs if specified
                    if ( isset( $atts['ids'] ) && ! empty( $atts['ids'] ) ) {
                        $args['post__in'] = explode( ',', $atts['ids'] );
                        $args['orderby'] = 'post__in';
                    }
                    
                    // Get FAQs
                    $shortcode_faqs = get_posts( $args );
                    
                    // Add to collection
                    $faqs = array_merge( $faqs, $shortcode_faqs );
                }
            }
        }
        
        // Remove duplicates
        $unique_faqs = array();
        $faq_ids = array();
        
        foreach ( $faqs as $faq ) {
            if ( ! in_array( $faq->ID, $faq_ids ) ) {
                $faq_ids[] = $faq->ID;
                $unique_faqs[] = $faq;
                
                // Track view count
                $this->track_faq_view( $faq->ID );
            }
        }
        
        return $unique_faqs;
    }

    /**
     * Track FAQ view count.
     *
     * @since    1.0.0
     * @param    int    $faq_id    The FAQ ID.
     */
    private function track_faq_view( $faq_id ) {
        // Get current count
        $count = get_post_meta( $faq_id, '_make_faq_view_count', true );
        
        // Increment count
        $count = $count ? intval( $count ) + 1 : 1;
        
        // Update count
        update_post_meta( $faq_id, '_make_faq_view_count', $count );
    }

    /**
     * AJAX handler for searching FAQs.
     *
     * @since    1.0.0
     */
    public function ajax_search_faqs() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_public_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Get search term
        $search_term = isset( $_POST['search'] ) ? sanitize_text_field( $_POST['search'] ) : '';
        
        if ( empty( $search_term ) ) {
            wp_send_json_error( array( 'message' => __( 'Please enter a search term.', 'make-faq' ) ) );
        }

        // Get category and tag filters
        $category = isset( $_POST['category'] ) ? sanitize_text_field( $_POST['category'] ) : '';
        $tag = isset( $_POST['tag'] ) ? sanitize_text_field( $_POST['tag'] ) : '';

        // Build query args
        $args = array(
            'post_type'      => 'make_faq',
            'posts_per_page' => 10,
            's'              => $search_term,
        );

        // Add taxonomy queries if needed
        $tax_query = array();

        if ( ! empty( $category ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_category',
                'field'    => 'slug',
                'terms'    => explode( ',', $category ),
            );
        }

        if ( ! empty( $tag ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_tag',
                'field'    => 'slug',
                'terms'    => explode( ',', $tag ),
            );
        }

        if ( ! empty( $tax_query ) ) {
            $args['tax_query'] = $tax_query;
        }

        // Get FAQs
        $faqs = get_posts( $args );

        // Track search term
        $this->track_search_term( $search_term );

        // Prepare response
        $response = array();

        if ( ! empty( $faqs ) ) {
            foreach ( $faqs as $faq ) {
                $response[] = array(
                    'id'       => $faq->ID,
                    'question' => $faq->post_title,
                    'answer'   => wp_trim_words( $faq->post_content, 30, '...' ),
                    'url'      => get_permalink( $faq->ID ),
                );
            }
        }

        wp_send_json_success( array(
            'results' => $response,
            'count'   => count( $response ),
        ) );
    }

    /**
     * Track search term.
     *
     * @since    1.0.0
     * @param    string    $term    The search term.
     */
    private function track_search_term( $term ) {
        // Get current search terms
        $search_terms = get_option( 'make_faq_search_terms', array() );
        
        // Add or increment term
        if ( isset( $search_terms[ $term ] ) ) {
            $search_terms[ $term ]++;
        } else {
            $search_terms[ $term ] = 1;
        }
        
        // Update option
        update_option( 'make_faq_search_terms', $search_terms );
    }
}