<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two hooks for
 * enqueuing the admin-specific stylesheet and JavaScript.
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/admin
 * @author     makeWeb
 */
class Make_FAQ_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        // Only load on FAQ post type screens
        $screen = get_current_screen();
        if ( ! $screen || 'make_faq' !== $screen->post_type ) {
            return;
        }

        wp_enqueue_style( $this->plugin_name, MAKE_FAQ_PLUGIN_URL . 'assets/css/make-faq-admin.css', array(), $this->version, 'all' );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        // Only load on FAQ post type screens
        $screen = get_current_screen();
        if ( ! $screen || 'make_faq' !== $screen->post_type ) {
            return;
        }

        wp_enqueue_script( $this->plugin_name, MAKE_FAQ_PLUGIN_URL . 'assets/js/make-faq-admin.js', array( 'jquery', 'wp-util' ), $this->version, true );
        
        // Enqueue the multiselect script for the edit screen
        if ( $screen && 'make_faq' === $screen->post_type && 'edit' === $screen->base ) {
            // Enqueue Select2 library from CDN
            wp_enqueue_style( 'select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css', array(), '4.0.13' );
            wp_enqueue_script( 'select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js', array( 'jquery' ), '4.0.13', true );
            
            // Enqueue our custom Select2 styles
            wp_enqueue_style( $this->plugin_name . '-select2', MAKE_FAQ_PLUGIN_URL . 'assets/css/make-faq-select2.css', array( 'select2' ), $this->version );
            
            // Enqueue our custom multiselect script
            wp_enqueue_script( $this->plugin_name . '-multiselect', MAKE_FAQ_PLUGIN_URL . 'assets/js/make-faq-multiselect.js', array( 'jquery', $this->plugin_name, 'select2' ), $this->version, true );
        }

        // Localize the script with data
        // Get debug mode setting
        $options = get_option( 'make_faq_settings', array() );
        $debug_mode = isset( $options['debug_mode'] ) ? (bool) $options['debug_mode'] : false;
        
        wp_localize_script(
            $this->plugin_name,
            'makeFaqAdmin',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'make_faq_admin_nonce' ),
                'debugMode' => $debug_mode,
                'strings' => array(
                    'editFaq'          => __( 'Edit FAQ', 'make-faq' ),
                    'addNewFaq'        => __( 'Add New FAQ', 'make-faq' ),
                    'saveFaq'          => __( 'Save FAQ', 'make-faq' ),
                    'cancel'           => __( 'Cancel', 'make-faq' ),
                    'saving'           => __( 'Saving...', 'make-faq' ),
                    'saved'            => __( 'Saved!', 'make-faq' ),
                    'error'            => __( 'Error saving FAQ. Please try again.', 'make-faq' ),
                    'confirmClose'     => __( 'You have unsaved changes. Are you sure you want to close?', 'make-faq' ),
                    'created'          => __( 'FAQ created successfully!', 'make-faq' ),
                    'editCategories'   => __( 'Edit Categories', 'make-faq' ),
                    'editTags'         => __( 'Edit Tags', 'make-faq' ),
                    'saveCategories'   => __( 'Save Categories', 'make-faq' ),
                    'saveTags'         => __( 'Save Tags', 'make-faq' ),
                    'categoriesSaved'  => __( 'Categories saved successfully!', 'make-faq' ),
                    'tagsSaved'        => __( 'Tags saved successfully!', 'make-faq' ),
                    'confirmDelete'    => __( 'Are you sure you want to move this FAQ to trash?', 'make-faq' ),
                    'deleted'          => __( 'FAQ moved to trash.', 'make-faq' ),
                ),
            )
        );
        
        // Add modals to the admin footer
        add_action( 'admin_footer', array( $this, 'render_modals' ) );
    }
    
    /**
     * Render the modals for categories and tags editing.
     *
     * @since    1.0.4
     */
    public function render_modals() {
        // Only render on FAQ list page
        $screen = get_current_screen();
        if ( ! $screen || 'make_faq' !== $screen->post_type || 'edit' !== $screen->base ) {
            return;
        }
        
        // Categories modal
        ?>
        <div class="make-faq-modal-overlay" id="make-faq-categories-modal-overlay"></div>
        <div class="make-faq-modal" id="make-faq-categories-modal">
            <div class="make-faq-modal-header">
                <h2><?php echo esc_html__( 'Edit Categories', 'make-faq' ); ?></h2>
                <span class="make-faq-modal-close dashicons dashicons-no-alt"></span>
            </div>
            <div class="make-faq-modal-content">
                <form id="make-faq-categories-form">
                    <input type="hidden" id="make-faq-categories-post-id" name="post_id" value="">
                    <div class="make-faq-form-field">
                        <label for="make-faq-categories-select"><?php echo esc_html__( 'Categories', 'make-faq' ); ?></label>
                        <select id="make-faq-categories-select" name="categories[]" multiple="multiple" class="make-faq-select2">
                            <!-- Options will be populated via JavaScript -->
                        </select>
                        <p class="description"><?php echo esc_html__( 'Select one or more categories for this FAQ.', 'make-faq' ); ?></p>
                    </div>
                    <p class="description"><?php echo esc_html__( 'Type to search existing categories or enter a new category name to create it.', 'make-faq' ); ?></p>
                </form>
            </div>
            <div class="make-faq-modal-footer">
                <div id="make-faq-categories-status"></div>
                <button type="button" id="make-faq-categories-cancel" class="button"><?php echo esc_html__( 'Cancel', 'make-faq' ); ?></button>
                <button type="button" id="make-faq-categories-save" class="button button-primary"><?php echo esc_html__( 'Save Categories', 'make-faq' ); ?></button>
            </div>
        </div>
        
        <!-- Tags modal -->
        <div class="make-faq-modal-overlay" id="make-faq-tags-modal-overlay"></div>
        <div class="make-faq-modal" id="make-faq-tags-modal">
            <div class="make-faq-modal-header">
                <h2><?php echo esc_html__( 'Edit Tags', 'make-faq' ); ?></h2>
                <span class="make-faq-modal-close dashicons dashicons-no-alt"></span>
            </div>
            <div class="make-faq-modal-content">
                <form id="make-faq-tags-form">
                    <input type="hidden" id="make-faq-tags-post-id" name="post_id" value="">
                    <div class="make-faq-form-field">
                        <label for="make-faq-tags-select"><?php echo esc_html__( 'Tags', 'make-faq' ); ?></label>
                        <select id="make-faq-tags-select" name="tags[]" multiple="multiple" class="make-faq-select2">
                            <!-- Options will be populated via JavaScript -->
                        </select>
                        <p class="description"><?php echo esc_html__( 'Select one or more tags for this FAQ.', 'make-faq' ); ?></p>
                    </div>
                    <p class="description"><?php echo esc_html__( 'Type to search existing tags or enter a new tag name to create it.', 'make-faq' ); ?></p>
                </form>
            </div>
            <div class="make-faq-modal-footer">
                <div id="make-faq-tags-status"></div>
                <button type="button" id="make-faq-tags-cancel" class="button"><?php echo esc_html__( 'Cancel', 'make-faq' ); ?></button>
                <button type="button" id="make-faq-tags-save" class="button button-primary"><?php echo esc_html__( 'Save Tags', 'make-faq' ); ?></button>
            </div>
        </div>
        <?php
    }

    /**
     * Add custom admin columns to the FAQ post type.
     *
     * @since    1.0.0
     * @param    array    $columns    The existing columns.
     * @return   array                The modified columns.
     */
    public function set_custom_edit_make_faq_columns( $columns ) {
        $new_columns = array();

        // Add checkbox column first
        if ( isset( $columns['cb'] ) ) {
            $new_columns['cb'] = $columns['cb'];
        }

        // Add title column (question)
        $new_columns['title'] = __( 'Question', 'make-faq' );

        // Add answer excerpt column
        $new_columns['answer'] = __( 'Answer', 'make-faq' );

        // Add categories and tags
        $new_columns['faq_categories'] = __( 'Categories', 'make-faq' );
        $new_columns['faq_tags'] = __( 'Tags', 'make-faq' );

        // Add actions column (renamed from quick_edit)
        $new_columns['actions'] = __( 'Actions', 'make-faq' );

        return $new_columns;
    }

    /**
     * Add content to custom admin columns for the FAQ post type.
     *
     * @since    1.0.0
     * @param    string    $column     The column name.
     * @param    int       $post_id    The post ID.
     */
    public function custom_make_faq_column( $column, $post_id ) {
        // Debug log to see which columns are being processed
        
        // Only process for make_faq post type to prevent duplicate processing
        if (get_post_type($post_id) !== 'make_faq') {
            return;
        }
        
        switch ( $column ) {
            case 'answer':
                $content = get_post_field( 'post_content', $post_id );
                echo wp_trim_words( $content, 20, '...' );
                break;

            case 'faq_categories':
                // Debug output if debug mode is enabled
                $options = get_option( 'make_faq_settings', array() );
                $debug_mode = isset( $options['debug_mode'] ) ? (bool) $options['debug_mode'] : false;
                
                echo '<div class="make-faq-category-content">';
                
                // Get terms directly using wp_get_object_terms
                $terms = wp_get_object_terms( $post_id, 'faq_category' );
                
                if ( $debug_mode ) {
                    echo '<!-- Debug: ' . print_r($terms, true) . ' -->';
                }
                
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                    $category_links = array();
                    foreach ( $terms as $term ) {
                        $category_links[] = sprintf(
                            '<a href="%s">%s</a>',
                            esc_url( admin_url( 'edit.php?post_type=make_faq&faq_category=' . $term->slug ) ),
                            esc_html( $term->name )
                        );
                    }
                    echo '<div class="make-faq-terms-display">' . implode( ', ', $category_links ) . '</div>';
                } else {
                    echo '<div class="make-faq-terms-display">—</div>';
                    if ( $debug_mode ) {
                        if ( is_wp_error( $terms ) ) {
                            echo '<!-- Error: ' . esc_html( $terms->get_error_message() ) . ' -->';
                        } else {
                            echo '<!-- No terms found -->';
                        }
                    }
                }
                echo '</div>';
                echo '<button type="button" class="button make-faq-edit-categories" data-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Edit', 'make-faq' ) . '</button>';
                break;

            case 'faq_tags':
                // Debug output if debug mode is enabled
                $options = get_option( 'make_faq_settings', array() );
                $debug_mode = isset( $options['debug_mode'] ) ? (bool) $options['debug_mode'] : false;
                
                echo '<div class="make-faq-tag-content">';
                
                // Get terms directly using wp_get_object_terms
                $terms = wp_get_object_terms( $post_id, 'faq_tag' );
                
                if ( $debug_mode ) {
                    echo '<!-- Debug: ' . print_r($terms, true) . ' -->';
                }
                
                if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
                    $tag_links = array();
                    foreach ( $terms as $term ) {
                        $tag_links[] = sprintf(
                            '<a href="%s">%s</a>',
                            esc_url( admin_url( 'edit.php?post_type=make_faq&faq_tag=' . $term->slug ) ),
                            esc_html( $term->name )
                        );
                    }
                    echo '<div class="make-faq-terms-display">' . implode( ', ', $tag_links ) . '</div>';
                } else {
                    echo '<div class="make-faq-terms-display">—</div>';
                    if ( $debug_mode ) {
                        if ( is_wp_error( $terms ) ) {
                            echo '<!-- Error: ' . esc_html( $terms->get_error_message() ) . ' -->';
                        } else {
                            echo '<!-- No terms found -->';
                        }
                    }
                }
                echo '</div>';
                echo '<button type="button" class="button make-faq-edit-tags" data-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Edit', 'make-faq' ) . '</button>';
                break;

            case 'actions':
                echo '<div class="make-faq-actions">';
                echo '<button type="button" class="button make-faq-quick-edit" data-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Edit', 'make-faq' ) . '</button>';
                echo '<button type="button" class="button make-faq-trash" data-id="' . esc_attr( $post_id ) . '">' . esc_html__( 'Trash', 'make-faq' ) . '</button>';
                echo '</div>';
                break;
        }
    }

    /**
     * Register admin settings.
     *
     * @since    1.0.0
     */
    public function register_admin_settings() {
        register_setting( 'make_faq_settings', 'make_faq_settings' );

        add_settings_section(
            'make_faq_general_settings',
            __( 'General Settings', 'make-faq' ),
            array( $this, 'render_general_settings_section' ),
            'make_faq_settings'
        );

        add_settings_field(
            'make_faq_display_style',
            __( 'Default Display Style', 'make-faq' ),
            array( $this, 'render_display_style_field' ),
            'make_faq_settings',
            'make_faq_general_settings'
        );

        add_settings_field(
            'make_faq_schema_enabled',
            __( 'Enable Schema Markup', 'make-faq' ),
            array( $this, 'render_schema_enabled_field' ),
            'make_faq_settings',
            'make_faq_general_settings'
        );
        
        add_settings_field(
            'make_faq_debug_mode',
            __( 'Debug Mode', 'make-faq' ),
            array( $this, 'render_debug_mode_field' ),
            'make_faq_settings',
            'make_faq_general_settings'
        );
    }

    /**
     * Render the general settings section.
     *
     * @since    1.0.0
     */
    public function render_general_settings_section() {
        echo '<p>' . esc_html__( 'Configure the general settings for the FAQ plugin.', 'make-faq' ) . '</p>';
    }

    /**
     * Render the display style field.
     *
     * @since    1.0.0
     */
    public function render_display_style_field() {
        $options = get_option( 'make_faq_settings', array() );
        $display_style = isset( $options['display_style'] ) ? $options['display_style'] : 'accordion';
        ?>
        <select name="make_faq_settings[display_style]">
            <option value="accordion" <?php selected( $display_style, 'accordion' ); ?>><?php esc_html_e( 'Accordion', 'make-faq' ); ?></option>
            <option value="toggle" <?php selected( $display_style, 'toggle' ); ?>><?php esc_html_e( 'Toggle', 'make-faq' ); ?></option>
            <option value="list" <?php selected( $display_style, 'list' ); ?>><?php esc_html_e( 'List', 'make-faq' ); ?></option>
        </select>
        <p class="description"><?php esc_html_e( 'Select the default display style for FAQs.', 'make-faq' ); ?></p>
        <?php
    }

    /**
     * Render the schema enabled field.
     *
     * @since    1.0.0
     */
    public function render_schema_enabled_field() {
        $options = get_option( 'make_faq_settings', array() );
        $schema_enabled = isset( $options['schema_enabled'] ) ? $options['schema_enabled'] : 1;
        ?>
        <label>
            <input type="checkbox" name="make_faq_settings[schema_enabled]" value="1" <?php checked( $schema_enabled, 1 ); ?>>
            <?php esc_html_e( 'Enable FAQPage schema markup for better SEO', 'make-faq' ); ?>
        </label>
        <?php
    }
    
    /**
     * Render the debug mode field.
     *
     * @since    1.0.3
     */
    public function render_debug_mode_field() {
        $options = get_option( 'make_faq_settings', array() );
        $debug_mode = isset( $options['debug_mode'] ) ? $options['debug_mode'] : 0;
        ?>
        <label>
            <input type="checkbox" name="make_faq_settings[debug_mode]" value="1" <?php checked( $debug_mode, 1 ); ?>>
            <?php esc_html_e( 'Enable AJAX debugging in browser console', 'make-faq' ); ?>
        </label>
        <p class="description"><?php esc_html_e( 'When enabled, AJAX requests and responses will be logged to the browser console for troubleshooting.', 'make-faq' ); ?></p>
        <?php
    }

    /**
     * Add admin menu items.
     *
     * @since    1.0.0
     */
    public function add_admin_menu() {
        // Remove the "Add New FAQ" submenu item
        global $submenu;
        if ( isset( $submenu['edit.php?post_type=make_faq'] ) ) {
            foreach ( $submenu['edit.php?post_type=make_faq'] as $key => $item ) {
                if ( $item[2] === 'post-new.php?post_type=make_faq' ) {
                    unset( $submenu['edit.php?post_type=make_faq'][$key] );
                    break;
                }
            }
        }
        
        add_submenu_page(
            'edit.php?post_type=make_faq',
            __( 'FAQ Settings', 'make-faq' ),
            __( 'Settings', 'make-faq' ),
            'manage_options',
            'make_faq_settings',
            array( $this, 'render_settings_page' )
        );

        add_submenu_page(
            'edit.php?post_type=make_faq',
            __( 'FAQ Analytics', 'make-faq' ),
            __( 'Analytics', 'make-faq' ),
            'manage_options',
            'make_faq_analytics',
            array( $this, 'render_analytics_page' )
        );

        add_submenu_page(
            'edit.php?post_type=make_faq',
            __( 'Import/Export FAQs', 'make-faq' ),
            __( 'Import/Export', 'make-faq' ),
            'manage_options',
            'make_faq_import_export',
            array( $this, 'render_import_export_page' )
        );
    }

    /**
     * Render the settings page.
     *
     * @since    1.0.0
     */
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields( 'make_faq_settings' );
                do_settings_sections( 'make_faq_settings' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render the analytics page.
     *
     * @since    1.0.0
     */
    public function render_analytics_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <div class="make-faq-analytics-container">
                <div class="make-faq-analytics-card">
                    <h2><?php esc_html_e( 'Most Popular FAQs', 'make-faq' ); ?></h2>
                    <div class="make-faq-analytics-content">
                        <?php $this->render_popular_faqs(); ?>
                    </div>
                </div>
                <div class="make-faq-analytics-card">
                    <h2><?php esc_html_e( 'Search Analytics', 'make-faq' ); ?></h2>
                    <div class="make-faq-analytics-content">
                        <?php $this->render_search_analytics(); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render the popular FAQs section.
     *
     * @since    1.0.0
     */
    private function render_popular_faqs() {
        // Get view count data
        $popular_faqs = $this->get_popular_faqs();

        if ( empty( $popular_faqs ) ) {
            echo '<p>' . esc_html__( 'No data available yet. FAQs will appear here once they start receiving views.', 'make-faq' ) . '</p>';
            return;
        }

        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'FAQ', 'make-faq' ) . '</th>';
        echo '<th>' . esc_html__( 'Views', 'make-faq' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        foreach ( $popular_faqs as $faq ) {
            echo '<tr>';
            echo '<td><a href="' . esc_url( get_edit_post_link( $faq->ID ) ) . '">' . esc_html( $faq->post_title ) . '</a></td>';
            echo '<td>' . esc_html( get_post_meta( $faq->ID, '_make_faq_view_count', true ) ) . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
    }

    /**
     * Get popular FAQs based on view count.
     *
     * @since    1.0.0
     * @return   array    The popular FAQs.
     */
    private function get_popular_faqs() {
        $args = array(
            'post_type'      => 'make_faq',
            'posts_per_page' => 10,
            'meta_key'       => '_make_faq_view_count',
            'orderby'        => 'meta_value_num',
            'order'          => 'DESC',
        );

        return get_posts( $args );
    }

    /**
     * Render the search analytics section.
     *
     * @since    1.0.0
     */
    private function render_search_analytics() {
        // Get search data
        $search_terms = get_option( 'make_faq_search_terms', array() );

        if ( empty( $search_terms ) ) {
            echo '<p>' . esc_html__( 'No search data available yet. Search terms will appear here once users start searching.', 'make-faq' ) . '</p>';
            return;
        }

        // Sort by count
        arsort( $search_terms );

        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th>' . esc_html__( 'Search Term', 'make-faq' ) . '</th>';
        echo '<th>' . esc_html__( 'Count', 'make-faq' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        $count = 0;
        foreach ( $search_terms as $term => $searches ) {
            echo '<tr>';
            echo '<td>' . esc_html( $term ) . '</td>';
            echo '<td>' . esc_html( $searches ) . '</td>';
            echo '</tr>';

            $count++;
            if ( $count >= 10 ) {
                break;
            }
        }

        echo '</tbody></table>';
    }

    /**
     * Render the import/export page.
     *
     * @since    1.0.0
     */
    public function render_import_export_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            
            <div class="make-faq-import-export-container">
                <?php
                $make_faq_notice = get_transient( 'make_faq_import_result' );
                if ( $make_faq_notice ) {
                    delete_transient( 'make_faq_import_result' );
                    $processed    = isset( $make_faq_notice['processed'] ) ? (int) $make_faq_notice['processed'] : 0;
                    $created      = isset( $make_faq_notice['created'] ) ? (int) $make_faq_notice['created'] : 0;
                    $updated      = isset( $make_faq_notice['updated'] ) ? (int) $make_faq_notice['updated'] : 0;
                    $skipped      = isset( $make_faq_notice['skipped'] ) ? (int) $make_faq_notice['skipped'] : 0;
                    $errors_count = isset( $make_faq_notice['errors_count'] ) ? (int) $make_faq_notice['errors_count'] : 0;
                    $deleted     = isset( $make_faq_notice['deleted'] ) ? (int) $make_faq_notice['deleted'] : 0;
                    $class        = $errors_count > 0 ? 'notice notice-warning' : 'notice notice-success';
                    echo '<div class="' . esc_attr( $class ) . '"><p>' . sprintf( esc_html__( 'Import complete. Processed: %1$d, Created: %2$d, Updated: %3$d, Skipped: %4$d, Deleted: %5$d. Errors: %6$d', 'make-faq' ), $processed, $created, $updated, $skipped, $deleted, $errors_count ) . '</p></div>';
                }
                ?>

                <div class="postbox" style="width:100%; padding:20px;">
                    <h2 class="hndle"><span><?php esc_html_e( 'Import Help & Guidelines', 'make-faq' ); ?></span></h2>
                    <div class="inside">
                        <p><?php esc_html_e( 'Use CSV with the following columns/fields. Notes on behavior:', 'make-faq' ); ?></p>
                        <ul class="ul-disc">
                            <li><strong><?php esc_html_e( 'ID', 'make-faq' ); ?></strong>: <?php esc_html_e( 'Leave blank to create a new FAQ. Provide an existing FAQ ID to update that FAQ.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Question', 'make-faq' ); ?></strong>: <?php esc_html_e( 'FAQ title. If blank and Answer is blank, the row is skipped.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Answer', 'make-faq' ); ?></strong>: <?php esc_html_e( 'FAQ content. If blank and Question is blank, the row is skipped.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Status', 'make-faq' ); ?></strong>: <?php esc_html_e( 'One of publish, draft, pending, or private. If blank or invalid, defaults to draft.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Date', 'make-faq' ); ?></strong>: <?php esc_html_e( 'Optional GMT datetime (e.g., 2025-08-14 10:30:00). If blank, WordPress sets the current date.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Categories', 'make-faq' ); ?></strong>: <?php esc_html_e( 'Semicolon-separated names. Missing terms will be created. A Default Category selected in the form is also applied.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Tags', 'make-faq' ); ?></strong>: <?php esc_html_e( 'Semicolon-separated names. Missing tags will be created.', 'make-faq' ); ?></li>
                            <li><?php esc_html_e( 'Blanks: If a field is blank, it is not forced except where noted (e.g., blank Status => draft). For updates, blank Question/Answer will overwrite with blank if provided; leave columns empty to retain current values by excluding them in your import file.', 'make-faq' ); ?></li>
                            <li><strong><?php esc_html_e( 'Import Mode', 'make-faq' ); ?></strong>: <?php esc_html_e( 'Amend (default): creates new and updates existing FAQs; does not delete any. Replace: synchronizes FAQs by deleting those not present in the import file. If a Default Category is selected, Replace will only delete FAQs within that category.', 'make-faq' ); ?></li>
                        </ul>
                        <p><?php esc_html_e( 'Tip: Export first to get the correct columns, then modify and re-import.', 'make-faq' ); ?></p>
                    </div>
                </div>
                <div class="make-faq-card">
                    <h2><?php esc_html_e( 'Export FAQs', 'make-faq' ); ?></h2>
                    <p><?php esc_html_e( 'Export your FAQs to a CSV file.', 'make-faq' ); ?></p>
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                        <input type="hidden" name="action" value="make_faq_export">
                        <?php wp_nonce_field( 'make_faq_export_nonce', 'make_faq_export_nonce' ); ?>
                        
                        
                        <p>
                            <label for="export_category"><?php esc_html_e( 'Category (optional):', 'make-faq' ); ?></label>
                            <select name="export_category" id="export_category">
                                <option value=""><?php esc_html_e( 'All Categories', 'make-faq' ); ?></option>
                                <?php
                                $categories = get_terms( array(
                                    'taxonomy'   => 'faq_category',
                                    'hide_empty' => false,
                                ) );
                                
                                foreach ( $categories as $category ) {
                                    echo '<option value="' . esc_attr( $category->slug ) . '">' . esc_html( $category->name ) . '</option>';
                                }
                                ?>
                            </select>
                        </p>
                        
                        <?php submit_button( __( 'Export', 'make-faq' ), 'primary', 'submit', false ); ?>
                    </form>
                </div>
                
                <div class="make-faq-card">
                    <h2><?php esc_html_e( 'Import FAQs', 'make-faq' ); ?></h2>
                    <p><?php esc_html_e( 'Import FAQs from a CSV file.', 'make-faq' ); ?></p>
                    <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="make_faq_import">
                        <?php wp_nonce_field( 'make_faq_import_nonce', 'make_faq_import_nonce' ); ?>
                        
                        <p>
                            <label for="import_file"><?php esc_html_e( 'Choose a file:', 'make-faq' ); ?></label>
                            <input type="file" name="import_file" id="import_file" accept=".csv">
                        </p>
                        
                        <p>
                            <label for="import_category"><?php esc_html_e( 'Default Category (optional):', 'make-faq' ); ?></label>
                            <select name="import_category" id="import_category">
                                <option value=""><?php esc_html_e( 'None', 'make-faq' ); ?></option>
                                <?php
                                foreach ( $categories as $category ) {
                                    echo '<option value="' . esc_attr( $category->slug ) . '">' . esc_html( $category->name ) . '</option>';
                                }
                                ?>
                            </select>
                        </p>
                        
                        <p>
                            <label for="import_mode"><?php esc_html_e( 'Import Mode:', 'make-faq' ); ?></label>
                            <select name="import_mode" id="import_mode">
                                <option value="amend" selected><?php esc_html_e( 'Amend (create new and update existing; do not delete missing)', 'make-faq' ); ?></option>
                                <option value="replace"><?php esc_html_e( 'Replace (synchronize: delete existing FAQs not in the file)', 'make-faq' ); ?></option>
                            </select>
                        </p>
                        
                        <?php submit_button( __( 'Import', 'make-faq' ), 'primary', 'submit', false ); ?>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Handle Export (admin-post.php?action=make_faq_export)
     */
    public function handle_export() {
        // Permission check: page already requires manage_options
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to export FAQs.', 'make-faq' ) );
        }

        // Nonce check
        if ( ! isset( $_POST['make_faq_export_nonce'] ) || ! wp_verify_nonce( $_POST['make_faq_export_nonce'], 'make_faq_export_nonce' ) ) {
            wp_die( esc_html__( 'Security check failed for export.', 'make-faq' ) );
        }

        // Force CSV format only
        $format = 'csv';

        // Optional category filter (slug)
        $category = isset( $_POST['export_category'] ) ? sanitize_text_field( wp_unslash( $_POST['export_category'] ) ) : '';

        // Build query for FAQs
        $args = array(
            'post_type'      => 'make_faq',
            'posts_per_page' => -1,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'post_status'    => 'any',
        );

        if ( $category !== '' ) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'faq_category',
                    'field'    => 'slug',
                    'terms'    => array( $category ),
                ),
            );
        }

        $faqs = get_posts( $args );

        // If no FAQs exist, create a single in-memory dummy row so CSV/JSON structure is preserved.
        $export_has_dummy = false;
        if ( empty( $faqs ) ) {
            $export_has_dummy = true;
            $dummy            = (object) array(
                'ID'            => 0,
                'post_title'    => __( 'Example question', 'make-faq' ),
                'post_content'  => __( 'Example answer. Replace me with your real FAQ content.', 'make-faq' ),
                // Ensure exported row shows as published so duplicated rows inherit this setting
                'post_status'   => 'publish',
                'post_date_gmt' => gmdate( 'Y-m-d H:i:s' ),
            );
            $faqs = array( $dummy );
        }

        // Prepare filename
        $site  = preg_replace( '/[^a-z0-9\-]+/i', '-', wp_parse_url( home_url(), PHP_URL_HOST ) );
        $date  = gmdate( 'Y-m-d-His' );
        $fname = sprintf( 'make-faq-export-%s-%s.%s', $site, $date, $format );

        // Prevent any previously buffered output from breaking file download
        if ( function_exists( 'ob_get_length' ) && ob_get_length() ) {
            ob_end_clean();
        }

        if ( 'json' === $format ) {
            // Build array
            $data = array();
            foreach ( $faqs as $faq ) {
                $categories = $export_has_dummy && (int) $faq->ID === 0
                    ? array()
                    : wp_get_post_terms( $faq->ID, 'faq_category', array( 'fields' => 'names' ) );
                $tags = $export_has_dummy && (int) $faq->ID === 0
                    ? array()
                    : wp_get_post_terms( $faq->ID, 'faq_tag', array( 'fields' => 'names' ) );
                $data[] = array(
                    'id'        => ($export_has_dummy && (int) $faq->ID === 0) ? '' : $faq->ID,
                    'question'  => $faq->post_title,
                    'answer'    => $faq->post_content,
                    'status'    => $faq->post_status,
                    'date'      => $faq->post_date_gmt,
                    'categories'=> is_array( $categories ) ? $categories : array(),
                    'tags'      => is_array( $tags ) ? $tags : array(),
                );
            }

            header( 'Content-Type: application/json; charset=utf-8' );
            header( 'Content-Disposition: attachment; filename=' . $fname );
            echo wp_json_encode( $data );
            exit;
        }

        // Default: CSV
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $fname );

        $output = fopen( 'php://output', 'w' );
        // BOM for Excel compatibility
        fprintf( $output, chr(0xEF) . chr(0xBB) . chr(0xBF) );

        // Header row
        fputcsv( $output, array( 'ID', 'Question', 'Answer', 'Status', 'Date', 'Categories', 'Tags' ) );

        foreach ( $faqs as $faq ) {
            $cats = $export_has_dummy && (int) $faq->ID === 0
                ? array()
                : wp_get_post_terms( $faq->ID, 'faq_category', array( 'fields' => 'names' ) );
            $tags = $export_has_dummy && (int) $faq->ID === 0
                ? array()
                : wp_get_post_terms( $faq->ID, 'faq_tag', array( 'fields' => 'names' ) );
            fputcsv( $output, array(
                ($export_has_dummy && (int) $faq->ID === 0) ? '' : $faq->ID,
                $faq->post_title,
                $faq->post_content,
                $faq->post_status,
                $faq->post_date_gmt,
                implode( '; ', is_array( $cats ) ? $cats : array() ),
                implode( '; ', is_array( $tags ) ? $tags : array() ),
            ) );
        }

        fclose( $output );
        exit;
    }

    /**
     * Handle Import (admin-post.php?action=make_faq_import)
     */
    public function handle_import() {
        // Permission check
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to import FAQs.', 'make-faq' ) );
        }

        // Nonce check
        if ( ! isset( $_POST['make_faq_import_nonce'] ) || ! wp_verify_nonce( $_POST['make_faq_import_nonce'], 'make_faq_import_nonce' ) ) {
            wp_die( esc_html__( 'Security check failed for import.', 'make-faq' ) );
        }

        // File validation
        if ( ! isset( $_FILES['import_file'] ) || empty( $_FILES['import_file']['tmp_name'] ) ) {
            wp_die( esc_html__( 'No import file uploaded.', 'make-faq' ) );
        }

        $file      = $_FILES['import_file'];
        $tmp_path  = $file['tmp_name'];
        $orig_name = $file['name'];
        $ext       = strtolower( pathinfo( $orig_name, PATHINFO_EXTENSION ) );

        // Optional default category (slug)
        $default_category_slug = isset( $_POST['import_category'] ) ? sanitize_text_field( wp_unslash( $_POST['import_category'] ) ) : '';

        $results = array(
            'processed' => 0,
            'created'   => 0,
            'updated'   => 0,
            'skipped'   => 0,
            'deleted'   => 0,
            'errors'    => array(),
            'messages'  => array(),
        );

        // Helper to ensure term exists by name (create if missing), returns term_id
        $ensure_terms = function( $names_str, $taxonomy ) {
            $ids = array();
            $names = array();
            if ( is_array( $names_str ) ) {
                $names = $names_str;
            } else {
                $parts = array_map( 'trim', explode( ';', (string) $names_str ) );
                foreach ( $parts as $p ) { if ( $p !== '' ) { $names[] = $p; } }
            }
            foreach ( $names as $name ) {
                if ( $name === '' ) { continue; }
                $term = term_exists( $name, $taxonomy );
                if ( 0 === $term || null === $term ) {
                    $insert = wp_insert_term( $name, $taxonomy );
                    if ( is_wp_error( $insert ) ) {
                        continue;
                    }
                    $term_id = (int) $insert['term_id'];
                } else {
                    $term_id = is_array( $term ) ? (int) $term['term_id'] : (int) $term;
                }
                $ids[] = $term_id;
            }
            return $ids;
        };

        // Parse input rows (CSV only)
        $rows = array();
        if ( 'csv' !== $ext ) {
            wp_die( esc_html__( 'Only CSV files are supported for import.', 'make-faq' ) );
        }
        // Read CSV with header row
        $fh = fopen( $tmp_path, 'r' );
        if ( ! $fh ) {
            wp_die( esc_html__( 'Unable to open CSV import file.', 'make-faq' ) );
        }
        // Try to skip BOM
        $bom = fread( $fh, 3 );
        if ( $bom !== chr(0xEF) . chr(0xBB) . chr(0xBF) ) {
            rewind( $fh );
        }
        $header = fgetcsv( $fh );
        if ( ! $header ) {
            fclose( $fh );
            wp_die( esc_html__( 'CSV file is empty or missing header.', 'make-faq' ) );
        }
        $map = array();
        foreach ( $header as $i => $col ) {
            $col = trim( $col );
            $map[$i] = strtolower( $col );
        }
        while ( ( $cols = fgetcsv( $fh ) ) !== false ) {
            if ( count( array_filter( $cols, function( $v ){ return trim( (string) $v ) !== ''; } ) ) === 0 ) {
                continue; // skip blank lines
            }
            $row = array(
                'ID'         => '',
                'Question'   => '',
                'Answer'     => '',
                'Status'     => '',
                'Date'       => '',
                'Categories' => '',
                'Tags'       => '',
            );
            foreach ( $cols as $i => $val ) {
                $key = isset( $map[$i] ) ? $map[$i] : '';
                if ( $key === '' ) { continue; }
                switch ( $key ) {
                    case 'id':         $row['ID'] = $val; break;
                    case 'question':   $row['Question'] = $val; break;
                    case 'answer':     $row['Answer'] = $val; break;
                    case 'status':     $row['Status'] = $val; break;
                    case 'date':       $row['Date'] = $val; break;
                    case 'categories': $row['Categories'] = $val; break;
                    case 'tags':       $row['Tags'] = $val; break;
                }
            }
            $rows[] = $row;
        }
        fclose( $fh );

        // Import mode: amend (default) or replace (synchronize)
        $import_mode = isset( $_POST['import_mode'] ) ? sanitize_key( wp_unslash( $_POST['import_mode'] ) ) : 'amend';
        if ( ! in_array( $import_mode, array( 'amend', 'replace' ), true ) ) {
            $import_mode = 'amend';
        }

        $imported_ids = array();

        // Process rows
        foreach ( $rows as $index => $row ) {
            $results['processed']++;
            $id        = isset( $row['ID'] ) ? absint( $row['ID'] ) : 0;
            $question  = isset( $row['Question'] ) ? wp_unslash( $row['Question'] ) : '';
            $answer    = isset( $row['Answer'] ) ? wp_unslash( $row['Answer'] ) : '';
            $status    = isset( $row['Status'] ) ? sanitize_key( $row['Status'] ) : '';
            $date_gmt  = isset( $row['Date'] ) ? sanitize_text_field( $row['Date'] ) : '';
            $cats_in   = isset( $row['Categories'] ) ? $row['Categories'] : '';
            $tags_in   = isset( $row['Tags'] ) ? $row['Tags'] : '';

            // Basic validation: need a question or answer to proceed
            if ( $question === '' && $answer === '' ) {
                $results['skipped']++;
                $results['messages'][] = sprintf( __( 'Row %d skipped: both Question and Answer are blank.', 'make-faq' ), $index + 2 );
                continue;
            }

            // Determine post status; default to 'publish' if provided, else 'draft'
            $post_status = in_array( $status, array( 'publish', 'draft', 'pending', 'private' ), true ) ? $status : 'draft';

            // Prepare post array
            $postarr = array(
                'post_type'   => 'make_faq',
                'post_title'  => $question,
                'post_content'=> $answer,
                'post_status' => $post_status,
            );
            // Date handling (expects GMT string); if blank, let WP set current date
            if ( $date_gmt !== '' ) {
                $time = strtotime( $date_gmt );
                if ( $time ) {
                    $postarr['post_date_gmt'] = gmdate( 'Y-m-d H:i:s', $time );
                    $postarr['post_date']     = get_date_from_gmt( $postarr['post_date_gmt'] );
                }
            }

            $is_update = false;
            if ( $id > 0 && get_post_type( $id ) === 'make_faq' ) {
                $postarr['ID'] = $id;
                $is_update = true;
                $new_id = wp_update_post( wp_slash( $postarr ), true );
            } else {
                $new_id = wp_insert_post( wp_slash( $postarr ), true );
            }

            if ( is_wp_error( $new_id ) ) {
                $results['errors'][] = sprintf( __( 'Row %1$d error: %2$s', 'make-faq' ), $index + 2, $new_id->get_error_message() );
                continue;
            }

            // Apply default category if provided
            if ( $default_category_slug !== '' ) {
                $def_term = get_term_by( 'slug', $default_category_slug, 'faq_category' );
                if ( $def_term && ! is_wp_error( $def_term ) ) {
                    wp_set_post_terms( $new_id, array( (int) $def_term->term_id ), 'faq_category', true );
                }
            }

            // Categories from file
            if ( $cats_in !== '' ) {
                $cat_ids = $ensure_terms( $cats_in, 'faq_category' );
                if ( ! empty( $cat_ids ) ) {
                    wp_set_post_terms( $new_id, $cat_ids, 'faq_category', true );
                }
            }
            // Tags from file
            if ( $tags_in !== '' ) {
                $tag_ids = $ensure_terms( $tags_in, 'faq_tag' );
                if ( ! empty( $tag_ids ) ) {
                    wp_set_post_terms( $new_id, $tag_ids, 'faq_tag', false );
                }
            }

            if ( $is_update ) {
                $results['updated']++;
                $results['messages'][] = sprintf( __( 'Row %1$d: Updated FAQ #%2$d', 'make-faq' ), $index + 2, $new_id );
            } else {
                $results['created']++;
                $results['messages'][] = sprintf( __( 'Row %1$d: Created FAQ #%2$d', 'make-faq' ), $index + 2, $new_id );
            }

            if ( $new_id > 0 ) {
                $imported_ids[] = (int) $new_id;
            }
        }

        // Replace mode: delete FAQs not present in the import file
        if ( $import_mode === 'replace' ) {
            $query_args = array(
                'post_type'      => 'make_faq',
                'posts_per_page' => -1,
                'fields'         => 'ids',
                'post_status'    => 'any',
            );
            if ( $default_category_slug !== '' ) {
                $query_args['tax_query'] = array(
                    array(
                        'taxonomy' => 'faq_category',
                        'field'    => 'slug',
                        'terms'    => array( $default_category_slug ),
                    ),
                );
            }
            $existing_ids = get_posts( $query_args );
            if ( is_array( $existing_ids ) ) {
                $to_delete = array_diff( array_map( 'intval', $existing_ids ), array_map( 'intval', $imported_ids ) );
                foreach ( $to_delete as $del_id ) {
                    if ( get_post_type( $del_id ) === 'make_faq' ) {
                        wp_trash_post( $del_id );
                        $results['deleted']++;
                    }
                }
            }
        }

        // Store summary in a transient and redirect back to Import/Export page
        set_transient( 'make_faq_import_result', array(
            'processed'    => (int) $results['processed'],
            'created'      => (int) $results['created'],
            'updated'      => (int) $results['updated'],
            'skipped'      => (int) $results['skipped'],
            'deleted'      => (int) $results['deleted'],
            'errors_count' => (int) count( (array) $results['errors'] ),
        ), 60 );

        $redirect = add_query_arg(
            array(
                'post_type' => 'make_faq',
                'page'      => 'make_faq_import_export',
                'import'    => '1',
            ),
            admin_url( 'edit.php' )
        );
        wp_redirect( $redirect );
        exit;
    }

    /**
     * Render the Import Results page, including Help panel
     */
    private function render_import_results_page( $results ) {
        if ( ! is_array( $results ) ) { $results = array(); }
        $processed = isset( $results['processed'] ) ? (int) $results['processed'] : 0;
        $created   = isset( $results['created'] ) ? (int) $results['created'] : 0;
        $updated   = isset( $results['updated'] ) ? (int) $results['updated'] : 0;
        $skipped   = isset( $results['skipped'] ) ? (int) $results['skipped'] : 0;
        $errors    = isset( $results['errors'] ) ? (array) $results['errors'] : array();
        $messages  = isset( $results['messages'] ) ? (array) $results['messages'] : array();

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'Import FAQs', 'make-faq' ) . '</h1>';

        echo '<div class="notice notice-success"><p>' . sprintf( esc_html__( 'Processed: %1$d, Created: %2$d, Updated: %3$d, Skipped: %4$d', 'make-faq' ), $processed, $created, $updated, $skipped ) . '</p></div>';

        if ( ! empty( $messages ) ) {
            echo '<h2>' . esc_html__( 'Details', 'make-faq' ) . '</h2>';
            echo '<ul class="ul-disc">';
            foreach ( $messages as $msg ) {
                echo '<li>' . esc_html( $msg ) . '</li>';
            }
            echo '</ul>';
        }

        if ( ! empty( $errors ) ) {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Some rows could not be imported:', 'make-faq' ) . '</p><ul class="ul-disc">';
            foreach ( $errors as $err ) {
                echo '<li>' . esc_html( $err ) . '</li>';
            }
            echo '</ul></div>';
        }

        // Help panel
        echo '<div class="postbox" style="max-width: 900px;">';
        echo '<h2 class="hndle"><span>' . esc_html__( 'Import Help & Guidelines', 'make-faq' ) . '</span></h2>';
        echo '<div class="inside">';
        echo '<p>' . esc_html__( 'Use CSV or JSON with the following columns/fields. Notes on behavior:', 'make-faq' ) . '</p>';
        echo '<ul class="ul-disc">';
        echo '<li><strong>' . esc_html__( 'ID', 'make-faq' ) . '</strong>: ' . esc_html__( 'Leave blank to create a new FAQ. Provide an existing FAQ ID to update that FAQ.', 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Question', 'make-faq' ) . '</strong>: ' . esc_html__( 'FAQ title. If blank and Answer is blank, the row is skipped.', 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Answer', 'make-faq' ) . '</strong>: ' . esc_html__( 'FAQ content. If blank and Question is blank, the row is skipped.', 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Status', 'make-faq' ) . '</strong>: ' . esc_html__( "One of publish, draft, pending, or private. If blank or invalid, defaults to draft.", 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Date', 'make-faq' ) . '</strong>: ' . esc_html__( 'Optional GMT datetime (e.g., 2025-08-14 10:30:00). If blank, WordPress sets the current date.', 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Categories', 'make-faq' ) . '</strong>: ' . esc_html__( 'Semicolon-separated names. Missing terms will be created. A Default Category selected in the form is also applied.', 'make-faq' ) . '</li>';
        echo '<li><strong>' . esc_html__( 'Tags', 'make-faq' ) . '</strong>: ' . esc_html__( 'Semicolon-separated names. Missing tags will be created.', 'make-faq' ) . '</li>';
        echo '<li>' . esc_html__( 'Blanks: If a field is blank, it is not forced except where noted (e.g., blank Status => draft). For updates, blank Question/Answer will overwrite with blank if provided; leave columns empty to retain current values by excluding them in your import file.', 'make-faq' ) . '</li>';
        echo '</ul>';
        echo '<p>' . esc_html__( 'Tip: Export first to get the correct columns, then modify and re-import.', 'make-faq' ) . '</p>';
        echo '</div></div>';

        echo '<p><a class="button button-primary" href="' . esc_url( admin_url( 'edit.php?post_type=make_faq&page=make_faq_import_export' ) ) . '">' . esc_html__( 'Back to Import/Export', 'make-faq' ) . '</a></p>';

        echo '</div>';
    }

    /**
     * AJAX handler for getting FAQ data.
     *
     * @since    1.0.0
     */
    public function ajax_get_faq() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Get all available categories for the dropdown
        $all_categories = array();
        $terms = get_terms( array(
            'taxonomy' => 'faq_category',
            'hide_empty' => false,
        ) );
        
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $all_categories[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name
                );
            }
        }

        // Check if we're editing an existing FAQ
        if ( isset( $_POST['id'] ) && ! empty( $_POST['id'] ) ) {
            $faq_id = intval( $_POST['id'] );
            $faq = get_post( $faq_id );

            if ( ! $faq || 'make_faq' !== $faq->post_type ) {
                wp_send_json_error( array( 'message' => __( 'FAQ not found.', 'make-faq' ) ) );
            }

            // Get categories and tags
            $categories = wp_get_post_terms( $faq_id, 'faq_category', array( 'fields' => 'ids' ) );
            $tags = wp_get_post_terms( $faq_id, 'faq_tag', array( 'fields' => 'names' ) );

            // Prepare response
            $response = array(
                'id'             => $faq_id,
                'question'       => $faq->post_title,
                'answer'         => $faq->post_content,
                'categories'     => $categories,
                'tags'           => implode( ', ', $tags ),
                'all_categories' => $all_categories,
                'is_new'         => false
            );
        } else {
            // New FAQ
            $response = array(
                'id'             => '',
                'question'       => '',
                'answer'         => '',
                'categories'     => array(),
                'tags'           => '',
                'all_categories' => $all_categories,
                'is_new'         => true
            );
        }

        wp_send_json_success( $response );
    }

    /**
     * AJAX handler for saving FAQ data.
     *
     * @since    1.0.0
     */
    public function ajax_save_faq() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Check for required fields
        if ( ! isset( $_POST['question'] ) || empty( $_POST['question'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Question field is required.', 'make-faq' ) ) );
        }

        $question = sanitize_text_field( $_POST['question'] );
        $answer = isset( $_POST['answer'] ) ? wp_kses_post( $_POST['answer'] ) : '';
        $categories = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : array();
        $tags = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';

        // Check if we're creating a new FAQ or updating an existing one
        if ( isset( $_POST['id'] ) && ! empty( $_POST['id'] ) ) {
            // Update existing FAQ
            $faq_id = intval( $_POST['id'] );
            $faq_data = array(
                'ID'           => $faq_id,
                'post_title'   => $question,
                'post_content' => $answer,
            );

            $updated = wp_update_post( $faq_data );

            if ( is_wp_error( $updated ) ) {
                wp_send_json_error( array( 'message' => $updated->get_error_message() ) );
            }
        } else {
            // Create new FAQ
            $faq_data = array(
                'post_title'   => $question,
                'post_content' => $answer,
                'post_type'    => 'make_faq',
                'post_status'  => 'publish',
            );

            $faq_id = wp_insert_post( $faq_data );

            if ( is_wp_error( $faq_id ) ) {
                wp_send_json_error( array( 'message' => $faq_id->get_error_message() ) );
            }
        }

        // Update categories
        if ( ! empty( $categories ) ) {
            wp_set_object_terms( $faq_id, $categories, 'faq_category' );
        } else {
            wp_set_object_terms( $faq_id, array(), 'faq_category' );
        }

        // Update tags
        if ( ! empty( $tags ) ) {
            wp_set_object_terms( $faq_id, explode( ',', $tags ), 'faq_tag' );
        } else {
            wp_set_object_terms( $faq_id, array(), 'faq_tag' );
        }

        // Get updated FAQ data for response
        $updated_faq = get_post( $faq_id );
        $updated_categories = wp_get_post_terms( $faq_id, 'faq_category', array( 'fields' => 'ids' ) );
        $updated_tags = wp_get_post_terms( $faq_id, 'faq_tag', array( 'fields' => 'names' ) );

        // Prepare response
        $response = array(
            'id'         => $faq_id,
            'question'   => $updated_faq->post_title,
            'answer'     => wp_trim_words( $updated_faq->post_content, 20, '...' ),
            'categories' => $updated_categories,
            'tags'       => implode( ', ', $updated_tags ),
            'message'    => __( 'FAQ updated successfully.', 'make-faq' ),
        );

        wp_send_json_success( $response );
    }
    
    /**
     * AJAX handler for getting categories data.
     *
     * @since    1.0.4
     */
    public function ajax_get_categories() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Check for post ID
        if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No FAQ ID provided.', 'make-faq' ) ) );
        }

        $post_id = intval( $_POST['post_id'] );
        
        // Get all available categories
        $all_categories = array();
        $terms = get_terms( array(
            'taxonomy' => 'faq_category',
            'hide_empty' => false,
        ) );
        
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $all_categories[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name
                );
            }
        }
        
        // Get selected categories for this post
        $selected_categories = wp_get_post_terms( $post_id, 'faq_category', array( 'fields' => 'ids' ) );
        
        // Prepare response
        $response = array(
            'post_id' => $post_id,
            'all_categories' => $all_categories,
            'selected_categories' => $selected_categories,
        );
        
        wp_send_json_success( $response );
    }
    
    /**
     * AJAX handler for saving categories data.
     *
     * @since    1.0.4
     */
    public function ajax_save_categories() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Check for required fields
        if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No FAQ ID provided.', 'make-faq' ) ) );
        }

        $post_id = intval( $_POST['post_id'] );
        $categories = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : array();
        
        // Update categories
        if ( ! empty( $categories ) ) {
            wp_set_object_terms( $post_id, $categories, 'faq_category' );
        } else {
            wp_set_object_terms( $post_id, array(), 'faq_category' );
        }
        
        // Get updated categories for display
        $terms = get_the_terms( $post_id, 'faq_category' );
        $categories_html = '';
        
        if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
            $category_links = array();
            foreach ( $terms as $term ) {
                $category_links[] = sprintf(
                    '<a href="%s">%s</a>',
                    esc_url( admin_url( 'edit.php?post_type=make_faq&faq_category=' . $term->slug ) ),
                    esc_html( $term->name )
                );
            }
            $categories_html = '<div class="make-faq-terms-display">' . implode( ', ', $category_links ) . '</div>';
        } else {
            $categories_html = '<div class="make-faq-terms-display">—</div>';
        }
        
        // Prepare response
        $response = array(
            'post_id' => $post_id,
            'categories_html' => $categories_html,
            'message' => __( 'Categories updated successfully.', 'make-faq' ),
        );
        
        wp_send_json_success( $response );
    }
    
    /**
     * AJAX handler for getting tags data.
     *
     * @since    1.0.4
     */
    public function ajax_get_tags() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Check for post ID
        if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No FAQ ID provided.', 'make-faq' ) ) );
        }

        $post_id = intval( $_POST['post_id'] );
        
        // Get tags for this post
        $tags = wp_get_post_terms( $post_id, 'faq_tag', array( 'fields' => 'names' ) );
        
        // Prepare response
        $response = array(
            'post_id' => $post_id,
            'tags' => implode( ', ', $tags ),
        );
        
        wp_send_json_success( $response );
    }
    
    /**
     * AJAX handler for saving tags data.
     *
     * @since    1.0.4
     */
    public function ajax_save_tags() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to edit FAQs.', 'make-faq' ) ) );
        }

        // Check for required fields
        if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No FAQ ID provided.', 'make-faq' ) ) );
        }

        $post_id = intval( $_POST['post_id'] );
        $tags = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';
        
        // Update tags
        if ( ! empty( $tags ) ) {
            wp_set_object_terms( $post_id, explode( ',', $tags ), 'faq_tag' );
        } else {
            wp_set_object_terms( $post_id, array(), 'faq_tag' );
        }
        
        // Get updated tags for display
        $terms = get_the_terms( $post_id, 'faq_tag' );
        $tags_html = '';
        
        if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
            $tag_links = array();
            foreach ( $terms as $term ) {
                $tag_links[] = sprintf(
                    '<a href="%s">%s</a>',
                    esc_url( admin_url( 'edit.php?post_type=make_faq&faq_tag=' . $term->slug ) ),
                    esc_html( $term->name )
                );
            }
            $tags_html = '<div class="make-faq-terms-display">' . implode( ', ', $tag_links ) . '</div>';
        } else {
            $tags_html = '<div class="make-faq-terms-display">—</div>';
        }
        
        // Prepare response
        $response = array(
            'post_id' => $post_id,
            'tags_html' => $tags_html,
            'message' => __( 'Tags updated successfully.', 'make-faq' ),
        );
        
        wp_send_json_success( $response );
    }
    
    /**
     * AJAX handler for adding a new category.
     *
     * @since    1.0.5
     */
    public function ajax_add_category() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'manage_categories' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to add categories.', 'make-faq' ) ) );
        }

        // Check for category name
        if ( ! isset( $_POST['category_name'] ) || empty( $_POST['category_name'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Category name is required.', 'make-faq' ) ) );
        }

        $category_name = sanitize_text_field( $_POST['category_name'] );
        
        // Check if category already exists
        $existing_term = get_term_by( 'name', $category_name, 'faq_category' );
        if ( $existing_term ) {
            wp_send_json_success( array(
                'id' => $existing_term->term_id,
                'name' => $existing_term->name,
                'message' => __( 'Category already exists and has been selected.', 'make-faq' ),
            ) );
            return;
        }
        
        // Create the new category
        $new_term = wp_insert_term( $category_name, 'faq_category' );
        
        if ( is_wp_error( $new_term ) ) {
            wp_send_json_error( array( 'message' => $new_term->get_error_message() ) );
            return;
        }
        
        // Return the new category data
        wp_send_json_success( array(
            'id' => $new_term['term_id'],
            'name' => $category_name,
            'message' => __( 'Category added successfully.', 'make-faq' ),
        ) );
    }
    
    /**
     * AJAX handler for trashing an FAQ.
     *
     * @since    1.0.4
     */
    public function ajax_trash_faq() {
        // Check nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'make_faq_admin_nonce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Security check failed.', 'make-faq' ) ) );
        }

        // Check permissions
        if ( ! current_user_can( 'delete_posts' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to trash FAQs.', 'make-faq' ) ) );
        }

        // Check for post ID
        if ( ! isset( $_POST['post_id'] ) || empty( $_POST['post_id'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No FAQ ID provided.', 'make-faq' ) ) );
        }

        $post_id = intval( $_POST['post_id'] );
        
        // Trash the post
        $trashed = wp_trash_post( $post_id );
        
        if ( ! $trashed ) {
            wp_send_json_error( array( 'message' => __( 'Error trashing FAQ.', 'make-faq' ) ) );
        }
        
        // Prepare response
        $response = array(
            'post_id' => $post_id,
            'message' => __( 'FAQ moved to trash.', 'make-faq' ),
        );
        
        wp_send_json_success( $response );
    }
}