/**
 * Admin JavaScript for Make FAQ plugin
 */
(function($) {
    'use strict';

    // Initialize the FAQ admin functionality
    $(document).ready(function() {
        // Debug log for initialization
        if (typeof makeFaqAdmin !== 'undefined' && makeFaqAdmin.debugMode) {
            console.log('[Make FAQ Debug] Initializing admin functionality');
            console.log('[Make FAQ Debug] Categories column exists:', $('.column-categories').length > 0);
            console.log('[Make FAQ Debug] Tags column exists:', $('.column-tags').length > 0);
        }
        
        // Initialize the Quick Edit modal
        initQuickEditModal();

        // Initialize the simplified editor
        initSimplifiedEditor();
        
        // Initialize the Categories and Tags modals
        initCategoriesModal();
        initTagsModal();
        
        // Initialize the trash functionality
        initTrashFunctionality();
        
        // Add a small delay to ensure DOM is fully loaded
        setTimeout(function() {
            if (typeof makeFaqAdmin !== 'undefined' && makeFaqAdmin.debugMode) {
                console.log('[Make FAQ Debug] After delay - Categories column exists:', $('.column-categories').length > 0);
                console.log('[Make FAQ Debug] After delay - Tags column exists:', $('.column-tags').length > 0);
                console.log('[Make FAQ Debug] Category content elements:', $('.make-faq-category-content').length);
                console.log('[Make FAQ Debug] Tag content elements:', $('.make-faq-tag-content').length);
            }
        }, 1000);
    });

    /**
     * Initialize the Quick Edit modal
     */
    function initQuickEditModal() {
        // Create modal HTML if it doesn't exist
        if ($('#make-faq-modal').length === 0) {
            createModalHTML();
        }

        // Quick Edit button click handler
        $(document).on('click', '.make-faq-quick-edit', function(e) {
            e.preventDefault();
            var faqId = $(this).data('id');
            openQuickEditModal(faqId);
        });

        // Add New FAQ button click handler
        $(document).on('click', '.page-title-action', function(e) {
            // Only intercept on the FAQ list page and not on taxonomy pages
            if (window.location.href.indexOf('post_type=make_faq') > -1 && 
                window.location.href.indexOf('taxonomy=') === -1 &&
                window.location.href.indexOf('tag_ID=') === -1) {
                e.preventDefault();
                openQuickEditModal(); // No ID means new FAQ
            }
        });

        // Close modal when clicking the close button or overlay
        $(document).on('click', '.make-faq-modal-close, .make-faq-modal-overlay', function(e) {
            e.preventDefault();
            closeQuickEditModal();
        });

        // Prevent modal from closing when clicking inside it
        $(document).on('click', '.make-faq-modal', function(e) {
            e.stopPropagation();
        });

        // Save button click handler
        $(document).on('click', '#make-faq-save', function(e) {
            e.preventDefault();
            saveFAQ();
        });

        // Cancel button click handler
        $(document).on('click', '#make-faq-cancel', function(e) {
            e.preventDefault();
            closeQuickEditModal();
        });

        // Handle ESC key to close modal
        $(document).on('keydown', function(e) {
            if (e.keyCode === 27 && $('.make-faq-modal').is(':visible')) {
                closeQuickEditModal();
            }
        });
    }

    /**
     * Create the modal HTML
     */
    function createModalHTML() {
        debugLog('Creating modal HTML');
        
        // Create a basic form structure that won't hang if there's an error
        var modalHTML = `
            <div class="make-faq-modal-overlay"></div>
            <div class="make-faq-modal" id="make-faq-modal">
                <div class="make-faq-modal-header">
                    <h2>${makeFaqAdmin.strings.editFaq}</h2>
                    <span class="make-faq-modal-close dashicons dashicons-no-alt"></span>
                </div>
                <div class="make-faq-modal-content">
                    <form id="make-faq-form">
                        <div class="make-faq-loading"></div> Loading...
                    </form>
                </div>
                <div class="make-faq-modal-footer">
                    <div id="make-faq-status"></div>
                    <button type="button" id="make-faq-cancel" class="button">${makeFaqAdmin.strings.cancel}</button>
                    <button type="button" id="make-faq-save" class="button button-primary" disabled>${makeFaqAdmin.strings.saveFaq}</button>
                </div>
            </div>
        `;
        
        $('body').append(modalHTML);
        debugLog('Modal HTML created and appended to body');
    }
    
    /**
     * Create the form HTML for the modal
     */
    function createFormHTML() {
        debugLog('Creating form HTML');
        
        try {
            var formHTML = `
                <input type="hidden" id="make-faq-id" name="id" value="">
                
                <div class="make-faq-form-field">
                    <label for="make-faq-question">${wp.i18n.__('Question', 'make-faq')}</label>
                    <input type="text" id="make-faq-question" name="question" required>
                </div>
                
                <div class="make-faq-form-field">
                    <label for="make-faq-answer">${wp.i18n.__('Answer', 'make-faq')}</label>
                    <div class="make-faq-editor-toolbar">
                        <button type="button" data-command="bold" title="${wp.i18n.__('Bold', 'make-faq')}"><span class="dashicons dashicons-editor-bold"></span></button>
                        <button type="button" data-command="italic" title="${wp.i18n.__('Italic', 'make-faq')}"><span class="dashicons dashicons-editor-italic"></span></button>
                        <button type="button" data-command="insertUnorderedList" title="${wp.i18n.__('Bullet List', 'make-faq')}"><span class="dashicons dashicons-editor-ul"></span></button>
                        <button type="button" data-command="insertOrderedList" title="${wp.i18n.__('Numbered List', 'make-faq')}"><span class="dashicons dashicons-editor-ol"></span></button>
                        <button type="button" data-command="createLink" title="${wp.i18n.__('Insert Link', 'make-faq')}"><span class="dashicons dashicons-admin-links"></span></button>
                    </div>
                    <textarea id="make-faq-answer" name="answer"></textarea>
                </div>
                
                <!-- Categories and Tags fields removed from main modal -->
            `;
            
            return formHTML;
        } catch (error) {
            debugLog('Error creating form HTML:', error);
            return '<div class="make-faq-error">Error creating form. Please refresh the page and try again.</div>';
        }
    }

    /**
     * Get category options for the select dropdown
     */
    function getCategoryOptions() {
        return '<option value="">Loading categories...</option>';
    }

    /**
     * Debug log function that only logs when debug mode is enabled
     */
    function debugLog(message, data) {
        if (makeFaqAdmin.debugMode) {
            console.log('[Make FAQ Debug]', message, data || '');
        }
    }
    
    /**
     * Open the Quick Edit modal for a specific FAQ or for a new FAQ if no ID is provided
     */
    function openQuickEditModal(faqId) {
        // Show loading state
        $('.make-faq-modal-overlay, .make-faq-modal').css('display', 'flex').show();
        $('#make-faq-form').html('<div class="make-faq-loading"></div> Loading...');
        
        debugLog('Opening modal for FAQ ID:', faqId || 'New FAQ');
        
        // Set the modal title based on whether we're editing or creating
        if (faqId) {
            $('.make-faq-modal-header h2').text(makeFaqAdmin.strings.editFaq);
        } else {
            $('.make-faq-modal-header h2').text(makeFaqAdmin.strings.addNewFaq || 'Add New FAQ');
        }
        
        // Set a timeout to prevent the modal from hanging indefinitely
        var modalTimeout = setTimeout(function() {
            debugLog('Modal loading timeout reached');
            $('#make-faq-form').html('<div class="make-faq-error">Loading timed out. Please try again.</div>');
            showStatus('error', 'Loading timed out. Please try again.');
        }, 10000); // 10 seconds timeout
        
        // Prepare request data
        var requestData = {
            action: 'make_faq_get_faq',
            nonce: makeFaqAdmin.nonce,
            id: faqId || ''
        };
        
        debugLog('Sending AJAX request to get FAQ data:', requestData);
        
        // Fetch FAQ data
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: requestData,
            success: function(response) {
                // Clear the timeout since we got a response
                clearTimeout(modalTimeout);
                
                debugLog('Received AJAX response:', response);
                
                if (response.success) {
                    populateForm(response.data);
                } else {
                    showStatus('error', response.data.message);
                    setTimeout(closeQuickEditModal, 2000);
                }
            },
            error: function(xhr, status, error) {
                // Clear the timeout since we got a response
                clearTimeout(modalTimeout);
                
                debugLog('AJAX error:', { xhr: xhr, status: status, error: error });
                $('#make-faq-form').html('<div class="make-faq-error">Error loading FAQ: ' + error + '</div>');
                showStatus('error', wp.i18n.__('Error loading FAQ. Please try again.', 'make-faq'));
                setTimeout(closeQuickEditModal, 2000);
            }
        });
    }

    /**
     * Populate the form with FAQ data
     */
    function populateForm(data) {
        debugLog('Populating form with data:', data);
        
        try {
            // First, replace the loading indicator with the actual form HTML
            var formHTML = createFormHTML();
            $('#make-faq-form').html(formHTML);
            
            // Then set form values
            $('#make-faq-id').val(data.id);
            $('#make-faq-question').val(data.question);
            $('#make-faq-answer').val(data.answer);
            
            debugLog('Form fields populated');
            
            // Enable the save button
            $('#make-faq-save').prop('disabled', false);
            
            debugLog('Form displayed successfully');
        } catch (error) {
            debugLog('Error populating form:', error);
            $('#make-faq-form').html('<div class="make-faq-error">Error loading form: ' + error.message + '</div>');
            showStatus('error', 'Error loading form: ' + error.message);
        }
    }

    // loadCategories function removed as it's no longer needed for the main modal

    /**
     * Close the Quick Edit modal
     */
    function closeQuickEditModal() {
        // Check for unsaved changes
        var hasChanges = false;
        
        // Hide modal
        $('.make-faq-modal-overlay, .make-faq-modal').hide();
        
        // Clear form
        $('#make-faq-form')[0].reset();
        $('#make-faq-status').html('').removeClass('make-faq-status make-faq-status-success make-faq-status-error');
    }

    /**
     * Save the FAQ
     */
    function saveFAQ() {
        debugLog('Saving FAQ');
        
        try {
            // Validate form
            if (!$('#make-faq-question').val().trim()) {
                showStatus('error', 'Question field is required.');
                return;
            }
            
            // Show saving status
            showStatus('info', `<div class="make-faq-loading"></div> ${makeFaqAdmin.strings.saving}`);
            
            // Disable save button
            $('#make-faq-save').prop('disabled', true);
            
            // Get form data
            var formData = {
                action: 'make_faq_save_faq',
                nonce: makeFaqAdmin.nonce,
                id: $('#make-faq-id').val(),
                question: $('#make-faq-question').val(),
                answer: $('#make-faq-answer').val()
                // Categories and Tags removed from main form
            };
            
            debugLog('Form data to save:', formData);
            
            // Check if this is a new FAQ or an existing one
            var isNew = !formData.id;
            
            // Set a timeout to prevent hanging on save
            var saveTimeout = setTimeout(function() {
                debugLog('Save timeout reached');
                showStatus('error', 'Save operation timed out. Please try again.');
                $('#make-faq-save').prop('disabled', false);
            }, 10000); // 10 seconds timeout
            
            // Send AJAX request
            $.ajax({
                url: makeFaqAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    // Clear the timeout
                    clearTimeout(saveTimeout);
                    
                    debugLog('Save response received:', response);
                    
                    if (response.success) {
                        // Show success message
                        if (isNew) {
                            showStatus('success', makeFaqAdmin.strings.created || 'FAQ created successfully!');
                            
                            // For new FAQs, we need to refresh the page to see the new entry
                            setTimeout(function() {
                                window.location.reload();
                            }, 1000);
                        } else {
                            showStatus('success', makeFaqAdmin.strings.saved);
                            
                            // Update the row in the table
                            updateTableRow(response.data);
                            
                            // Close modal after a delay
                            setTimeout(closeQuickEditModal, 1000);
                        }
                    } else {
                        // Show error message
                        showStatus('error', response.data.message);
                        
                        // Enable save button
                        $('#make-faq-save').prop('disabled', false);
                    }
                },
                error: function(xhr, status, error) {
                    // Clear the timeout
                    clearTimeout(saveTimeout);
                    
                    debugLog('Save error:', { xhr: xhr, status: status, error: error });
                    
                    // Show error message
                    showStatus('error', makeFaqAdmin.strings.error + (error ? ': ' + error : ''));
                    
                    // Enable save button
                    $('#make-faq-save').prop('disabled', false);
                }
            });
        } catch (error) {
            debugLog('Error in saveFAQ function:', error);
            showStatus('error', 'Error preparing to save: ' + error.message);
            $('#make-faq-save').prop('disabled', false);
        }
    }

    /**
     * Update the table row with new FAQ data
     */
    function updateTableRow(data) {
        var row = $('tr#post-' + data.id);
        
        // Update question
        row.find('.column-title .row-title').text(data.question);
        
        // Update answer excerpt
        row.find('.column-answer').text(data.answer);
        
        // Categories and tags would need to be updated as well
        // This would require additional data from the server
    }

    /**
     * Show a status message
     */
    function showStatus(type, message) {
        var statusClass = 'make-faq-status make-faq-status-' + type;
        $('#make-faq-status').html(message).removeClass().addClass(statusClass);
    }

    /**
     * Initialize the simplified editor
     */
    function initSimplifiedEditor() {
        // Editor toolbar button click handler
        $(document).on('click', '.make-faq-editor-toolbar button', function(e) {
            e.preventDefault();
            
            var command = $(this).data('command');
            
            if (command === 'createLink') {
                var url = prompt(wp.i18n.__('Enter URL:', 'make-faq'));
                if (url) {
                    document.execCommand(command, false, url);
                }
            } else {
                document.execCommand(command, false, null);
            }
            
            // Focus back on the editor
            $('#make-faq-answer').focus();
        });
    }
    
    /**
     * Initialize the Categories modal
     */
    function initCategoriesModal() {
        // Edit Categories button click handler
        $(document).on('click', '.make-faq-edit-categories', function(e) {
            e.preventDefault();
            var faqId = $(this).data('id');
            openCategoriesModal(faqId);
        });
        
        // Close modal when clicking the close button or overlay
        $(document).on('click', '#make-faq-categories-modal .make-faq-modal-close, #make-faq-categories-modal-overlay', function(e) {
            e.preventDefault();
            closeCategoriesModal();
        });
        
        // Prevent modal from closing when clicking inside it
        $(document).on('click', '#make-faq-categories-modal', function(e) {
            e.stopPropagation();
        });
        
        // Save button click handler
        $(document).on('click', '#make-faq-categories-save', function(e) {
            e.preventDefault();
            saveCategories();
        });
        
        // Cancel button click handler
        $(document).on('click', '#make-faq-categories-cancel', function(e) {
            e.preventDefault();
            closeCategoriesModal();
        });
        
        // Add new category button click handler
        $(document).on('click', '#make-faq-add-category', function(e) {
            e.preventDefault();
            addNewCategory();
        });
        
        // Allow pressing Enter in the new category input to add the category
        $(document).on('keypress', '#make-faq-new-category', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                addNewCategory();
            }
        });
    }
    
    /**
     * Open the Categories modal for a specific FAQ
     */
    function openCategoriesModal(faqId) {
        debugLog('Opening categories modal for FAQ ID:', faqId);
        
        // Show loading state
        $('#make-faq-categories-modal-overlay, #make-faq-categories-modal').css('display', 'flex').show();
        
        // Set the post ID
        $('#make-faq-categories-post-id').val(faqId);
        
        // Reset Select2 if it exists
        if ($.fn.select2) {
            try {
                const $select = $('#make-faq-categories-select');
                if ($select.hasClass('select2-hidden-accessible')) {
                    $select.select2('destroy');
                }
                $select.empty();
            } catch (error) {
                debugLog('Error resetting Select2:', error);
            }
        }
        
        // Fetch categories data
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_get_categories',
                nonce: makeFaqAdmin.nonce,
                post_id: faqId
            },
            success: function(response) {
                debugLog('Categories data received:', response);
                
                if (response.success) {
                    populateCategoriesForm(response.data);
                } else {
                    showCategoriesStatus('error', response.data.message);
                    setTimeout(closeCategoriesModal, 2000);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Error fetching categories:', error);
                showCategoriesStatus('error', 'Error loading categories. Please try again.');
                setTimeout(closeCategoriesModal, 2000);
            }
        });
    }
    
    /**
     * Populate the categories form with data
     */
    function populateCategoriesForm(data) {
        debugLog('Populating categories form with data:', data);
        
        // Clear any existing options
        const $select = $('#make-faq-categories-select');
        $select.empty();
        
        // Add options for each category
        if (data.all_categories && data.all_categories.length > 0) {
            data.all_categories.forEach(function(category) {
                const isSelected = data.selected_categories.includes(category.id);
                const option = new Option(category.name, category.id, isSelected, isSelected);
                $select.append(option);
            });
        }
        
        // Initialize Select2
        try {
            $select.select2({
                placeholder: 'Select categories',
                allowClear: true,
                width: '100%',
                tags: true,
                createTag: function(params) {
                    // Trim the term
                    var term = $.trim(params.term);
                    
                    // Don't allow empty terms
                    if (term === '') {
                        return null;
                    }
                    
                    // Don't allow terms with special characters that might cause issues
                    if (/[<>&#%=?]/.test(term)) {
                        // Show a warning in the status area
                        showCategoriesStatus('error', 'Category names cannot contain special characters: < > & # % = ?');
                        return null;
                    }
                    
                    // Return the new tag with a prefix to identify it
                    return {
                        id: 'new:' + term,
                        text: term + ' (new)',
                        newTag: true
                    };
                },
                theme: 'default'
            });
            
            debugLog('Select2 initialized successfully');
        } catch (error) {
            console.error('Error initializing Select2:', error);
            showCategoriesStatus('error', 'Error initializing categories selector. Please try again.');
        }
    }
    
    /**
     * Save the categories
     */
    function saveCategories() {
        debugLog('Saving categories');
        
        // Show saving status
        showCategoriesStatus('info', `<div class="make-faq-loading"></div> ${makeFaqAdmin.strings.saving}`);
        
        // Disable save button
        $('#make-faq-categories-save').prop('disabled', true);
        
        // Get selected categories from Select2
        var selectedCategories = $('#make-faq-categories-select').val() || [];
        debugLog('Selected categories:', selectedCategories);
        
        // Process new categories first
        var newCategories = [];
        var existingCategories = [];
        
        selectedCategories.forEach(function(categoryId) {
            if (typeof categoryId === 'string' && categoryId.startsWith('new:')) {
                // This is a new category
                newCategories.push(categoryId.substring(4)); // Remove 'new:' prefix
            } else {
                // This is an existing category
                existingCategories.push(categoryId);
            }
        });
        
        // If we have new categories, create them first
        if (newCategories.length > 0) {
            processNewCategories(newCategories, existingCategories);
            return;
        }
        
        // If no new categories, proceed with saving existing ones
        var formData = {
            action: 'make_faq_save_categories',
            nonce: makeFaqAdmin.nonce,
            post_id: $('#make-faq-categories-post-id').val(),
            categories: existingCategories
        };
        
        debugLog('Categories form data:', formData);
        
        // Send the save request
        sendSaveCategoriesRequest(formData);
    }
    
    /**
     * Update the categories in the table
     */
    function updateCategoriesInTable(data) {
        var postId = data.post_id;
        var categoriesHtml = data.categories_html;
        
        // Find the row and update the categories content
        $('tr#post-' + postId + ' .column-faq_categories .make-faq-category-content').html(categoriesHtml);
        
        // Log for debugging
        debugLog('Updated categories in table for post ID:', postId);
    }
    
    /**
     * Show a status message in the categories modal
     */
    function showCategoriesStatus(type, message) {
        var statusClass = 'make-faq-status make-faq-status-' + type;
        $('#make-faq-categories-status').html(message).removeClass().addClass(statusClass);
    }
    
    /**
     * Process new categories created with Select2
     * 
     * @param {Array} newCategories - Array of new category names
     * @param {Array} existingCategories - Array of existing category IDs
     */
    function processNewCategories(newCategories, existingCategories) {
        debugLog('Processing new categories:', newCategories);
        
        // If there are no new categories, just save the existing ones
        if (newCategories.length === 0) {
            var formData = {
                action: 'make_faq_save_categories',
                nonce: makeFaqAdmin.nonce,
                post_id: $('#make-faq-categories-post-id').val(),
                categories: existingCategories
            };
            
            sendSaveCategoriesRequest(formData);
            return;
        }
        
        // Process the first new category
        var categoryName = newCategories.shift();
        
        // Send AJAX request to add new category
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_add_category',
                nonce: makeFaqAdmin.nonce,
                category_name: categoryName
            },
            success: function(response) {
                if (response.success) {
                    // Add the new category ID to the existing categories
                    existingCategories.push(response.data.id);
                    
                    // Process the next new category or save if done
                    if (newCategories.length > 0) {
                        processNewCategories(newCategories, existingCategories);
                    } else {
                        // All new categories processed, now save with the updated list
                        var formData = {
                            action: 'make_faq_save_categories',
                            nonce: makeFaqAdmin.nonce,
                            post_id: $('#make-faq-categories-post-id').val(),
                            categories: existingCategories
                        };
                        
                        sendSaveCategoriesRequest(formData);
                    }
                } else {
                    // Show error message
                    showCategoriesStatus('error', response.data.message || 'Error adding category');
                    $('#make-faq-categories-save').prop('disabled', false);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Add category error:', error);
                showCategoriesStatus('error', 'Error adding category. Please try again.');
                $('#make-faq-categories-save').prop('disabled', false);
            }
        });
    }
    
    /**
     * Send the save categories AJAX request
     * 
     * @param {Object} formData - The form data to send
     */
    function sendSaveCategoriesRequest(formData) {
        debugLog('Sending save categories request:', formData);
        
        // Send AJAX request
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: formData,
            success: function(response) {
                debugLog('Categories save response:', response);
                
                if (response.success) {
                    // Show success message
                    showCategoriesStatus('success', makeFaqAdmin.strings.categoriesSaved);
                    
                    // Update the categories in the table
                    updateCategoriesInTable(response.data);
                    
                    // Close modal after a delay
                    setTimeout(closeCategoriesModal, 1000);
                } else {
                    // Show error message
                    showCategoriesStatus('error', response.data.message);
                    
                    // Enable save button
                    $('#make-faq-categories-save').prop('disabled', false);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Categories save error:', error);
                
                // Show error message
                showCategoriesStatus('error', 'Error saving categories. Please try again.');
                
                // Enable save button
                $('#make-faq-categories-save').prop('disabled', false);
            }
        });
    }
    
    /**
     * Add a new category
     */
    function addNewCategory() {
        var categoryName = $('#make-faq-new-category').val().trim();
        
        if (!categoryName) {
            showCategoriesStatus('error', 'Please enter a category name.');
            return;
        }
        
        // Show loading status
        showCategoriesStatus('info', `<div class="make-faq-loading"></div> Adding category...`);
        
        // Disable add button
        $('#make-faq-add-category').prop('disabled', true);
        
        // Send AJAX request to add new category
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_add_category',
                nonce: makeFaqAdmin.nonce,
                category_name: categoryName
            },
            success: function(response) {
                debugLog('Add category response:', response);
                
                if (response.success) {
                    // Show success message
                    showCategoriesStatus('success', 'Category added successfully!');
                    
                    // Add the new category to the select dropdown and select it
                    var newOption = new Option(response.data.name, response.data.id, true, true);
                    $('#make-faq-categories-select').append(newOption).trigger('change');
                    
                    // Clear the input field
                    $('#make-faq-new-category').val('');
                } else {
                    // Show error message
                    showCategoriesStatus('error', response.data.message || 'Error adding category.');
                }
                
                // Enable add button
                $('#make-faq-add-category').prop('disabled', false);
            },
            error: function(xhr, status, error) {
                debugLog('Add category error:', error);
                
                // Show error message
                showCategoriesStatus('error', 'Error adding category. Please try again.');
                
                // Enable add button
                $('#make-faq-add-category').prop('disabled', false);
            }
        });
    }
    
    /**
     * Close the Categories modal
     */
    function closeCategoriesModal() {
        // Hide modal
        $('#make-faq-categories-modal-overlay, #make-faq-categories-modal').hide();
        
        // Clear form
        $('#make-faq-categories-form')[0].reset();
        $('#make-faq-categories-status').html('').removeClass('make-faq-status make-faq-status-success make-faq-status-error');
        
        // Enable save button
        $('#make-faq-categories-save').prop('disabled', false);
    }
    
    /**
     * Initialize the Tags modal
     */
    function initTagsModal() {
        // Edit Tags button click handler
        $(document).on('click', '.make-faq-edit-tags', function(e) {
            e.preventDefault();
            var faqId = $(this).data('id');
            openTagsModal(faqId);
        });
        
        // Close modal when clicking the close button or overlay
        $(document).on('click', '#make-faq-tags-modal .make-faq-modal-close, #make-faq-tags-modal-overlay', function(e) {
            e.preventDefault();
            closeTagsModal();
        });
        
        // Prevent modal from closing when clicking inside it
        $(document).on('click', '#make-faq-tags-modal', function(e) {
            e.stopPropagation();
        });
        
        // Save button click handler
        $(document).on('click', '#make-faq-tags-save', function(e) {
            e.preventDefault();
            saveTags();
        });
        
        // Cancel button click handler
        $(document).on('click', '#make-faq-tags-cancel', function(e) {
            e.preventDefault();
            closeTagsModal();
        });
        
        // Add new tag button click handler
        $(document).on('click', '#make-faq-add-tag', function(e) {
            e.preventDefault();
            addNewTag();
        });
        
        // Allow pressing Enter in the new tag input to add the tag
        $(document).on('keypress', '#make-faq-new-tag', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                addNewTag();
            }
        });
    }
    
    /**
     * Open the Tags modal for a specific FAQ
     */
    function openTagsModal(faqId) {
        debugLog('Opening tags modal for FAQ ID:', faqId);
        
        // Show loading state
        $('#make-faq-tags-modal-overlay, #make-faq-tags-modal').css('display', 'flex').show();
        
        // Set the post ID
        $('#make-faq-tags-post-id').val(faqId);
        
        // Reset Select2 if it exists
        if ($.fn.select2) {
            try {
                const $select = $('#make-faq-tags-select');
                if ($select.hasClass('select2-hidden-accessible')) {
                    $select.select2('destroy');
                }
                $select.empty();
            } catch (error) {
                debugLog('Error resetting Select2:', error);
            }
        }
        
        // Fetch tags data
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_get_tags',
                nonce: makeFaqAdmin.nonce,
                post_id: faqId
            },
            success: function(response) {
                debugLog('Tags data received:', response);
                
                if (response.success) {
                    populateTagsForm(response.data);
                } else {
                    showTagsStatus('error', response.data.message);
                    setTimeout(closeTagsModal, 2000);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Error fetching tags:', error);
                showTagsStatus('error', 'Error loading tags. Please try again.');
                setTimeout(closeTagsModal, 2000);
            }
        });
    }
    
    /**
     * Populate the tags form with data
     */
    function populateTagsForm(data) {
        debugLog('Populating tags form with data:', data);
        
        // Clear any existing options
        const $select = $('#make-faq-tags-select');
        $select.empty();
        
        // Parse tags string into array
        const tags = data.tags ? data.tags.split(',').map(tag => tag.trim()).filter(tag => tag) : [];
        
        // Get all available tags
        const allTags = getAllAvailableTags();
        
        // Add options for each tag
        if (allTags && allTags.length > 0) {
            allTags.forEach(function(tag) {
                const isSelected = tags.includes(tag.name);
                const option = new Option(tag.name, tag.name, isSelected, isSelected);
                $select.append(option);
            });
        }
        
        // Add selected tags that might not be in allTags
        tags.forEach(function(tag) {
            if (!allTags.some(t => t.name === tag)) {
                const option = new Option(tag, tag, true, true);
                $select.append(option);
            }
        });
        
        // Initialize Select2
        try {
            $select.select2({
                placeholder: 'Select tags',
                allowClear: true,
                width: '100%',
                tags: true,
                createTag: function(params) {
                    // Trim the term
                    var term = $.trim(params.term);
                    
                    // Don't allow empty terms
                    if (term === '') {
                        return null;
                    }
                    
                    // Don't allow terms with special characters that might cause issues
                    if (/[<>&#%=?]/.test(term)) {
                        // Show a warning in the status area
                        showTagsStatus('error', 'Tag names cannot contain special characters: < > & # % = ?');
                        return null;
                    }
                    
                    // Return the new tag
                    return {
                        id: term,
                        text: term + ' (new)',
                        newTag: true
                    };
                },
                theme: 'default'
            });
            
            debugLog('Select2 initialized successfully for tags');
        } catch (error) {
            console.error('Error initializing Select2 for tags:', error);
            showTagsStatus('error', 'Error initializing tags selector. Please try again.');
        }
    }
    
    /**
     * Get all available tags
     * 
     * @return {Array} Array of tag objects
     */
    function getAllAvailableTags() {
        // Extract tags from existing elements on the page
        const tagElements = [];
        
        // Look for tags in the tag column cells
        $('.make-faq-tag-content .make-faq-terms-display a').each(function() {
            const tagName = $(this).text().trim();
            if (tagName && tagName !== '—' && !tagElements.some(t => t.name === tagName)) {
                tagElements.push({ id: tagName, name: tagName });
            }
        });
        
        debugLog('Found tags:', tagElements);
        return tagElements;
    }
    
    /**
     * Save the tags
     */
    function saveTags() {
        debugLog('Saving tags');
        
        // Show saving status
        showTagsStatus('info', `<div class="make-faq-loading"></div> ${makeFaqAdmin.strings.saving}`);
        
        // Disable save button
        $('#make-faq-tags-save').prop('disabled', true);
        
        // Get selected tags from Select2
        var selectedTags = $('#make-faq-tags-select').val() || [];
        debugLog('Selected tags:', selectedTags);
        
        // Join tags into a comma-separated string
        var tagsString = selectedTags.join(', ');
        
        // Get form data
        var formData = {
            action: 'make_faq_save_tags',
            nonce: makeFaqAdmin.nonce,
            post_id: $('#make-faq-tags-post-id').val(),
            tags: tagsString
        };
        
        debugLog('Tags form data:', formData);
        
        // Send AJAX request
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: formData,
            success: function(response) {
                debugLog('Tags save response:', response);
                
                if (response.success) {
                    // Show success message
                    showTagsStatus('success', makeFaqAdmin.strings.tagsSaved);
                    
                    // Update the tags in the table
                    updateTagsInTable(response.data);
                    
                    // Close modal after a delay
                    setTimeout(closeTagsModal, 1000);
                } else {
                    // Show error message
                    showTagsStatus('error', response.data.message);
                    
                    // Enable save button
                    $('#make-faq-tags-save').prop('disabled', false);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Tags save error:', error);
                
                // Show error message
                showTagsStatus('error', 'Error saving tags. Please try again.');
                
                // Enable save button
                $('#make-faq-tags-save').prop('disabled', false);
            }
        });
    }
    
    /**
     * Update the tags in the table
     */
    function updateTagsInTable(data) {
        var postId = data.post_id;
        var tagsHtml = data.tags_html;
        
        // Find the row and update the tags content
        $('tr#post-' + postId + ' .column-faq_tags .make-faq-tag-content').html(tagsHtml);
        
        // Log for debugging
        debugLog('Updated tags in table for post ID:', postId);
    }
    
    /**
     * Show a status message in the tags modal
     */
    function showTagsStatus(type, message) {
        var statusClass = 'make-faq-status make-faq-status-' + type;
        $('#make-faq-tags-status').html(message).removeClass().addClass(statusClass);
    }
    
    /**
     * Add a new tag
     */
    function addNewTag() {
        var tagName = $('#make-faq-new-tag').val().trim();
        
        if (!tagName) {
            showTagsStatus('error', 'Please enter a tag name.');
            return;
        }
        
        // Get current tags
        var currentTags = $('#make-faq-tags-input').val();
        var tagArray = currentTags ? currentTags.split(',').map(function(tag) {
            return tag.trim();
        }) : [];
        
        // Check if tag already exists
        if (tagArray.includes(tagName)) {
            showTagsStatus('info', 'Tag already exists in the list.');
            $('#make-faq-new-tag').val('');
            return;
        }
        
        // Add the new tag to the list
        tagArray.push(tagName);
        
        // Update the tags input
        $('#make-faq-tags-input').val(tagArray.join(', '));
        
        // Clear the new tag input
        $('#make-faq-new-tag').val('');
        
        // Show success message
        showTagsStatus('success', 'Tag added to the list.');
    }
    
    /**
     * Close the Tags modal
     */
    function closeTagsModal() {
        // Hide modal
        $('#make-faq-tags-modal-overlay, #make-faq-tags-modal').hide();
        
        // Clear form
        $('#make-faq-tags-form')[0].reset();
        $('#make-faq-tags-status').html('').removeClass('make-faq-status make-faq-status-success make-faq-status-error');
        
        // Enable save button
        $('#make-faq-tags-save').prop('disabled', false);
    }
    
    /**
     * Initialize the trash functionality
     */
    function initTrashFunctionality() {
        // Trash button click handler
        $(document).on('click', '.make-faq-trash', function(e) {
            e.preventDefault();
            var faqId = $(this).data('id');
            
            // Confirm before trashing
            if (confirm(makeFaqAdmin.strings.confirmDelete)) {
                trashFaq(faqId);
            }
        });
    }
    
    /**
     * Move an FAQ to trash
     */
    function trashFaq(faqId) {
        debugLog('Trashing FAQ ID:', faqId);
        
        // Send AJAX request
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_trash',
                nonce: makeFaqAdmin.nonce,
                post_id: faqId
            },
            success: function(response) {
                debugLog('Trash response:', response);
                
                if (response.success) {
                    // Show success message
                    alert(makeFaqAdmin.strings.deleted);
                    
                    // Remove the row from the table
                    $('tr#post-' + faqId).fadeOut(400, function() {
                        $(this).remove();
                    });
                } else {
                    // Show error message
                    alert(response.data.message);
                }
            },
            error: function(xhr, status, error) {
                debugLog('Trash error:', error);
                
                // Show error message
                alert('Error moving FAQ to trash. Please try again.');
            }
        });
    }
    
    // Expose modal functions globally for use in make-faq-multiselect.js
    window.makeFaqAdmin_openCategoriesModal = openCategoriesModal;
    window.makeFaqAdmin_openTagsModal = openTagsModal;

})(jQuery);