/**
 * Multi-select component for Make FAQ plugin
 * 
 * This file provides the multi-select functionality for categories and tags
 * both inline and within modals.
 */
(function($) {
    'use strict';

    // Make the openCategoriesModal and openTagsModal functions available globally
    window.openCategoriesModal = function(faqId) {
        if (typeof window.makeFaqAdmin_openCategoriesModal === 'function') {
            window.makeFaqAdmin_openCategoriesModal(faqId);
        } else {
            console.error('openCategoriesModal function not available');
        }
    };
    
    window.openTagsModal = function(faqId) {
        if (typeof window.makeFaqAdmin_openTagsModal === 'function') {
            window.makeFaqAdmin_openTagsModal(faqId);
        } else {
            console.error('openTagsModal function not available');
        }
    };

    // Initialize the multi-select components
    $(document).ready(function() {
        if (typeof makeFaqAdmin !== 'undefined' && makeFaqAdmin.debugMode) {
            console.log('[Make FAQ Debug] Initializing multi-select components');
        }
        
        // Initialize for categories
        initMultiSelect('categories');
        
        // Initialize for tags
        initMultiSelect('tags');
        
        // Add the multiselect to the modals
        initModalMultiSelect();
    });

    /**
     * Initialize multi-select for a specific taxonomy
     * 
     * @param {string} type - Either 'categories' or 'tags'
     */
    function initMultiSelect(type) {
        const taxonomyType = type === 'categories' ? 'category' : 'tag';
        const taxonomyName = type === 'categories' ? 'faq_category' : 'faq_tag';
        const columnClass = type === 'categories' ? '.column-faq_categories' : '.column-faq_tags';
    
        // Initialize the editing interface
        $(document).ready(function() {
            // Find all content containers and initialize them
            $(`.make-faq-${taxonomyType}-content`).each(function() {
                const $container = $(this);
                const $editButton = $container.next(`.make-faq-edit-${taxonomyType}s`);
                
                // Store the post ID as a data attribute on the container for easier access
                const postId = $editButton.data('id');
                $container.attr('data-post-id', postId);
            
                // Make the terms display clickable to open the modal
                $container.find('.make-faq-terms-display').on('click', function() {
                    // Get the post ID from the container's data attribute
                    const containerId = $container.attr('data-post-id');
                    
                    if (containerId) {
                        // Open the modal instead of creating an inline editor
                        if (type === 'categories') {
                            openCategoriesModal(containerId);
                        } else {
                            openTagsModal(containerId);
                        }
                    } else {
                        console.error('No FAQ ID found for', $container);
                        $container.html('<div class="make-faq-error">Error: No FAQ ID found</div>');
                    }
                });
            
                // Add a helpful tooltip
                $container.find('.make-faq-terms-display').attr('title', `Click to edit ${type}`);
            });
        });
    
        // Redirect button clicks to the modal
        $(document).on('click', `.make-faq-edit-${taxonomyType}s`, function(e) {
            e.preventDefault();
            const postId = $(this).data('id');
            
            // Open the modal instead of creating an inline editor
            if (type === 'categories') {
                openCategoriesModal(postId);
            } else {
                openTagsModal(postId);
            }
        });
    }
    
    /**
     * Initialize multiselect inside the modals
     */
    function initModalMultiSelect() {
        // Add multiselect to categories modal
        $(document).on('click', '#make-faq-categories-save', function(e) {
            e.preventDefault();
            saveCategories();
        });
        
        // Add multiselect to tags modal
        $(document).on('click', '#make-faq-tags-save', function(e) {
            e.preventDefault();
            saveTags();
        });
    }

    /**
     * Create a multi-select component
     * 
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     */
    function createMultiSelect(postId, type, container) {
        const taxonomyType = type === 'categories' ? 'category' : 'tag';
        const getAction = type === 'categories' ? 'make_faq_get_categories' : 'make_faq_get_tags';
        const saveAction = type === 'categories' ? 'make_faq_save_categories' : 'make_faq_save_tags';
        
        // Show loading indicator
        container.html('<div class="make-faq-loading"></div> Loading...');
        
        // Fetch data
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: getAction,
                nonce: makeFaqAdmin.nonce,
                post_id: postId
            },
            success: function(response) {
                if (response.success) {
                    renderMultiSelect(postId, type, container, response.data);
                } else {
                    container.html(`<div class="make-faq-error">${response.data.message || 'Error loading data'}</div>`);
                }
            },
            error: function(xhr, status, error) {
                container.html(`<div class="make-faq-error">Error: ${error}</div>`);
            }
        });
    }

    /**
     * Render the multi-select component
     * 
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     * @param {object} data - The data from the server
     */
    function renderMultiSelect(postId, type, container, data) {
        const taxonomyType = type === 'categories' ? 'category' : 'tag';
        const isCategories = type === 'categories';
        
        // Create the HTML structure with beginner-friendly instructions
        let html = `
            <div class="make-faq-terms-display"></div>
            <div class="make-faq-multiselect">
                <div class="make-faq-multiselect-header">
                    <h4 class="make-faq-multiselect-title">Select ${type} for this FAQ</h4>
                    <p class="make-faq-multiselect-instructions">Click on items to select/deselect. Changes are saved automatically.</p>
                    <input type="text" class="make-faq-multiselect-search" placeholder="Type to filter ${type}...">
                </div>
                <div class="make-faq-multiselect-body">
                    <div class="make-faq-multiselect-options-wrapper">
                        <ul class="make-faq-multiselect-options">
        `;
        
        // Add options with simplified UI
        if (isCategories && data.all_categories && data.all_categories.length > 0) {
            data.all_categories.forEach(function(category) {
                const isSelected = data.selected_categories.includes(category.id);
                html += `
                    <li class="make-faq-multiselect-option ${isSelected ? 'selected' : ''}">
                        <label class="make-faq-pill">
                            <input type="checkbox" value="${category.id}" ${isSelected ? 'checked' : ''}>
                            <span>${category.name}</span>
                            <span class="make-faq-checkmark">${isSelected ? '✓' : ''}</span>
                        </label>
                    </li>
                `;
            });
        } else if (!isCategories) {
            // For tags, handle both when data.tags exists and when it's empty
            const tags = data.tags ? data.tags.split(',').map(tag => tag.trim()).filter(tag => tag) : [];
            const allTags = getAllTags();
            
            // Log for debugging
            console.log('Rendering tags:', { dataTags: data.tags, parsedTags: tags, allTags: allTags });
            
            // Always render all available tags
            allTags.forEach(function(tag) {
                const isSelected = tags.includes(tag.name);
                
                // Skip the placeholder tag if we have actual tags selected
                if (tag.id === 'add-new-tag' && tags.length > 0) {
                    return;
                }
                
                html += `
                    <li class="make-faq-multiselect-option ${isSelected ? 'selected' : ''}">
                        <label class="make-faq-pill">
                            <input type="checkbox" value="${tag.name}" ${isSelected ? 'checked' : ''}>
                            <span>${tag.name}</span>
                            <span class="make-faq-checkmark">${isSelected ? '✓' : ''}</span>
                        </label>
                    </li>
                `;
            });
        }
        
        // Add "Add new" option with improved UI
        html += `
                        </ul>
                    </div>
                </div>
                <div class="make-faq-multiselect-footer">
                    <div class="make-faq-add-new-wrapper">
                        <input type="text" class="make-faq-new-${taxonomyType}" placeholder="Type new ${taxonomyType} name here...">
                        <button type="button" class="button make-faq-add-${taxonomyType}">
                            <span class="dashicons dashicons-plus-alt2"></span> Add
                        </button>
                    </div>
                    <div class="make-faq-multiselect-actions">
                        <button type="button" class="button make-faq-multiselect-done">Done</button>
                    </div>
                </div>
            </div>
        `;
        
        // Add the HTML to the container
        container.html(html);
        
        // Update the terms display
        updateTermsDisplay(container, type, isCategories ? data.all_categories : null, 
            isCategories ? data.selected_categories : data.tags);
        
        // Add event listeners
        addMultiSelectEventListeners(postId, type, container);
    }

    /**
     * Add event listeners to the multi-select component
     * 
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     */
    function addMultiSelectEventListeners(postId, type, container) {
        const taxonomyType = type === 'categories' ? 'category' : 'tag';
        const isCategories = type === 'categories';
        
        // Pill click event (more beginner-friendly than checkboxes)
        container.find('.make-faq-multiselect-option label').on('click', function(e) {
            // Prevent default to handle it ourselves
            e.preventDefault();
            
            const $option = $(this).closest('.make-faq-multiselect-option');
            const $checkbox = $option.find('input[type="checkbox"]');
            const newCheckedState = !$checkbox.prop('checked');
            
            // Update checkbox and selected state
            $checkbox.prop('checked', newCheckedState);
            $option.toggleClass('selected', newCheckedState);
            
            // Update checkmark
            $option.find('.make-faq-checkmark').text(newCheckedState ? '✓' : '');
            
            // Save changes
            saveChanges(postId, type, container);
        });
        
        // Search input event
        container.find('.make-faq-multiselect-search').on('input', function() {
            const searchTerm = $(this).val().toLowerCase();
            
            container.find('.make-faq-multiselect-option').each(function() {
                const optionText = $(this).find('span').first().text().toLowerCase();
                $(this).toggle(optionText.includes(searchTerm));
            });
        });
        
        // Add new button click
        container.find(`.make-faq-add-${taxonomyType}`).on('click', function() {
            addNewTerm(postId, type, container);
        });
        
        // Add new input enter key
        container.find(`.make-faq-new-${taxonomyType}`).on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                addNewTerm(postId, type, container);
            }
        });
        
        // Done button click
        container.find('.make-faq-multiselect-done').on('click', function() {
            // Hide the multiselect and show the terms display
            container.find('.make-faq-multiselect').hide();
            container.find('.make-faq-terms-display').show();
            
            // Remove the multiselect component completely to allow reopening
            setTimeout(function() {
                container.find('.make-faq-multiselect').remove();
            }, 100);
            
            // Show a success message
            const successMsg = $('<div class="make-faq-status make-faq-status-success">Changes saved!</div>');
            container.append(successMsg);
            setTimeout(function() {
                successMsg.fadeOut(function() {
                    $(this).remove();
                });
            }, 1500);
        });
    }

    /**
     * Add a new term
     * 
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     */
    function addNewTerm(postId, type, container) {
        const taxonomyType = type === 'categories' ? 'category' : 'tag';
        const isCategories = type === 'categories';
        const newTermInput = container.find(`.make-faq-new-${taxonomyType}`);
        const newTermName = newTermInput.val().trim();
        
        if (!newTermName) {
            // Show a helpful message if the input is empty
            const errorMsg = $('<div class="make-faq-status make-faq-status-error">Please enter a name first</div>');
            container.find('.make-faq-add-new-wrapper').after(errorMsg);
            setTimeout(function() {
                errorMsg.fadeOut(function() {
                    $(this).remove();
                });
            }, 1500);
            return;
        }
        
        // Show loading indicator
        const $addButton = container.find(`.make-faq-add-${taxonomyType}`);
        const originalButtonText = $addButton.html();
        $addButton.prop('disabled', true).html('<span class="make-faq-loading"></span> Adding...');
        
        if (isCategories) {
            // Add new category via AJAX
            $.ajax({
                url: makeFaqAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'make_faq_add_category',
                    nonce: makeFaqAdmin.nonce,
                    category_name: newTermName
                },
                success: function(response) {
                    if (response.success) {
                        // Add the new category to the list with the new pill design
                        const newOption = `
                            <li class="make-faq-multiselect-option selected">
                                <label class="make-faq-pill">
                                    <input type="checkbox" value="${response.data.id}" checked>
                                    <span>${response.data.name}</span>
                                    <span class="make-faq-checkmark">✓</span>
                                </label>
                            </li>
                        `;
                        container.find('.make-faq-multiselect-options').append(newOption);
                        
                        // Clear the input
                        newTermInput.val('');
                        
                        // Add event listener to the new pill
                        addPillEventListener(container.find('.make-faq-multiselect-options li:last-child label'), postId, type, container);
                        
                        // Save changes
                        saveChanges(postId, type, container);
                        
                        // Show success message
                        const successMsg = $('<div class="make-faq-status make-faq-status-success">Added successfully!</div>');
                        container.find('.make-faq-add-new-wrapper').after(successMsg);
                        setTimeout(function() {
                            successMsg.fadeOut(function() {
                                $(this).remove();
                            });
                        }, 1500);
                    } else {
                        // Show error message
                        const errorMsg = $('<div class="make-faq-status make-faq-status-error">' + (response.data.message || 'Error adding category') + '</div>');
                        container.find('.make-faq-add-new-wrapper').after(errorMsg);
                        setTimeout(function() {
                            errorMsg.fadeOut(function() {
                                $(this).remove();
                            });
                        }, 2000);
                    }
                    
                    // Reset button
                    $addButton.prop('disabled', false).html(originalButtonText);
                },
                error: function() {
                    // Show error message
                    const errorMsg = $('<div class="make-faq-status make-faq-status-error">Error adding category</div>');
                    container.find('.make-faq-add-new-wrapper').after(errorMsg);
                    setTimeout(function() {
                        errorMsg.fadeOut(function() {
                            $(this).remove();
                        });
                    }, 2000);
                    
                    // Reset button
                    $addButton.prop('disabled', false).html(originalButtonText);
                }
            });
        } else {
            // For tags, just add it to the list with the new pill design
            const newOption = `
                <li class="make-faq-multiselect-option selected">
                    <label class="make-faq-pill">
                        <input type="checkbox" value="${newTermName}" checked>
                        <span>${newTermName}</span>
                        <span class="make-faq-checkmark">✓</span>
                    </label>
                </li>
            `;
            container.find('.make-faq-multiselect-options').append(newOption);
            
            // Clear the input
            newTermInput.val('');
            
            // Add event listener to the new pill
            addPillEventListener(container.find('.make-faq-multiselect-options li:last-child label'), postId, type, container);
            
            // Save changes
            saveChanges(postId, type, container);
            
            // Show success message
            const successMsg = $('<div class="make-faq-status make-faq-status-success">Added successfully!</div>');
            container.find('.make-faq-add-new-wrapper').after(successMsg);
            setTimeout(function() {
                successMsg.fadeOut(function() {
                    $(this).remove();
                });
            }, 1500);
            
            // Reset button
            $addButton.prop('disabled', false).html(originalButtonText);
        }
    }
    
    /**
     * Helper function to add click event listener to a pill
     * 
     * @param {jQuery} $pill - The pill element
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     */
    function addPillEventListener($pill, postId, type, container) {
        $pill.on('click', function(e) {
            // Prevent default to handle it ourselves
            e.preventDefault();
            
            const $option = $(this).closest('.make-faq-multiselect-option');
            const $checkbox = $option.find('input[type="checkbox"]');
            const newCheckedState = !$checkbox.prop('checked');
            
            // Update checkbox and selected state
            $checkbox.prop('checked', newCheckedState);
            $option.toggleClass('selected', newCheckedState);
            
            // Update checkmark
            $option.find('.make-faq-checkmark').text(newCheckedState ? '✓' : '');
            
            // Save changes
            saveChanges(postId, type, container);
        });
    }

    /**
     * Save changes to the server
     * 
     * @param {number} postId - The post ID
     * @param {string} type - Either 'categories' or 'tags'
     * @param {jQuery} container - The container element
     */
    function saveChanges(postId, type, container) {
        const isCategories = type === 'categories';
        const saveAction = isCategories ? 'make_faq_save_categories' : 'make_faq_save_tags';
        const taxonomyLabel = isCategories ? 'categories' : 'tags';
        
        // Get selected values
        let selectedValues;
        let selectedCount = 0;
        
        if (isCategories) {
            selectedValues = [];
            container.find('.make-faq-multiselect-option.selected input[type="checkbox"]').each(function() {
                selectedValues.push($(this).val());
                selectedCount++;
            });
        } else {
            const selectedTags = [];
            container.find('.make-faq-multiselect-option.selected input[type="checkbox"]').each(function() {
                selectedTags.push($(this).val());
                selectedCount++;
            });
            selectedValues = selectedTags.join(', ');
        }
        
        // Show saving indicator with clear message
        const savingMsg = $('<div class="make-faq-saving-indicator"><span class="make-faq-loading"></span> Saving your ' + taxonomyLabel + '...</div>');
        
        // Remove any existing saving indicators
        container.find('.make-faq-saving-indicator').remove();
        
        // Add the new indicator
        container.find('.make-faq-multiselect-header').after(savingMsg);
        
        // Save via AJAX
        $.ajax({
            url: makeFaqAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: saveAction,
                nonce: makeFaqAdmin.nonce,
                post_id: postId,
                [isCategories ? 'categories' : 'tags']: selectedValues
            },
            success: function(response) {
                if (response.success) {
                    // Update the display
                    container.find('.make-faq-terms-display').html(response.data[isCategories ? 'categories_html' : 'tags_html']);
                    
                    // Remove saving indicator
                    savingMsg.remove();
                    
                    // Show success message with count information
                    const successMsg = $('<div class="make-faq-status make-faq-status-success">' + 
                        (selectedCount > 0 ? 
                            selectedCount + ' ' + taxonomyLabel + ' saved successfully!' : 
                            'All ' + taxonomyLabel + ' removed successfully!') + 
                        '</div>');
                    
                    // Position the message in a visible area
                    container.find('.make-faq-multiselect-header').after(successMsg);
                    
                    // Fade out after a delay
                    setTimeout(function() {
                        successMsg.fadeOut(function() {
                            $(this).remove();
                        });
                    }, 1500);
                } else {
                    // Remove saving indicator
                    savingMsg.remove();
                    
                    // Show error message
                    const errorMsg = $('<div class="make-faq-status make-faq-status-error">Error saving ' + taxonomyLabel + '. Please try again.</div>');
                    container.find('.make-faq-multiselect-header').after(errorMsg);
                    
                    // Fade out after a delay
                    setTimeout(function() {
                        errorMsg.fadeOut(function() {
                            $(this).remove();
                        });
                    }, 2000);
                }
            },
            error: function() {
                // Remove saving indicator
                savingMsg.remove();
                
                // Show error message
                const errorMsg = $('<div class="make-faq-status make-faq-status-error">Error saving ' + taxonomyLabel + '. Please try again.</div>');
                container.find('.make-faq-multiselect-header').after(errorMsg);
                
                // Fade out after a delay
                setTimeout(function() {
                    errorMsg.fadeOut(function() {
                        $(this).remove();
                    });
                }, 2000);
            }
        });
    }

    /**
     * Update the terms display
     * 
     * @param {jQuery} container - The container element
     * @param {string} type - Either 'categories' or 'tags'
     * @param {Array} allTerms - All available terms (for categories)
     * @param {Array|string} selectedTerms - Selected terms
     */
    function updateTermsDisplay(container, type, allTerms, selectedTerms) {
        const isCategories = type === 'categories';
        let displayHtml = '';
        
        if (isCategories) {
            if (allTerms && allTerms.length > 0 && selectedTerms && selectedTerms.length > 0) {
                const selectedTermNames = [];
                allTerms.forEach(function(term) {
                    if (selectedTerms.includes(term.id)) {
                        selectedTermNames.push(term.name);
                    }
                });
                
                if (selectedTermNames.length > 0) {
                    displayHtml = selectedTermNames.join(', ');
                } else {
                    displayHtml = '—';
                }
            } else {
                displayHtml = '—';
            }
        } else {
            // For tags
            if (selectedTerms && typeof selectedTerms === 'string' && selectedTerms.trim()) {
                displayHtml = selectedTerms;
            } else {
                displayHtml = '—';
            }
        }
        
        container.find('.make-faq-terms-display').html(displayHtml);
    }

    /**
     * Get all available tags
     * 
     * @return {Array} Array of tag objects
     */
    function getAllTags() {
        // We need to extract tags from the data we already have
        // This is a workaround since we don't have a dedicated endpoint to get all tags
        
        // First, try to get tags from existing elements on the page
        const tagElements = [];
        
        // Look for tags in the tag column cells
        $('.make-faq-tag-content .make-faq-terms-display a').each(function() {
            const tagName = $(this).text().trim();
            if (tagName && tagName !== '—' && !tagElements.some(t => t.name === tagName)) {
                tagElements.push({ id: tagName, name: tagName });
            }
        });
        
        // If we found tags on the page, return them
        if (tagElements.length > 0) {
            console.log('Found existing tags:', tagElements);
            return tagElements;
        }
        
        // If no tags were found, create a default empty tag to allow adding new ones
        console.log('No existing tags found, creating empty tag option');
        return [{ id: 'add-new-tag', name: 'Add a new tag...' }];
    }

})(jQuery);