/**
 * Public JavaScript for Make FAQ plugin
 */
(function($) {
    'use strict';

    // Initialize the FAQ functionality
    $(document).ready(function() {
        // Initialize accordions and toggles
        initFaqDisplay();

        // Initialize search functionality
        initFaqSearch();
    });

    /**
     * Initialize FAQ display (accordion, toggle, list)
     */
    function initFaqDisplay() {
        // Handle question click for all display styles
        $('.make-faq-container .make-faq-question').on('click', function() {
            var $item = $(this).parent('.make-faq-item');
            var $container = $item.closest('.make-faq-container');
            var displayStyle = $container.data('display-style');
            
            // Skip interaction for list style
            if (displayStyle === 'list') {
                return;
            }
            
            // For accordion, close other items first
            if (displayStyle === 'accordion') {
                $container.find('.make-faq-item').not($item).removeClass('active');
            }
            
            // Toggle active state
            $item.toggleClass('active');
            
            // Update ARIA attributes
            var $question = $(this);
            var isExpanded = $item.hasClass('active');
            $question.attr('aria-expanded', isExpanded ? 'true' : 'false');
            
            // Track view if opened
            if (isExpanded) {
                trackFaqView($item.data('id'));
            }
        });
        
        // Add accessibility attributes
        $('.make-faq-question').attr('role', 'button').attr('tabindex', '0');
        
        // Handle keyboard navigation
        $('.make-faq-question').on('keydown', function(e) {
            // Enter or Space key
            if (e.which === 13 || e.which === 32) {
                e.preventDefault();
                $(this).click();
            }
        });
    }

    /**
     * Initialize FAQ search functionality
     */
    function initFaqSearch() {
        // Handle search input for filtering FAQs
        $('.make-faq-search-input').on('input', function() {
            var searchTerm = $(this).val().toLowerCase().trim();
            filterFaqs(searchTerm);
        });
    
        // Handle form submission
        $('.make-faq-search-form').on('submit', function(e) {
            // Form submission is handled normally to update the URL
            // The filtering is done by the input event and on page load
        });
    
        // Handle clear button
        $('.make-faq-search-clear').on('click', function() {
            // Clear the search input
            var $form = $(this).closest('form');
            $form.find('.make-faq-search-input').val('');
        
            // Show all FAQs
            filterFaqs('');
        
            // Redirect to page without search parameter
            var url = window.location.href.split('?')[0];
            window.history.replaceState({}, document.title, url);
        });
    
        // Check for search parameter in URL on page load
        var urlParams = new URLSearchParams(window.location.search);
        var searchParam = urlParams.get('faq_search');
    
        if (searchParam) {
            // Set the search input value
            $('.make-faq-search-input').val(searchParam);
        
            // Filter FAQs based on the search parameter
            filterFaqs(searchParam.toLowerCase());
        }
    }

    /**
     * Filter FAQs based on search term
     */
    function filterFaqs(searchTerm) {
        // Get all FAQ items
        var $faqItems = $('.make-faq-item');
        var visibleCount = 0;
    
        // If search term is empty, show all FAQs
        if (!searchTerm) {
            $faqItems.show();
            $('.make-faq-no-results').remove();
            return;
        }
    
        // Loop through each FAQ item
        $faqItems.each(function() {
            var $item = $(this);
            var question = $item.find('.make-faq-question').text().toLowerCase();
            var answer = $item.find('.make-faq-answer').text().toLowerCase();
        
            // Check if the question or answer contains the search term
            if (question.includes(searchTerm) || answer.includes(searchTerm)) {
                $item.show();
                visibleCount++;
            } else {
                $item.hide();
            }
        });
    
        // Show no results message if no FAQs are visible
        if (visibleCount === 0) {
            // Remove existing no results message
            $('.make-faq-no-results').remove();
        
            // Add no results message after the FAQ container
            $('.make-faq-container').after('<div class="make-faq-no-results">' + 
                wp.i18n.__('No FAQs found matching your search.', 'make-faq') + 
                '</div>');
        } else {
            // Remove no results message if FAQs are visible
            $('.make-faq-no-results').remove();
        }
    }

    /**
     * Track FAQ view
     */
    function trackFaqView(faqId) {
        // Only track if we have an ID
        if (!faqId) {
            return;
        }
        
        // Send AJAX request to track view
        $.ajax({
            url: makeFaqPublic.ajaxUrl,
            type: 'POST',
            data: {
                action: 'make_faq_track_view',
                nonce: makeFaqPublic.nonce,
                id: faqId
            }
        });
    }

})(jQuery);