<?php
/**
 * Register Gutenberg blocks for the plugin.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 */

/**
 * Register Gutenberg blocks for the plugin.
 *
 * This class handles the registration and rendering of Gutenberg blocks
 * for displaying FAQs and the FAQ search functionality.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 * @author     makeWeb
 */
class Make_FAQ_Blocks {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the Gutenberg blocks.
     *
     * @since    1.0.0
     */
    public function register_blocks() {
        // Check if Gutenberg is active
        if ( ! function_exists( 'register_block_type' ) ) {
            return;
        }

        // Register block scripts and styles
        wp_register_script(
            'make-faq-blocks',
            MAKE_FAQ_PLUGIN_URL . 'assets/js/blocks.js',
            array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n' ),
            $this->version,
            true
        );

        wp_register_style(
            'make-faq-blocks-editor',
            MAKE_FAQ_PLUGIN_URL . 'assets/css/blocks-editor.css',
            array( 'wp-edit-blocks' ),
            $this->version
        );

        wp_register_style(
            'make-faq-blocks-style',
            MAKE_FAQ_PLUGIN_URL . 'assets/css/blocks-style.css',
            array(),
            $this->version
        );

        // Register localization data
        wp_localize_script(
            'make-faq-blocks',
            'makeFaqData',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'make_faq_nonce' ),
            )
        );

        // Register FAQ Display block
        register_block_type( 'make-faq/faq-display', array(
            'editor_script' => 'make-faq-blocks',
            'editor_style'  => 'make-faq-blocks-editor',
            'style'         => 'make-faq-blocks-style',
            'render_callback' => array( $this, 'render_faq_display_block' ),
            'attributes'      => array(
                'category' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'tag' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'ids' => array(
                    'type' => 'array',
                    'default' => array(),
                    'items' => array(
                        'type' => 'number',
                    ),
                ),
                'displayStyle' => array(
                    'type' => 'string',
                    'default' => 'accordion',
                ),
                'orderBy' => array(
                    'type' => 'string',
                    'default' => 'date',
                ),
                'order' => array(
                    'type' => 'string',
                    'default' => 'DESC',
                ),
                'limit' => array(
                    'type' => 'number',
                    'default' => -1,
                ),
                'className' => array(
                    'type' => 'string',
                    'default' => '',
                ),
            ),
        ) );

        // Register FAQ Search block
        register_block_type( 'make-faq/faq-search', array(
            'editor_script' => 'make-faq-blocks',
            'editor_style'  => 'make-faq-blocks-editor',
            'style'         => 'make-faq-blocks-style',
            'render_callback' => array( $this, 'render_faq_search_block' ),
            'attributes'      => array(
                'placeholder' => array(
                    'type' => 'string',
                    'default' => 'Search FAQs...',
                ),
                'buttonText' => array(
                    'type' => 'string',
                    'default' => 'Search',
                ),
                'category' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'tag' => array(
                    'type' => 'string',
                    'default' => '',
                ),
                'className' => array(
                    'type' => 'string',
                    'default' => '',
                ),
            ),
        ) );
    }

    /**
     * Render the FAQ Display block.
     *
     * @since    1.0.0
     * @param    array    $attributes    The block attributes.
     * @return   string                  The block HTML.
     */
    public function render_faq_display_block( $attributes ) {
        // Extract attributes
        $category = isset( $attributes['category'] ) ? $attributes['category'] : '';
        $tag = isset( $attributes['tag'] ) ? $attributes['tag'] : '';
        $ids = isset( $attributes['ids'] ) ? $attributes['ids'] : array();
        
        // Get default display style from settings if not specified
        if (isset($attributes['displayStyle']) && !empty($attributes['displayStyle'])) {
            $display_style = $attributes['displayStyle'];
        } else {
            $options = get_option('make_faq_settings', array());
            $display_style = isset($options['display_style']) ? $options['display_style'] : 'accordion';
        }
        
        $order_by = isset( $attributes['orderBy'] ) ? $attributes['orderBy'] : 'date';
        $order = isset( $attributes['order'] ) ? $attributes['order'] : 'DESC';
        $limit = isset( $attributes['limit'] ) ? intval( $attributes['limit'] ) : -1;
        $class_name = isset( $attributes['className'] ) ? $attributes['className'] : '';

        // Build query args
        $args = array(
            'post_type' => 'make_faq',
            'posts_per_page' => $limit,
            'orderby' => $order_by,
            'order' => $order,
        );

        // Add taxonomy queries if needed
        $tax_query = array();

        if ( ! empty( $category ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_category',
                'field'    => 'slug',
                'terms'    => explode( ',', $category ),
            );
        }

        if ( ! empty( $tag ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_tag',
                'field'    => 'slug',
                'terms'    => explode( ',', $tag ),
            );
        }

        if ( ! empty( $tax_query ) ) {
            $args['tax_query'] = $tax_query;
        }

        // Add post IDs if specified
        if ( ! empty( $ids ) ) {
            $args['post__in'] = $ids;
            $args['orderby'] = 'post__in';
        }

        // Get FAQs
        $faqs = get_posts( $args );

        // Start output buffer
        ob_start();

        // Check if we have FAQs
        if ( ! empty( $faqs ) ) {
            // Add container classes
            $container_classes = array(
                'make-faq-container',
                'make-faq-' . $display_style,
            );

            if ( ! empty( $class_name ) ) {
                $container_classes[] = $class_name;
            }

            // Start container
            echo '<div class="' . esc_attr( implode( ' ', $container_classes ) ) . '" data-display-style="' . esc_attr( $display_style ) . '">';

            // Loop through FAQs
            foreach ( $faqs as $faq ) {
                // Get FAQ data
                $question = $faq->post_title;
                $answer = $faq->post_content;

                // Apply filters to allow customization
                $question = apply_filters( 'makefaq_faq_item_question', $question, $faq );
                $answer = apply_filters( 'makefaq_faq_item_content', $answer, $faq );

                // Output FAQ item with semantic HTML and accessibility attributes
                echo '<div class="make-faq-item" data-id="' . esc_attr( $faq->ID ) . '">';
                echo '<h3 class="make-faq-question" id="faq-question-' . esc_attr( $faq->ID ) . '" aria-expanded="false" aria-controls="faq-answer-' . esc_attr( $faq->ID ) . '">' . wp_kses_post( $question ) . '</h3>';
                echo '<div class="make-faq-answer" id="faq-answer-' . esc_attr( $faq->ID ) . '" aria-labelledby="faq-question-' . esc_attr( $faq->ID ) . '">' . wp_kses_post( $answer ) . '</div>';
                echo '</div>';
            }

            // End container
            echo '</div>';
        } else {
            // No FAQs found
            echo '<div class="make-faq-no-results">' . esc_html__( 'No FAQs found.', 'make-faq' ) . '</div>';
        }

        // Return output buffer
        return ob_get_clean();
    }

    /**
     * Render the FAQ Search block.
     *
     * @since    1.0.0
     * @param    array    $attributes    The block attributes.
     * @return   string                  The block HTML.
     */
    public function render_faq_search_block( $attributes ) {
        // Extract attributes
        $placeholder = isset( $attributes['placeholder'] ) ? $attributes['placeholder'] : 'Search FAQs...';
        $button_text = isset( $attributes['buttonText'] ) ? $attributes['buttonText'] : 'Search';
        $category = isset( $attributes['category'] ) ? $attributes['category'] : '';
        $tag = isset( $attributes['tag'] ) ? $attributes['tag'] : '';
        $class_name = isset( $attributes['className'] ) ? $attributes['className'] : '';

        // Add container classes
        $container_classes = array(
            'make-faq-search-container',
        );

        if ( ! empty( $class_name ) ) {
            $container_classes[] = $class_name;
        }

        // Check for query string search parameter
        $search_value = '';
        if ( isset( $_GET['faq_search'] ) ) {
            $search_value = sanitize_text_field( $_GET['faq_search'] );
        }

        // Start output buffer
        ob_start();

        // Output search form
        ?>
        <div class="<?php echo esc_attr( implode( ' ', $container_classes ) ); ?>">
            <form class="make-faq-search-form" method="get">
                <?php if ( ! empty( $category ) ) : ?>
                    <input type="hidden" name="faq_category" value="<?php echo esc_attr( $category ); ?>">
                <?php endif; ?>
                <?php if ( ! empty( $tag ) ) : ?>
                    <input type="hidden" name="faq_tag" value="<?php echo esc_attr( $tag ); ?>">
                <?php endif; ?>
                <div class="make-faq-search-input-wrapper">
                    <input type="text" name="faq_search" class="make-faq-search-input" placeholder="<?php echo esc_attr( $placeholder ); ?>" value="<?php echo esc_attr( $search_value ); ?>">
                    <button type="submit" class="make-faq-search-button"><?php echo esc_html( $button_text ); ?></button>
                    <?php if ( ! empty( $search_value ) ) : ?>
                        <button type="button" class="make-faq-search-clear"><?php esc_html_e( 'Clear', 'make-faq' ); ?></button>
                    <?php endif; ?>
                </div>
            </form>
        </div>
        <?php

        // Track search term if provided
        if ( ! empty( $search_value ) ) {
            // Store search term in a transient for tracking
            $search_terms = get_option( 'make_faq_search_terms', array() );
            if ( isset( $search_terms[$search_value] ) ) {
                $search_terms[$search_value]++;
            } else {
                $search_terms[$search_value] = 1;
            }
            update_option( 'make_faq_search_terms', $search_terms );
        }

        // Return output buffer
        return ob_get_clean();
    }
}