<?php
/**
 * Divi module integration for the plugin.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 */

/**
 * Divi module integration for the plugin.
 *
 * This class handles the integration with the Divi Builder,
 * creating custom modules for displaying FAQs.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 * @author     makeWeb
 */
class Make_FAQ_Divi {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Initialize Divi integration.
     *
     * @since    1.0.0
     */
    public function init() {
        // Check if Divi is active
        if ( ! $this->is_divi_active() ) {
            return;
        }

        // Add custom modules
        add_action( 'et_builder_ready', array( $this, 'register_modules' ) );
        
        // Add custom CSS and JS for Divi Builder
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_divi_assets' ) );
    }

    /**
     * Check if Divi theme or Divi Builder plugin is active.
     *
     * @since    1.0.0
     * @return   boolean    True if Divi is active, false otherwise.
     */
    private function is_divi_active() {
        // Check if Divi theme is active
        $theme = wp_get_theme();
        if ( 'Divi' === $theme->get( 'Name' ) || 'Divi' === $theme->get( 'Template' ) ) {
            return true;
        }

        // Check if Divi Builder plugin is active
        if ( class_exists( 'ET_Builder_Plugin' ) ) {
            return true;
        }

        return false;
    }

    /**
     * Register Divi modules.
     *
     * @since    1.0.0
     */
    public function register_modules() {
        // Include module files
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/divi/make-faq-display-module.php';
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/divi/make-faq-search-module.php';
        
        // Register modules
        if ( class_exists( 'ET_Builder_Module' ) ) {
            new Make_FAQ_Display_Module();
            new Make_FAQ_Search_Module();
        }
    }

    /**
     * Enqueue Divi-specific assets.
     *
     * @since    1.0.0
     */
    public function enqueue_divi_assets() {
        // Only load on frontend when Divi is active
        if ( ! $this->is_divi_active() ) {
            return;
        }

        // Enqueue WordPress i18n script for translation functions
        wp_enqueue_script( 'wp-i18n' );
        
        // Enqueue Divi-specific JS for frontend functionality
        wp_enqueue_script(
            $this->plugin_name . '-divi',
            MAKE_FAQ_PLUGIN_URL . 'assets/js/make-faq-divi.js',
            array( 'jquery', 'wp-i18n' ),
            $this->version,
            true
        );

        // Enqueue Divi-specific CSS
        wp_enqueue_style(
            $this->plugin_name . '-divi',
            MAKE_FAQ_PLUGIN_URL . 'assets/css/make-faq-divi.css',
            array(),
            $this->version,
            'all'
        );
        
        // Localize the script with data
        wp_localize_script(
            $this->plugin_name . '-divi',
            'makeFaqPublic',
            array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'make_faq_public_nonce' ),
            )
        );
    }
}