<?php
/**
 * Handle Slipstream integration.
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 */

class Make_FAQ_Slipstream {

    /**
     * The ID of this plugin.
     *
     * @since    1.6.31
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.6.31
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.6.31
     * @param    string    $plugin_name       The name of the plugin.
     * @param    string    $version           The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register Slipstream integration.
     *
     * @since    1.6.31
     */
    public function init() {
        if ( ! class_exists( 'Slipstream_Registry' ) ) {
            return;
        }

        add_action( 'slipstream_init', array( $this, 'register_with_slipstream' ) );
        
        // Custom editor "striped back" functionality
        add_action( 'current_screen', array( $this, 'maybe_strip_back_editor' ) );
    }

    /**
     * Register make_faq post type with Slipstream.
     *
     * @since    1.6.31
     * @param    Slipstream_Registry $registry The Slipstream registry instance.
     */
    public function register_with_slipstream( $registry ) {
        $registry->register_post_type_for_owner( 'make_faq', array(
            'name'        => __( 'FAQs', 'make-faq' ),
            'description' => __( 'Manage your frequently asked questions.', 'make-faq' ),
            'icon'        => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor" class="w-6 h-6"><path stroke-linecap="round" stroke-linejoin="round" d="M9.879 7.519c1.171-1.025 3.071-1.025 4.242 0 1.172 1.025 1.172 2.687 0 3.712-.203.179-.43.326-.67.442-.745.361-1.45.999-1.45 1.827v.75M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-9 5.25h.008v.008H12v-.008z" /></svg>',
            'icon_bg'     => '#10b981',
            'priority'    => 20,
        ) );

        // Grant full capabilities to Owner role
        $registry->add_owner_capability_all( 'make_faq' );

        // Remove unnecessary columns in Slipstream dashboard
        $registry->remove_post_type_columns( 'make_faq', array( 'author' ) );
    }

    /**
     * Strip back the editor for FAQs when in Slipstream view.
     *
     * @since    1.6.31
     */
    public function maybe_strip_back_editor() {
        $screen = get_current_screen();

        if ( ! $screen || 'make_faq' !== $screen->post_type ) {
            return;
        }

        // Check if we are in Slipstream environment (typically checked via a global or constant, 
        // but often Slipstream might use a specific query arg or the 'Owner' role)
        // If the user specifically asked for "striped back post editor for the faqs" 
        // and we are integrating with Slipstream, we should apply these changes.
        
        if ( current_user_can( 'owner' ) ) {
            add_action( 'admin_head', array( $this, 'apply_editor_stripping' ) );
            add_filter( 'block_editor_settings_all', array( $this, 'restrict_block_editor_features' ), 10, 2 );
        }
    }

    /**
     * Apply CSS to hide unnecessary editor elements for a "striped back" feel.
     *
     * @since    1.6.31
     */
    public function apply_editor_stripping() {
        ?>
        <style>
            /* Hide unnecessary meta boxes and editor elements for FAQs in Slipstream */
            #postcustom, #commentstatusdiv, #commentsdiv, #slugdiv, #authordiv, #revisionsdiv { display: none !important; }
            .editor-post-publish-panel__link { display: none !important; }
            /* Add more selectors as needed to clean up the UI */
        </style>
        <?php
    }

    /**
     * Restrict Gutenberg block editor features for a cleaner experience.
     *
     * @since    1.6.31
     * @param    array                   $settings The editor settings.
     * @param    WP_Block_Editor_Context $context  The editor context.
     * @return   array                             Modified settings.
     */
    public function restrict_block_editor_features( $settings, $context ) {
        if ( 'make_faq' === $context->post->post_type ) {
            // Disable some features for a "striped back" editor
            $settings['richEditingEnabled'] = true;
            $settings['codeEditingEnabled'] = false; // Disable code view
            // $settings['hasFixedToolbar'] = true;
        }
        return $settings;
    }
}
