<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes
 * @author     makeWeb
 */
class Make_FAQ {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Make_FAQ_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'MAKE_FAQ_VERSION' ) ) {
            $this->version = MAKE_FAQ_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'make-faq';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->define_post_types();
        $this->define_blocks();
        $this->define_shortcodes();
        $this->define_divi_hooks();
        $this->define_slipstream_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Make_FAQ_Loader. Orchestrates the hooks of the plugin.
     * - Make_FAQ_i18n. Defines internationalization functionality.
     * - Make_FAQ_Admin. Defines all hooks for the admin area.
     * - Make_FAQ_Public. Defines all hooks for the public side of the site.
     * - Make_FAQ_Post_Types. Registers custom post types and taxonomies.
     * - Make_FAQ_Blocks. Registers Gutenberg blocks.
     * - Make_FAQ_Shortcodes. Registers shortcodes.
     * - Make_FAQ_Slipstream. Handles Slipstream integration.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-i18n.php';

        /**
         * The class responsible for registering custom post types and taxonomies.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-post-types.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'admin/class-make-faq-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'public/class-make-faq-public.php';

        /**
         * The class responsible for registering Gutenberg blocks.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-blocks.php';

        /**
         * The class responsible for registering shortcodes.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-shortcodes.php';

        /**
         * The class responsible for Divi Builder integration.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-divi.php';

        /**
         * The class responsible for Slipstream integration.
         */
        require_once MAKE_FAQ_PLUGIN_DIR . 'includes/class-make-faq-slipstream.php';

        $this->loader = new Make_FAQ_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Make_FAQ_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new Make_FAQ_i18n();
        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new Make_FAQ_Admin( $this->get_plugin_name(), $this->get_version() );

        // Enqueue admin styles and scripts
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

        // Add custom admin columns
        $this->loader->add_filter( 'manage_make_faq_posts_columns', $plugin_admin, 'set_custom_edit_make_faq_columns' );
        $this->loader->add_filter( 'manage_edit-make_faq_columns', $plugin_admin, 'set_custom_edit_make_faq_columns' );
        $this->loader->add_action( 'manage_make_faq_posts_custom_column', $plugin_admin, 'custom_make_faq_column', 10, 2 );

        // Add custom admin actions
        $this->loader->add_action( 'admin_init', $plugin_admin, 'register_admin_settings' );
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );

        // AJAX handlers for quick edit modal
        $this->loader->add_action( 'wp_ajax_make_faq_get_faq', $plugin_admin, 'ajax_get_faq' );
        $this->loader->add_action( 'wp_ajax_make_faq_save_faq', $plugin_admin, 'ajax_save_faq' );
        
        // AJAX handlers for categories and tags
        $this->loader->add_action( 'wp_ajax_make_faq_get_categories', $plugin_admin, 'ajax_get_categories' );
        $this->loader->add_action( 'wp_ajax_make_faq_save_categories', $plugin_admin, 'ajax_save_categories' );
        $this->loader->add_action( 'wp_ajax_make_faq_add_category', $plugin_admin, 'ajax_add_category' );
        $this->loader->add_action( 'wp_ajax_make_faq_get_tags', $plugin_admin, 'ajax_get_tags' );
        $this->loader->add_action( 'wp_ajax_make_faq_save_tags', $plugin_admin, 'ajax_save_tags' );
        
        // AJAX handler for trash functionality
        $this->loader->add_action( 'wp_ajax_make_faq_trash', $plugin_admin, 'ajax_trash_faq' );

        // Admin Post handler for Export
        $this->loader->add_action( 'admin_post_make_faq_export', $plugin_admin, 'handle_export' );
        // Admin Post handler for Import
        $this->loader->add_action( 'admin_post_make_faq_import', $plugin_admin, 'handle_import' );
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new Make_FAQ_Public( $this->get_plugin_name(), $this->get_version() );

        // Enqueue public styles and scripts
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

        // Add schema markup to the head
        $this->loader->add_action( 'wp_head', $plugin_public, 'add_faq_schema' );
    }

    /**
     * Register custom post types and taxonomies.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_post_types() {
        $plugin_post_types = new Make_FAQ_Post_Types();
        $this->loader->add_action( 'init', $plugin_post_types, 'register_post_types' );
        $this->loader->add_action( 'init', $plugin_post_types, 'register_taxonomies' );
    }

    /**
     * Register Gutenberg blocks.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_blocks() {
        $plugin_blocks = new Make_FAQ_Blocks( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_action( 'init', $plugin_blocks, 'register_blocks' );
    }

    /**
     * Register shortcodes.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_shortcodes() {
        $plugin_shortcodes = new Make_FAQ_Shortcodes( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_action( 'init', $plugin_shortcodes, 'register_shortcodes' );
    }

    /**
     * Initialize Divi Builder integration.
     *
     * @since    1.3.6
     * @access   private
     */
    private function define_divi_hooks() {
        $plugin_divi = new Make_FAQ_Divi( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_action( 'init', $plugin_divi, 'init' );
    }

    /**
     * Initialize Slipstream integration.
     *
     * @since    1.6.31
     * @access   private
     */
    private function define_slipstream_hooks() {
        $plugin_slipstream = new Make_FAQ_Slipstream( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_action( 'init', $plugin_slipstream, 'init' );
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Make_FAQ_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}