<?php
/**
 * Divi FAQ Display Module
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes/divi
 */

if ( ! class_exists( 'ET_Builder_Module' ) ) {
    return;
}

/**
 * Divi FAQ Display Module
 *
 * This class creates a custom Divi module for displaying FAQs.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes/divi
 * @author     makeWeb
 */
class Make_FAQ_Display_Module extends ET_Builder_Module {

    /**
     * Module slug.
     *
     * @var string
     */
    public $slug = 'make_faq_display';

    /**
     * Visual Builder support.
     *
     * @var string
     */
    public $vb_support = 'off';

    /**
     * Module init.
     */
    public function init() {
        $this->name = esc_html__( 'FAQ Display', 'make-faq' );
        $this->icon = 'j';
        $this->main_css_element = '%%order_class%%.make-faq-divi-module';
        
        $this->settings_modal_toggles = array(
            'general' => array(
                'toggles' => array(
                    'content' => esc_html__( 'Content', 'make-faq' ),
                    'display' => esc_html__( 'Display', 'make-faq' ),
                ),
            ),
            'advanced' => array(
                'toggles' => array(
                    'question' => esc_html__( 'Question', 'make-faq' ),
                    'answer'   => esc_html__( 'Answer', 'make-faq' ),
                    'item'     => esc_html__( 'Item', 'make-faq' ),
                ),
            ),
        );
    }

    /**
     * Divi Advanced Fields configuration (borders, etc.).
     * Adds border controls for Question, Answer, and Item wrap.
     */
    public function get_advanced_fields_config() {
        $advanced_fields = array();

        // Typography (full text config) for Question and Answer
        $advanced_fields['fonts'] = array(
            'question' => array(
                'label'       => esc_html__( 'Question', 'make-faq' ),
                'tab_slug'    => 'advanced',
                'toggle_slug' => 'question',
                'css'         => array(
                    'main'      => '%%order_class%% .make-faq-question',
                    'important' => 'all',
                ),
                // Enable hover tab for color and other font props
                'hover'       => 'tabs',
            ),
            'answer' => array(
                'label'       => esc_html__( 'Answer', 'make-faq' ),
                'tab_slug'    => 'advanced',
                'toggle_slug' => 'answer',
                'css'         => array(
                    'main'      => '%%order_class%% .make-faq-answer',
                    'important' => 'all',
                ),
                'hover'       => 'tabs',
            ),
        );

        // Borders for Question, Answer, and Item
        $advanced_fields['borders'] = array(
            'question' => array(
                'label_prefix' => esc_html__( 'Question', 'make-faq' ),
                'tab_slug'     => 'advanced',
                'toggle_slug'  => 'question',
                'css'          => array(
                    'main' => array(
                        'border_radii'  => '%%order_class%% .make-faq-question',
                        'border_styles' => '%%order_class%% .make-faq-question',
                    ),
                ),
            ),
            'answer' => array(
                'label_prefix' => esc_html__( 'Answer', 'make-faq' ),
                'tab_slug'     => 'advanced',
                'toggle_slug'  => 'answer',
                'css'          => array(
                    'main' => array(
                        'border_radii'  => '%%order_class%% .make-faq-answer',
                        'border_styles' => '%%order_class%% .make-faq-answer',
                    ),
                ),
            ),
            'item' => array(
                'label_prefix' => esc_html__( 'Item', 'make-faq' ),
                'tab_slug'     => 'advanced',
                'toggle_slug'  => 'item',
                'css'          => array(
                    'main' => array(
                        'border_radii'  => '%%order_class%% .make-faq-item',
                        'border_styles' => '%%order_class%% .make-faq-item',
                    ),
                ),
            ),
        );

        return $advanced_fields;
    }

    /**
     * Get the settings fields data for this element.
     *
     * @return array
     */
    public function get_fields() {
        $categories = $this->get_faq_categories();
        $tags = $this->get_faq_tags();
        
        return array(
            'category' => array(
                'label'             => esc_html__( 'Category', 'make-faq' ),
                'type'              => 'select',
                'option_category'   => 'basic_option',
                'options'           => $categories,
                'default'           => '',
                'description'       => esc_html__( 'Select a category to filter FAQs.', 'make-faq' ),
                'toggle_slug'       => 'content',
            ),
            'tag' => array(
                'label'             => esc_html__( 'Tag', 'make-faq' ),
                'type'              => 'select',
                'option_category'   => 'basic_option',
                'options'           => $tags,
                'default'           => '',
                'description'       => esc_html__( 'Select a tag to filter FAQs.', 'make-faq' ),
                'toggle_slug'       => 'content',
            ),
            'ids' => array(
                'label'             => esc_html__( 'Specific FAQ IDs', 'make-faq' ),
                'type'              => 'text',
                'option_category'   => 'basic_option',
                'description'       => esc_html__( 'Enter comma-separated FAQ IDs to display specific FAQs.', 'make-faq' ),
                'toggle_slug'       => 'content',
            ),
            'display_style' => array(
                'label'             => esc_html__( 'Display Style', 'make-faq' ),
                'type'              => 'select',
                'option_category'   => 'configuration',
                'options'           => array(
                    'list'      => esc_html__( 'List', 'make-faq' ),
                    'accordion' => esc_html__( 'Accordion', 'make-faq' ),
                    'toggle'    => esc_html__( 'Toggle', 'make-faq' ),
                ),
                'default'           => 'list',
                'description'       => esc_html__( 'Select how to display the FAQs.', 'make-faq' ),
                'toggle_slug'       => 'display',
            ),
            'orderby' => array(
                'label'             => esc_html__( 'Order By', 'make-faq' ),
                'type'              => 'select',
                'option_category'   => 'configuration',
                'options'           => array(
                    'date'     => esc_html__( 'Date', 'make-faq' ),
                    'title'    => esc_html__( 'Title', 'make-faq' ),
                    'rand'     => esc_html__( 'Random', 'make-faq' ),
                ),
                'default'           => 'date',
                'description'       => esc_html__( 'Select how to order the FAQs.', 'make-faq' ),
                'toggle_slug'       => 'display',
            ),
            'order' => array(
                'label'             => esc_html__( 'Order', 'make-faq' ),
                'type'              => 'select',
                'option_category'   => 'configuration',
                'options'           => array(
                    'DESC' => esc_html__( 'Descending', 'make-faq' ),
                    'ASC'  => esc_html__( 'Ascending', 'make-faq' ),
                ),
                'default'           => 'DESC',
                'description'       => esc_html__( 'Select the order direction.', 'make-faq' ),
                'toggle_slug'       => 'display',
            ),
            'limit' => array(
                'label'             => esc_html__( 'Number of FAQs', 'make-faq' ),
                'type'              => 'range',
                'option_category'   => 'configuration',
                'range_settings'    => array(
                    'min'  => -1,
                    'max'  => 50,
                    'step' => 1,
                ),
                'default'           => '-1',
                'description'       => esc_html__( 'Enter the number of FAQs to display (-1 for all).', 'make-faq' ),
                'toggle_slug'       => 'display',
            ),
            
            // Question styling options
            'question_background_color' => array(
                'label'             => esc_html__( 'Question Background Color', 'make-faq' ),
                'type'              => 'color-alpha',
                'custom_color'      => true,
                'default'           => '#f9f9f9',
                'description'       => esc_html__( 'Choose a background color for the question.', 'make-faq' ),
                'toggle_slug'       => 'question',
                'tab_slug'          => 'advanced',
                'hover'             => 'tabs',
            ),
            'question_font_size' => array(
                'label'             => esc_html__( 'Question Font Size', 'make-faq' ),
                'type'              => 'range',
                'option_category'   => 'font_option',
                'range_settings'    => array(
                    'min'  => 12,
                    'max'  => 36,
                    'step' => 1,
                ),
                'default'           => '16px',
                'description'       => esc_html__( 'Set the font size for the question.', 'make-faq' ),
                'toggle_slug'       => 'question',
                'tab_slug'          => 'advanced',
                'mobile_options'    => true,
                'validate_unit'     => true,
            ),
            'question_padding' => array(
                'label'             => esc_html__( 'Question Padding', 'make-faq' ),
                'type'              => 'custom_padding',
                'option_category'   => 'layout',
                'default'           => '15px|15px|15px|15px',
                'description'       => esc_html__( 'Set the padding for the question.', 'make-faq' ),
                'toggle_slug'       => 'question',
                'tab_slug'          => 'advanced',
                'mobile_options'    => true,
            ),
            'question_vertical_gap' => array(
                'label'             => esc_html__( 'Vertical Gap Between Questions', 'make-faq' ),
                'type'              => 'range',
                'option_category'   => 'layout',
                'range_settings'    => array(
                    'min'  => 0,
                    'max'  => 100,
                    'step' => 1,
                ),
                'default'           => '0px',
                'validate_unit'     => true,
                'description'       => esc_html__( 'Set vertical space between questions.', 'make-faq' ),
                'toggle_slug'       => 'question',
                'tab_slug'          => 'advanced',
            ),
            
            // Answer styling options
            'answer_background_color' => array(
                'label'             => esc_html__( 'Answer Background Color', 'make-faq' ),
                'type'              => 'color-alpha',
                'custom_color'      => true,
                'default'           => '#ffffff',
                'description'       => esc_html__( 'Choose a background color for the answer.', 'make-faq' ),
                'toggle_slug'       => 'answer',
                'tab_slug'          => 'advanced',
            ),
            'answer_font_size' => array(
                'label'             => esc_html__( 'Answer Font Size', 'make-faq' ),
                'type'              => 'range',
                'option_category'   => 'font_option',
                'range_settings'    => array(
                    'min'  => 12,
                    'max'  => 24,
                    'step' => 1,
                ),
                'default'           => '14px',
                'description'       => esc_html__( 'Set the font size for the answer.', 'make-faq' ),
                'toggle_slug'       => 'answer',
                'tab_slug'          => 'advanced',
                'mobile_options'    => true,
                'validate_unit'     => true,
            ),
            'answer_padding' => array(
                'label'             => esc_html__( 'Answer Padding', 'make-faq' ),
                'type'              => 'custom_padding',
                'option_category'   => 'layout',
                'default'           => '15px|15px|15px|15px',
                'description'       => esc_html__( 'Set the padding for the answer.', 'make-faq' ),
                'toggle_slug'       => 'answer',
                'tab_slug'          => 'advanced',
                'mobile_options'    => true,
            ),
        );
    }

    /**
     * Get FAQ categories for the select field.
     *
     * @return array
     */
    private function get_faq_categories() {
        $options = array(
            '' => esc_html__( 'All Categories', 'make-faq' ),
        );
        
        $terms = get_terms( array(
            'taxonomy'   => 'faq_category',
            'hide_empty' => false,
        ) );
        
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $options[ $term->slug ] = $term->name;
            }
        }
        
        return $options;
    }

    /**
     * Get FAQ tags for the select field.
     *
     * @return array
     */
    private function get_faq_tags() {
        $options = array(
            '' => esc_html__( 'All Tags', 'make-faq' ),
        );
        
        $terms = get_terms( array(
            'taxonomy'   => 'faq_tag',
            'hide_empty' => false,
        ) );
        
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $options[ $term->slug ] = $term->name;
            }
        }
        
        return $options;
    }

    /**
     * Render the module output.
     *
     * @param array  $attrs       List of attributes.
     * @param string $content     Content being processed.
     * @param string $render_slug Slug of module that is used for rendering output.
     *
     * @return string
     */
    public function render( $attrs, $content = null, $render_slug ) {
        // Get module attributes
        $category = $this->props['category'];
        $tag = $this->props['tag'];
        $ids = $this->props['ids'];
        $display_style = $this->props['display_style'];
        $orderby = $this->props['orderby'];
        $order = $this->props['order'];
        $limit = (int) $this->props['limit'];
        
        // Build query args
        $args = array(
            'post_type'      => 'make_faq',
            'posts_per_page' => $limit,
            'orderby'        => $orderby,
            'order'          => $order,
        );
        
        // Add taxonomy queries if needed
        $tax_query = array();
        
        if ( ! empty( $category ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_category',
                'field'    => 'slug',
                'terms'    => explode( ',', $category ),
            );
        }
        
        if ( ! empty( $tag ) ) {
            $tax_query[] = array(
                'taxonomy' => 'faq_tag',
                'field'    => 'slug',
                'terms'    => explode( ',', $tag ),
            );
        }
        
        if ( ! empty( $tax_query ) ) {
            $args['tax_query'] = $tax_query;
        }
        
        // Add post IDs if specified
        if ( ! empty( $ids ) ) {
            $args['post__in'] = explode( ',', $ids );
            $args['orderby'] = 'post__in';
        }
        
        // Get FAQs
        $faqs = get_posts( $args );
        
        // Start output buffer
        ob_start();
        
        // Check if we have FAQs
        if ( ! empty( $faqs ) ) {
            // Get styling properties
            $question_bg_color = isset($this->props['question_background_color']) ? $this->props['question_background_color'] : '#f9f9f9';
            $question_text_color = isset($this->props['question_text_color']) ? $this->props['question_text_color'] : '';
            $question_font_size = isset($this->props['question_font_size']) ? $this->props['question_font_size'] : '16px';
            $question_padding = isset($this->props['question_padding']) ? $this->props['question_padding'] : '15px|15px|15px|15px';
            
            $answer_bg_color = isset($this->props['answer_background_color']) ? $this->props['answer_background_color'] : '#ffffff';
            $answer_text_color = isset($this->props['answer_text_color']) ? $this->props['answer_text_color'] : '';
            $answer_font_size = isset($this->props['answer_font_size']) ? $this->props['answer_font_size'] : '14px';
            $answer_padding = isset($this->props['answer_padding']) ? $this->props['answer_padding'] : '15px|15px|15px|15px';
            
            // Parse padding values
            $question_padding_values = explode('|', $question_padding);
            $question_padding_top = isset($question_padding_values[0]) ? $question_padding_values[0] : '15px';
            $question_padding_right = isset($question_padding_values[1]) ? $question_padding_values[1] : '15px';
            $question_padding_bottom = isset($question_padding_values[2]) ? $question_padding_values[2] : '15px';
            $question_padding_left = isset($question_padding_values[3]) ? $question_padding_values[3] : '15px';
            
            $answer_padding_values = explode('|', $answer_padding);
            $answer_padding_top = isset($answer_padding_values[0]) ? $answer_padding_values[0] : '15px';
            $answer_padding_right = isset($answer_padding_values[1]) ? $answer_padding_values[1] : '15px';
            $answer_padding_bottom = isset($answer_padding_values[2]) ? $answer_padding_values[2] : '15px';
            $answer_padding_left = isset($answer_padding_values[3]) ? $answer_padding_values[3] : '15px';
            
            // Add container classes
            $container_classes = array(
                'make-faq-container',
                'make-faq-' . $display_style,
                'make-faq-divi-module',
                $this->module_classname($render_slug),
            );
            
            // Get module ID for the container
            $module_id = $this->props['module_id'];
            
            // Apply styles using Divi's set_style method
            if (!empty($question_bg_color)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-question',
                    'declaration' => sprintf('background-color: %1$s;', $question_bg_color),
                ));
            }
            // Built-in Hover styles for question (Divi hover tabs)
            $question_bg_color_hover   = isset($this->props['question_background_color__hover']) ? $this->props['question_background_color__hover'] : '';
            $question_text_color_hover = isset($this->props['question_text_color__hover']) ? $this->props['question_text_color__hover'] : '';
            if (!empty($question_bg_color_hover)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-question:hover',
                    'declaration' => sprintf('background-color: %1$s;', $question_bg_color_hover),
                ));
            }
            if (!empty($question_font_size)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-question',
                    'declaration' => sprintf('font-size: %1$s;', $question_font_size),
                ));
            }
            if (!empty($question_padding)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-question',
                    'declaration' => sprintf(
                        'padding: %1$s %2$s %3$s %4$s;',
                        $question_padding_top,
                        $question_padding_right,
                        $question_padding_bottom,
                        $question_padding_left
                    ),
                ));
            }
            
            if (!empty($answer_bg_color)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-answer',
                    'declaration' => sprintf('background-color: %1$s;', $answer_bg_color),
                ));
            }
            if (!empty($answer_font_size)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-answer',
                    'declaration' => sprintf('font-size: %1$s;', $answer_font_size),
                ));
            }
            if (!empty($answer_padding)) {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-answer',
                    'declaration' => sprintf(
                        'padding: %1$s %2$s %3$s %4$s;',
                        $answer_padding_top,
                        $answer_padding_right,
                        $answer_padding_bottom,
                        $answer_padding_left
                    ),
                ));
            }
            
            // Vertical gap between questions
            $question_vertical_gap = isset($this->props['question_vertical_gap']) ? trim($this->props['question_vertical_gap']) : '';
            if ($question_vertical_gap !== '' && $question_vertical_gap !== '0' && $question_vertical_gap !== '0px') {
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-item',
                    'declaration' => sprintf('margin-bottom: %1$s;', $question_vertical_gap),
                ));
                ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-item:last-child',
                    'declaration' => 'margin-bottom: 0;',
                ));
            }
            
            // Start container
            echo '<div id="' . esc_attr($module_id) . '" class="' . esc_attr( implode( ' ', $container_classes ) ) . '" data-display-style="' . esc_attr( $display_style ) . '">';
            
            // Loop through FAQs
            foreach ( $faqs as $faq ) {
                // Get FAQ data
                $question = $faq->post_title;
                $answer = $faq->post_content;
                
                // Apply filters to allow customization
                $question = apply_filters( 'makefaq_faq_item_question', $question, $faq );
                $answer = apply_filters( 'makefaq_faq_item_content', $answer, $faq );
                
                // Output FAQ item with semantic HTML and accessibility attributes
                echo '<div class="make-faq-item" data-id="' . esc_attr( $faq->ID ) . '">';
                echo '<h3 class="make-faq-question" id="faq-question-' . esc_attr( $faq->ID ) . '" aria-expanded="false" aria-controls="faq-answer-' . esc_attr( $faq->ID ) . '">' . wp_kses_post( $question ) . '</h3>';
                echo '<div class="make-faq-answer" id="faq-answer-' . esc_attr( $faq->ID ) . '" aria-labelledby="faq-question-' . esc_attr( $faq->ID ) . '">' . wp_kses_post( $answer ) . '</div>';
                echo '</div>';
            }
            
            // End container
            echo '</div>';
        } else {
            // No FAQs found
            echo '<div class="make-faq-no-results">' . esc_html__( 'No FAQs found.', 'make-faq' ) . '</div>';
        }
        
        // Return output buffer
        return ob_get_clean();
    }

}