<?php
/**
 * Divi FAQ Search Module
 *
 * @link       https://makeweb.com
 * @since      1.0.0
 *
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes/divi
 */

if ( ! class_exists( 'ET_Builder_Module' ) ) {
    return;
}

/**
 * Divi FAQ Search Module
 *
 * This class creates a custom Divi module for searching FAQs.
 *
 * @since      1.0.0
 * @package    Make_FAQ
 * @subpackage Make_FAQ/includes/divi
 * @author     makeWeb
 */
class Make_FAQ_Search_Module extends ET_Builder_Module {

    /**
     * Module slug.
     *
     * @var string
     */
    public $slug = 'make_faq_search';

    /**
     * Visual Builder support.
     *
     * @var string
     */
    public $vb_support = 'off';

    /**
     * Module init.
     */
    public function init() {
        $this->name = esc_html__( 'FAQ Search', 'make-faq' );
        $this->icon = 's';
        $this->main_css_element = '%%order_class%%.make-faq-search-divi-module';

        $this->settings_modal_toggles = array(
                'general' => array(
                        'toggles' => array(
                                'content' => esc_html__( 'Content', 'make-faq' ),
                        ),
                ),
                'advanced' => array(
                        'toggles' => array(
                                'search_form'   => esc_html__( 'Search Form', 'make-faq' ),
                                'module'        => esc_html__( 'Module', 'make-faq' ),
                        ),
                ),
        );
    }

    /**
     * Get the settings fields data for this element.
     *
     * @return array
     */
    /**
     * Divi Advanced Fields configuration for Search module (typography, etc.)
     */
    public function get_advanced_fields_config() {
        $advanced_fields = array();

        // Typography for Input
        $advanced_fields['fonts'] = array(
                'input' => array(
                        'label'       => esc_html__( 'Input ', 'make-faq' ),
                        'tab_slug'    => 'advanced',
                        'toggle_slug' => 'search_form',
                        'css'         => array(
                                'main'      => '%%order_class%% .make-faq-search-input',
                                'important' => 'all',
                        ),
                        'hover'       => 'tabs',
                ),
        );

        // Borders for Input (per-corner radii and styles via Divi UI) and for the main element
        $advanced_fields['borders'] = array(
                'input' => array(
                        'label_prefix' => esc_html__( 'Input', 'make-faq' ),
                        'tab_slug'     => 'advanced',
                        'toggle_slug'  => 'search_form',
                        'css'          => array(
                                'main' => array(
                                        'border_radii'  => '%%order_class%% .make-faq-search-input',
                                        'border_styles' => '%%order_class%% .make-faq-search-input',
                                ),
                        ),
                ),
                'module' => array(
                        'label_prefix' => esc_html__( 'Module', 'make-faq' ),
                        'tab_slug'     => 'advanced',
                        'toggle_slug'  => 'module',
                        'css'          => array(
                                'main' => array(
                                        'border_radii'  => $this->main_css_element,
                                        'border_styles' => $this->main_css_element,
                                ),
                        ),
                ),
        );

        // Built-in Divi Button styler
        $advanced_fields['button'] = array(
                'button' => array(
                        'label'          => et_builder_i18n( 'Button' ),
                        'css'            => array(
                                'main'         => $this->main_css_element . ' .et_pb_button',
                                'limited_main' => $this->main_css_element . ' .et_pb_button',
                        ),
                        'box_shadow'     => false,
                        'margin_padding' => false,
                ),
        );

        // Background for the main element (Module)
        $advanced_fields['background'] = array(
                'tab_slug'    => 'advanced',
                'toggle_slug' => 'module',
                'css'         => array(
                        'main' => $this->main_css_element,
                ),
        );

        // Enable module spacing (Margin & Padding) and Sizing controls on the main element
        $advanced_fields['margin_padding'] = array(
                'css' => array(
                        'main' => $this->main_css_element,
                ),
        );
        // Use Divi's max_width control with module_alignment mapped to the main element
        $advanced_fields['max_width'] = array(
                'css' => array(
                        'main' => $this->main_css_element,
                        'module_alignment' => $this->main_css_element,
                ),
                'use_max_width' => true,
        );

        return $advanced_fields;
    }

    public function get_fields() {
        $categories = $this->get_faq_categories();
        $tags = $this->get_faq_tags();

        return array(
                'placeholder' => array(
                        'label'             => esc_html__( 'Placeholder Text', 'make-faq' ),
                        'type'              => 'text',
                        'option_category'   => 'basic_option',
                        'default'           => esc_html__( 'Search FAQs...', 'make-faq' ),
                        'description'       => esc_html__( 'Enter the placeholder text for the search input.', 'make-faq' ),
                        'toggle_slug'       => 'content',
                ),
                'button_text' => array(
                        'label'             => esc_html__( 'Button Text', 'make-faq' ),
                        'type'              => 'text',
                        'option_category'   => 'basic_option',
                        'default'           => esc_html__( 'Search', 'make-faq' ),
                        'description'       => esc_html__( 'Enter the text for the search button.', 'make-faq' ),
                        'toggle_slug'       => 'content',
                ),
                'category' => array(
                        'label'             => esc_html__( 'Limit to Category', 'make-faq' ),
                        'type'              => 'select',
                        'option_category'   => 'basic_option',
                        'options'           => $categories,
                        'default'           => '',
                        'description'       => esc_html__( 'Limit search results to a specific category.', 'make-faq' ),
                        'toggle_slug'       => 'content',
                ),
                'tag' => array(
                        'label'             => esc_html__( 'Limit to Tag', 'make-faq' ),
                        'type'              => 'select',
                        'option_category'   => 'basic_option',
                        'options'           => $tags,
                        'default'           => '',
                        'description'       => esc_html__( 'Limit search results to a specific tag.', 'make-faq' ),
                        'toggle_slug'       => 'content',
                ),


            // Search Form Styling
                'input_background_color' => array(
                        'label'             => esc_html__( 'Input Background Color', 'make-faq' ),
                        'type'              => 'color-alpha',
                        'custom_color'      => true,
                        'default'           => '#ffffff',
                        'description'       => esc_html__( 'Choose a background color for the search input.', 'make-faq' ),
                        'toggle_slug'       => 'search_form',
                        'tab_slug'          => 'advanced',
                ),
                'input_padding' => array(
                        'label'             => esc_html__( 'Input Padding', 'make-faq' ),
                        'type'              => 'custom_padding',
                        'option_category'   => 'layout',
                        'default'           => '8px|12px|8px|12px',
                        'description'       => esc_html__( 'Set the padding for the search input.', 'make-faq' ),
                        'toggle_slug'       => 'search_form',
                        'tab_slug'          => 'advanced',
                        'mobile_options'    => true,
                ),
                'input_button_gap' => array(
                        'label'             => esc_html__( 'Gap Between Input and Button', 'make-faq' ),
                        'type'              => 'range',
                        'option_category'   => 'layout',
                        'range_settings'    => array(
                                'min'  => 0,
                                'max'  => 50,
                                'step' => 1,
                        ),
                        'default'           => '0px',
                        'validate_unit'     => true,
                        'description'       => esc_html__( 'Set horizontal space between the input and the button.', 'make-faq' ),
                        'toggle_slug'       => 'search_form',
                        'tab_slug'          => 'advanced',
                ),

        );
    }

    /**
     * Get FAQ categories for the select field.
     *
     * @return array
     */
    private function get_faq_categories() {
        $options = array(
                '' => esc_html__( 'All Categories', 'make-faq' ),
        );

        $terms = get_terms( array(
                'taxonomy'   => 'faq_category',
                'hide_empty' => false,
        ) );

        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $options[ $term->slug ] = $term->name;
            }
        }

        return $options;
    }

    /**
     * Get FAQ tags for the select field.
     *
     * @return array
     */
    private function get_faq_tags() {
        $options = array(
                '' => esc_html__( 'All Tags', 'make-faq' ),
        );

        $terms = get_terms( array(
                'taxonomy'   => 'faq_tag',
                'hide_empty' => false,
        ) );

        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $term ) {
                $options[ $term->slug ] = $term->name;
            }
        }

        return $options;
    }

    /**
     * Render the module output.
     *
     * @param array  $attrs       List of attributes.
     * @param string $content     Content being processed.
     * @param string $render_slug Slug of module that is used for rendering output.
     *
     * @return string
     */
    public function render( $attrs, $content = null, $render_slug ) {
        // Get module attributes
        $placeholder = $this->props['placeholder'];
        $button_text = $this->props['button_text'];
        $category = $this->props['category'];
        $tag = $this->props['tag'];

        // Add container classes
        $container_classes = array(
                'make-faq-search-container',
                'make-faq-search-divi-module',
                $this->module_classname($render_slug),
        );

        // Check for query string search parameter
        $search_value = '';
        if ( isset( $_GET['faq_search'] ) ) {
            $search_value = sanitize_text_field( $_GET['faq_search'] );
        }

        // Start output buffer
        ob_start();

        // Get styling properties for the input
        $input_bg_color = isset($this->props['input_background_color']) ? $this->props['input_background_color'] : '#ffffff';
        $input_text_color = isset($this->props['input_text_color']) ? $this->props['input_text_color'] : '';
        $input_padding = isset($this->props['input_padding']) ? $this->props['input_padding'] : '8px|12px|8px|12px';
        $input_button_gap = isset($this->props['input_button_gap']) ? trim($this->props['input_button_gap']) : '';


        // Parse padding values for input
        $input_padding_values = explode('|', $input_padding);
        $input_padding_top = isset($input_padding_values[0]) ? $input_padding_values[0] : '8px';
        $input_padding_right = isset($input_padding_values[1]) ? $input_padding_values[1] : '12px';
        $input_padding_bottom = isset($input_padding_values[2]) ? $input_padding_values[2] : '8px';
        $input_padding_left = isset($input_padding_values[3]) ? $input_padding_values[3] : '12px';

        // Get module ID for the container
        $module_id = $this->props['module_id'];

        // Apply styles using Divi's set_style method
        // Input styles
        if (!empty($input_bg_color)) {
            ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-search-input',
                    'declaration' => sprintf('background-color: %1$s;', $input_bg_color),
            ));
        }
        if (!empty($input_text_color)) {
            ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-search-input',
                    'declaration' => sprintf('color: %1$s;', $input_text_color),
            ));
        }
        if (!empty($input_padding)) {
            ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-search-input',
                    'declaration' => sprintf(
                            'padding: %1$s %2$s %3$s %4$s;',
                            $input_padding_top,
                            $input_padding_right,
                            $input_padding_bottom,
                            $input_padding_left
                    ),
            ));
        }
        
        // Gap between input and button
        if ($input_button_gap !== '' && $input_button_gap !== '0' && $input_button_gap !== '0px') {
            ET_Builder_Element::set_style($render_slug, array(
                    'selector'    => '%%order_class%% .make-faq-search-input-wrapper',
                    'declaration' => sprintf('gap: %1$s; column-gap: %1$s;', $input_button_gap),
            ));
        }


        // Output search form
        ?>

        <div id="<?php echo esc_attr($module_id); ?>" class="<?php echo esc_attr( implode( ' ', $container_classes ) ); ?>">
            <form class="make-faq-search-form" method="get">
                <?php if ( ! empty( $category ) ) : ?>
                    <input type="hidden" name="faq_category" value="<?php echo esc_attr( $category ); ?>">
                <?php endif; ?>

                <?php if ( ! empty( $tag ) ) : ?>
                    <input type="hidden" name="faq_tag" value="<?php echo esc_attr( $tag ); ?>">
                <?php endif; ?>

                <div class="make-faq-search-input-wrapper">
                    <div class="make-faq-input-holder<?php echo $search_value !== '' ? ' has-value' : ''; ?>">
                        <input type="text" name="faq_search" class="make-faq-search-input" placeholder="<?php echo esc_attr( $placeholder ); ?>" value="<?php echo esc_attr( $search_value ); ?>">
                        <button type="button" class="make-faq-search-clear" aria-label="<?php esc_attr_e('Clear search', 'make-faq'); ?>" title="<?php esc_attr_e('Clear search', 'make-faq'); ?>">
                            <svg class="make-faq-clear-icon" width="16" height="16" viewBox="0 0 24 24" aria-hidden="true" focusable="false">
                                <path fill="currentColor" d="M18.3 5.71a1 1 0 0 0-1.41 0L12 10.59 7.11 5.7A1 1 0 0 0 5.7 7.11L10.59 12l-4.9 4.89a1 1 0 1 0 1.41 1.41L12 13.41l4.89 4.9a1 1 0 0 0 1.41-1.41L13.41 12l4.9-4.89a1 1 0 0 0-.01-1.4z"/>
                            </svg>
                        </button>
                    </div>
                    <button type="submit" class="make-faq-search-button et_pb_button"><?php echo esc_html( $button_text ); ?></button>
                </div>
            </form>
        </div>
        <?php

        // Track search term if provided
        if ( ! empty( $search_value ) ) {
            // Store search term in a transient for tracking
            $search_terms = get_option( 'make_faq_search_terms', array() );
            if ( isset( $search_terms[$search_value] ) ) {
                $search_terms[$search_value]++;
            } else {
                $search_terms[$search_value] = 1;
            }
            update_option( 'make_faq_search_terms', $search_terms );
        }

        // Return output buffer
        return ob_get_clean();
    }

}