<?php
defined('ABSPATH') || exit;

// Register Import/Export submenu under Glossaries
add_action('admin_menu', function () {
	add_submenu_page(
		'edit.php?post_type=glossaries',
		'Glossary CSV Import/Export',
		'Import / Export',
		'manage_options',
		'mw-glossary-import-export',
		'mw_glossary_render_import_export_page'
	);
});

// Handle export action
add_action('admin_init', function () {
	if (!is_admin()) return;
	if (!current_user_can('manage_options')) return;
	if (!isset($_GET['page']) || $_GET['page'] !== 'mw-glossary-import-export') return;
	if (!isset($_GET['mw_glossary_export']) || $_GET['mw_glossary_export'] !== '1') return;
	check_admin_referer('mw_glossary_export');

	$filename = 'glossary-export-' . date('Y-m-d-His') . '.csv';
	header('Content-Type: text/csv; charset=utf-8');
	header('Content-Disposition: attachment; filename=' . $filename);
	$fh = fopen('php://output', 'w');
	// CSV header now includes ID to enable safe overrides on import
	fputcsv($fh, ['ID', 'Title', 'Description']);

	$query = new WP_Query([
		'post_type' => 'glossaries',
		'posts_per_page' => -1,
		'orderby' => 'title',
		'order' => 'ASC',
		'post_status' => 'any',
	]);

	if ($query->have_posts()) {
		while ($query->have_posts()) {
			$query->the_post();
			$id    = get_the_ID();
			$title = get_the_title();
			$desc  = function_exists('get_field') ? (string) get_field('description') : (string) get_post_meta($id, 'description', true);
			// Strip tags and normalize whitespace for CSV safety
			$desc_text = trim(wp_strip_all_tags($desc));
			fputcsv($fh, [$id, $title, $desc_text]);
		}
		wp_reset_postdata();
	}

	fclose($fh);
	exit;
});

// Handle import action
add_action('admin_post_mw_glossary_import', function () {
	if (!current_user_can('manage_options')) wp_die('Insufficient permissions');
	check_admin_referer('mw_glossary_import');

	$redirect = admin_url('edit.php?post_type=glossaries&page=mw-glossary-import-export');

	if (!isset($_FILES['mw_glossary_csv']) || empty($_FILES['mw_glossary_csv']['tmp_name'])) {
		wp_safe_redirect(add_query_arg(['mw_glossary_msg' => 'no_file'], $redirect));
		exit;
	}

	$file = $_FILES['mw_glossary_csv'];
	if (!empty($file['error'])) {
		wp_safe_redirect(add_query_arg(['mw_glossary_msg' => 'upload_error'], $redirect));
		exit;
	}

	$handle = fopen($file['tmp_name'], 'r');
	if (!$handle) {
		wp_safe_redirect(add_query_arg(['mw_glossary_msg' => 'open_error'], $redirect));
		exit;
	}

	// Optionally skip BOM
	$firstBytes = fread($handle, 3);
	if ($firstBytes !== "\xEF\xBB\xBF") {
		// not BOM, rewind
		fseek($handle, 0);
	}

	// Read header
	$header = fgetcsv($handle);
	if (!$header) {
		fclose($handle);
		wp_safe_redirect(add_query_arg(['mw_glossary_msg' => 'empty'], $redirect));
		exit;
	}

	// Normalize columns to ID, Title, Description
	$map = ['id' => null, 'title' => null, 'description' => null];
	foreach ($header as $i => $col) {
		$lc = strtolower(trim($col));
		if ($lc === 'id') $map['id'] = $i;
		if ($lc === 'title') $map['title'] = $i;
		if ($lc === 'description') $map['description'] = $i;
	}
	if ($map['title'] === null && $map['id'] === null) {
		fclose($handle);
		wp_safe_redirect(add_query_arg(['mw_glossary_msg' => 'bad_header'], $redirect));
		exit;
	}
	$hasDescCol = ($map['description'] !== null);

	$created = 0; $updated = 0; $skipped = 0;
	while (($row = fgetcsv($handle)) !== false) {
		$idVal = ($map['id'] !== null && isset($row[$map['id']])) ? intval(trim($row[$map['id']])) : 0;
		$title = ($map['title'] !== null && isset($row[$map['title']])) ? trim($row[$map['title']]) : '';
		$desc  = ($hasDescCol && isset($row[$map['description']])) ? trim($row[$map['description']]) : null;
		if ($title === '' && $idVal <= 0) { $skipped++; continue; }

		$updatedById = false;
		// If ID provided, try to update that specific post
		if ($idVal > 0) {
			$post = get_post($idVal);
			if ($post && $post->post_type === 'glossaries') {
				// Update title and description to override safely
				wp_update_post([
					'ID' => $idVal,
					'post_title' => ($title !== '' ? $title : $post->post_title),
				]);
				if ($hasDescCol) {
					if (function_exists('update_field')) {
						update_field('description', (string)$desc, $idVal);
					} else {
						update_post_meta($idVal, 'description', (string)$desc);
					}
				}
				$updated++;
				$updatedById = true;
			}
		}

		if ($updatedById) {
			continue;
		}

		// No valid ID update — fallback to find existing by exact title in glossaries
		if ($title === '') { $skipped++; continue; }
		$existing_post = get_page_by_title($title, OBJECT, 'glossaries');

		if ($existing_post) {
			$post_id = $existing_post->ID;
			// Update description only (title is the same here)
			if ($hasDescCol) {
				if (function_exists('update_field')) {
					update_field('description', (string)$desc, $post_id);
				} else {
					update_post_meta($post_id, 'description', (string)$desc);
				}
			}
			$updated++;
		} else {
			$post_id = wp_insert_post([
				'post_title' => $title,
				'post_type' => 'glossaries',
				'post_status' => 'publish',
			]);
			if (!is_wp_error($post_id)) {
				if ($hasDescCol) {
					if (function_exists('update_field')) {
						update_field('description', (string)$desc, $post_id);
					} else {
						update_post_meta($post_id, 'description', (string)$desc);
					}
				}
				$created++;
			} else {
				$skipped++;
			}
		}
	}
	fclose($handle);
	wp_safe_redirect(add_query_arg([
		'mw_glossary_msg' => 'import_ok',
		'c' => $created,
		'u' => $updated,
		's' => $skipped,
	], $redirect));
	exit;
});

function mw_glossary_render_import_export_page() {
	if (!current_user_can('manage_options')) return;
	// Notices
	$msg = isset($_GET['mw_glossary_msg']) ? sanitize_text_field($_GET['mw_glossary_msg']) : '';
	?>
	<div class="wrap">
		<h1>Glossary Import / Export</h1>
		<?php if ($msg): ?>
			<?php if ($msg === 'import_ok'): ?>
				<div class="notice notice-success"><p>Import completed. Created: <?php echo isset($_GET['c']) ? intval($_GET['c']) : 0; ?>, Updated: <?php echo isset($_GET['u']) ? intval($_GET['u']) : 0; ?>, Skipped: <?php echo isset($_GET['s']) ? intval($_GET['s']) : 0; ?>.</p></div>
			<?php elseif ($msg === 'no_file'): ?>
				<div class="notice notice-error"><p>Please choose a CSV file to upload.</p></div>
			<?php elseif ($msg === 'upload_error'): ?>
				<div class="notice notice-error"><p>There was an error uploading the file.</p></div>
			<?php elseif ($msg === 'open_error'): ?>
				<div class="notice notice-error"><p>Unable to open uploaded file.</p></div>
			<?php elseif ($msg === 'empty'): ?>
				<div class="notice notice-error"><p>The CSV appears to be empty.</p></div>
			<?php elseif ($msg === 'bad_header'): ?>
				<div class="notice notice-error"><p>CSV must include at least a Title or ID column (Description is optional).</p></div>
			<?php endif; ?>
		<?php endif; ?>

		<p>Use this page to export your current glossaries to CSV, or import new/updated entries from a CSV file. The CSV should contain the following columns:</p>
		<ul>
			<li><strong>Title</strong> (required)</li>
			<li><strong>ID</strong> (optional; when provided, the importer will update that exact Glossary post, overriding its Title and Description)</li>
			<li><strong>Description</strong> (optional; HTML allowed, will be stored in the Description metabox)</li>
		</ul>

		<h2>Export</h2>
		<p>Download a CSV of all current glossary entries.</p>
		<p>
			<a class="button button-primary" href="<?php echo wp_nonce_url(admin_url('edit.php?post_type=glossaries&page=mw-glossary-import-export&mw_glossary_export=1'), 'mw_glossary_export'); ?>">Export CSV</a>
		</p>

		<hr />

		<h2>Import</h2>
		<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
			<?php wp_nonce_field('mw_glossary_import'); ?>
			<input type="hidden" name="action" value="mw_glossary_import" />
			<p>
				<input type="file" name="mw_glossary_csv" accept=".csv,text/csv" required />
			</p>
			<p class="description">Importer logic: if an <strong>ID</strong> is provided, that exact Glossary post will be updated (title and description). If <strong>ID</strong> is blank, the importer will update by exact <strong>Title</strong> match or create a new Glossary to avoid duplicates.</p>
			<p>
				<input type="submit" class="button button-primary" value="Import CSV" />
			</p>
		</form>

		<hr />

		<h2>Instructions: Using the Description Metabox</h2>
		<p>Each glossary entry uses a metabox labeled <strong>Description</strong> on the Glossary post edit screen. Enter the explanation or definition here. This field supports basic formatting.</p>
		<p>Imported CSV <em>Description</em> values are saved into this same metabox. You can edit them later in the editor.</p>

		<h2>Shortcode</h2>
		<p>Display your glossary anywhere using the shortcode:</p>
		<pre><code>[mw_glossary]</code></pre>
	</div>
	<?php
}
