/* Notice Bar Admin JS (custom settings page)
 * Enhances the admin UI with conditional logic, preview, validation, and helpers.
 */
(function($){
  function pad(n){ return (n<10?'0':'') + n; }
  function ampm(h){ return h>=12 ? 'pm' : 'am'; }
  function to12h(h){ h = h%12; return h===0?12:h; }
  function formatDate(d){
    // Format: d/m/Y g:i a
    const day = d.getDate();
    const month = d.getMonth()+1;
    const year = d.getFullYear();
    const hours = d.getHours();
    const minutes = d.getMinutes();
    return day + '/' + month + '/' + year + ' ' + to12h(hours) + ':' + pad(minutes) + ' ' + ampm(hours);
  }

  function clamp(num, min, max){ return Math.min(Math.max(num, min), max); }

  $(function(){
    const $wrap = $('.noticebar-admin');
    if (!$wrap.length) return;

    // Tabs
    const $tabNav = $wrap.find('.noticebar-tabs .tab-nav');
    const $tabBtns = $tabNav.find('[role="tab"]');
    const $panels = $wrap.find('.noticebar-tabs .tab-panels .tab-panel');

    function activateTab(name){
      $tabBtns.attr('aria-selected', 'false');
      $tabBtns.filter('[data-tab="'+name+'"]').attr('aria-selected', 'true');
      $panels.addClass('is-hidden');
      const $panel = $panels.filter('[data-tab="'+name+'"]').removeClass('is-hidden');
      // Re-run conditional UI in the now-visible panel
      updateScheduleVisibility();
      updateMobileVisibility();
      updateActionFields();
      updateBgVisibility();
      updateCounters();
      updatePreview();
    }

    $tabNav.on('click', '[role="tab"]', function(){
      const name = $(this).data('tab');
      activateTab(name);
    });

    // Elements
    function getDisplay(){
      const $sel = $('input.nba-display-option:checked');
      return $sel.length ? $sel.val() : 'Active';
    }
    const $starts = $('#nba_starts');
    const $ends = $('#nba_ends');
    const $mobileToggle = $('#nba_mobile_toggle');
    const $mobileMsgRow = $('.field-mobile-message');
    const $msg = $('#nba_message');
    const $msgCounter = $('#nba_message_counter');
    const $mobileMsg = $('#nba_mobile_message');
    const $mobileCounter = $('#nba_mobile_message_counter');

    const $actionType = $('#nba_action_type');
    // Updated selectors to match per-row structure
    const $actionLinkRows = $('.action-link-row');
    const $actionButtonRows = $('.action-button-row');
    const $linkText = $('#nba_link_text');
    const $buttonText = $('#nba_button_text');
    // Unified Action URL + New Tab controls
    const $actionUrl = $('#nba_action_url');
    const $actionNewTab = $('#nba_action_new_tab');

    const $fontSize = $('#nba_font_size');
    const $fontColour = $('#nba_font_colour');
    const $linkColour = $('#nba_link_colour');

    const $bgStyle = $('#nba_bg_style');
    const $bgColour = $('#nba_bg_colour');
    const $bgGradStart = $('#nba_bg_grad_start');
    const $bgGradEnd = $('#nba_bg_grad_end');
    const $bgDirection = $('#nba_bg_direction');

    const $btnBg = $('#nba_btn_bg');
    const $btnBgHover = $('#nba_btn_bg_hover');
    const $btnFont = $('#nba_btn_font');

    // Close button controls (now in General tab)
    const $closeEnable = $('#nba_close_enable');
    const $closeDurationRow = $('.close-duration-row');
    const $closeDuration = $('#nba_close_duration');

    // Preview elements
    const $prevWrap = $('.noticebar-preview');
    const $prevBar = $('#nba_preview_bar');
    const $prevMsg = $('#nba_preview_message');
    const $prevBtn = $('#nba_preview_button');
    const $prevLink = $('#nba_preview_link');
    const $prevStyle = $('#nba_preview_style');
    const $deviceToggle = $('.preview-device-toggle');
    const $btnDesktop = $('.device-btn.device-desktop');
    const $btnMobile = $('.device-btn.device-mobile');

    // Color pickers
    $('.nba-color').wpColorPicker({
      change: function(){ updatePreview(); },
      clear: function(){ updatePreview(); }
    });

    // Conditional visibility
    function updateScheduleVisibility(){
      const isSched = getDisplay() === 'Scheduled';
      // Toggle entire row (including header) per request
      const $row = $('.field-row[data-group="schedule"]');
      $row.toggleClass('is-hidden', !isSched);
      const $fields = $('.dt-helpers .field-starts, .dt-helpers .field-ends');
      $fields.toggleClass('is-hidden', !isSched);
      $starts.prop('disabled', !isSched);
      $ends.prop('disabled', !isSched);
      // Toggle helper text and timezone hint
      $('.dt-help').toggleClass('is-hidden', !isSched);
      $('.dt-timezone-hint').toggleClass('is-hidden', !isSched);
    }

    // Admin menu icon color to reflect display state
    function parseEtDate(str){
      if (!str) return null;
      // Expected: d/m/Y g:i a
      // Example: 28/11/2025 9:05 am
      const m = str.trim().match(/^([0-3]?\d)\/(1?\d)\/(\d{4})\s+(\d{1,2}):(\d{2})\s*([ap]m)$/i);
      if (!m) return null;
      let [_, d, mo, y, h, mi, ampmFlag] = m;
      d = parseInt(d,10); mo = parseInt(mo,10)-1; y = parseInt(y,10);
      h = parseInt(h,10); mi = parseInt(mi,10);
      const isPm = /pm/i.test(ampmFlag);
      if (h === 12) h = isPm ? 12 : 0; else if (isPm) h += 12;
      const dt = new Date(y, mo, d, h, mi, 0, 0);
      return isNaN(dt.getTime()) ? null : dt;
    }

    function isWithinWindow(now, startStr, endStr){
      const start = parseEtDate(startStr);
      const end = parseEtDate(endStr);
      const afterStart = !start || (now.getTime() >= start.getTime());
      const beforeEnd = !end || (now.getTime() <= end.getTime());
      return afterStart && beforeEnd;
    }

    function updateMenuIconColor(){
      const val = getDisplay();
      let css = '';
      if (val === 'Active') {
        css = '#adminmenu a.menu-top.toplevel_page_noticebar div.wp-menu-image:before{color:#00C3AA !important;}';
      } else if (val === 'Scheduled') {
        const now = new Date();
        if (isWithinWindow(now, $starts.val(), $ends.val())) {
          css = '#adminmenu a.menu-top.toplevel_page_noticebar div.wp-menu-image:before{color:#00C3AA !important;}';
        } else {
          css = '';
        }
      } else {
        css = '';
      }
      let $style = $('#nba_menu_icon_color');
      if (css){
        if ($style.length){
          $style.text(css);
        } else {
          $('<style id="nba_menu_icon_color">'+css+'</style>').appendTo('head');
        }
      } else if ($style.length){
        $style.remove();
      }
    }

    function updateMobileVisibility(){
      const on = $mobileToggle.is(':checked');
      $mobileMsgRow.toggleClass('is-hidden', !on);
      $mobileMsg.prop('disabled', !on);
      // Show device toggle only when using different text on mobile
      $deviceToggle.toggleClass('is-hidden', !on);
    }

    function updateActionFields(){
      const t = $actionType.val();
      $actionLinkRows.toggleClass('is-hidden', t !== 'Link')
        .find('input').prop('disabled', t !== 'Link');
      $actionButtonRows.toggleClass('is-hidden', t !== 'Button')
        .find('input').prop('disabled', t !== 'Button');
      updatePreview();
    }

    function updateCloseVisibility(){
      const on = String($closeEnable.val()||'0') === '1';
      $closeDurationRow.toggleClass('is-hidden', !on);
      if ($closeDuration.length){
        $closeDuration.prop('disabled', !on);
      }
      // Toggle preview close button
      $('#nba_preview_close').toggleClass('is-hidden', !on);
    }

    // Validation helpers
    function validateUrl($input){
      const val = ($input.val()||'').trim();
      const $row = $input.closest('.field-url');
      const $hint = $row.find('.url-hint');
      if (!val){
        $row.removeClass('field-error');
        $input.removeAttr('aria-invalid');
        $hint.text('');
        return true;
      }
      try {
        // Ensure only host/path without scheme in field; disallow spaces
        if (/\s/.test(val)) throw new Error('Invalid');
        // Disallow scheme text in the field (we render prefix separately)
        if (/^https?:\/\//i.test(val)) throw new Error('Include only the domain and path without scheme');
        $row.removeClass('field-error');
        $input.removeAttr('aria-invalid');
        $hint.text('');
        return true;
      } catch(e){
        $row.addClass('field-error');
        $input.attr('aria-invalid', 'true');
        $hint.text('Please enter a domain and path only (no http/https).');
        return false;
      }
    }

    // Clean URL helper for field: strip protocols, www., trailing slash
    function cleanUrlText(str){
      if (!str) return '';
      let v = String(str).trim();
      v = v.replace(/^(https?:\/\/)/i, '');
      v = v.replace(/^www\./i, '');
      v = v.replace(/\/$/, '');
      return v;
    }

    function constructHrefFromField(){
      const base = cleanUrlText($actionUrl.val());
      return base ? ('https://' + base) : '#';
    }

    function updateCounters(){
      $msgCounter.text(($msg.val()||'').length + ' chars');
      $mobileCounter.text(($mobileMsg.val()||'').length + ' chars');
    }

    function fontSizeCss(){
      const v = $fontSize.val();
      if (v === 'S') return 'font-size: 0.875rem;';
      if (v === 'L') return 'font-size: 1.25rem;';
      return 'font-size: 1rem;';
    }

    function backgroundCss(){
      const t = $bgStyle.val();
      if (t === 'Gradient'){
        const a = $bgGradStart.val();
        const b = $bgGradEnd.val();
        const d = clamp(parseInt($bgDirection.val()||'90', 10) || 90, 0, 360);
        return 'background: linear-gradient(' + d + 'deg, ' + a + ', ' + b + ');';
      }
      return 'background-color: ' + ($bgColour.val()||'#1B7AE6') + ';';
    }

    function updatePreview(){
      // Message
      const text = ($msg.val()||'').trim();
      const mobileText = ($mobileMsg.val()||'').trim();
      const useMobile = $mobileToggle.is(':checked') && mobileText.length>0;
      let html = '';
      if (useMobile){
        html = '<span class="mobile-message">' + escapeHtml(mobileText) + '</span>' +
               '<span class="desktop-message">' + escapeHtml(text) + '</span>';
      } else {
        html = escapeHtml(text);
      }
      $prevMsg.html(html);

      // Font + background
      const color = $fontColour.val()||'#ffffff';
      const style = [fontSizeCss(), 'color: ' + color + ';', backgroundCss()].join(' ');
      $prevBar.attr('style', style);

      // Action element
      const t = $actionType.val();
      // Button preview
      const btnTxtVal = ($buttonText.val()||'').trim() || 'Button';
      const unifiedUrl = constructHrefFromField();
      const targetAttr = $actionNewTab.is(':checked') ? '_blank' : '';
      const relAttr = $actionNewTab.is(':checked') ? 'noopener noreferrer' : '';
      $prevBtn.text(btnTxtVal)
        .attr('href', unifiedUrl)
        .attr('target', targetAttr)
        .attr('rel', relAttr)
        .toggleClass('is-hidden', t !== 'Button')
        .css({ color: ($btnFont.val()||''), backgroundColor: ($btnBg.val()||'') });
      // Link preview
      const linkTxtVal = ($linkText.val()||'').trim() || 'Read more';
      $prevLink.text(linkTxtVal)
        .attr('href', unifiedUrl)
        .attr('target', targetAttr)
        .attr('rel', relAttr)
        .toggleClass('is-hidden', t !== 'Link')
        .css({ color: ($linkColour.val()||'') });

      // Hover CSS for button
      const hover = ($btnBgHover.val()||'');
      if (hover){
        $prevStyle.text('.noticebar-preview #nba_preview_button:hover{ background-color: ' + hover + ' !important; }');
      } else {
        $prevStyle.text('');
      }
    }

    // Device toggle handlers
    function setPreviewMode(mode){
      // mode: 'desktop' | 'mobile'
      const isMobile = mode === 'mobile';
      $prevWrap.toggleClass('is-mobile', isMobile).toggleClass('is-desktop', !isMobile);
      $btnDesktop.attr('aria-pressed', (!isMobile).toString());
      $btnMobile.attr('aria-pressed', (isMobile).toString());
    }
    $deviceToggle.on('click', '.device-btn', function(){
      if ($(this).hasClass('device-desktop')) setPreviewMode('desktop');
      else setPreviewMode('mobile');
    });

    function escapeHtml(str){
      return (str||'').replace(/[&<>"']/g, function(c){
        return ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c]);
      });
    }

    // Initialize Flatpickr on schedule inputs (modern datetime picker)
    function normalizeFlatpickrValue(fp, dateStr){
      // flatpickr with K -> AM/PM upper; convert to required d/m/Y g:i a (lowercase am/pm, hour without leading zero)
      if (!dateStr) return '';
      // Expect like 28/11/2025 09:05 AM
      const m = dateStr.match(/^(\d{1,2})\/(\d{1,2})\/(\d{4})\s+(\d{2}):(\d{2})\s*([AP]M)$/);
      if (!m) return dateStr.toLowerCase();
      let [_, d, mo, y, h, mi, ampmUp] = m;
      h = String(parseInt(h,10)); // remove leading zero
      const ampmLow = ampmUp.toLowerCase();
      return d + '/' + mo + '/' + y + ' ' + h + ':' + mi + ' ' + ampmLow;
    }

    function initFlatpickr(){
      if (typeof flatpickr === 'undefined') return;
      const opts = {
        enableTime: true,
        time_24hr: false,
        minuteIncrement: 5,
        allowInput: true,
        dateFormat: 'd/m/Y h:i K', // pretty + AM/PM, we will post-process to lowercase am/pm and remove hour leading zero
        onChange: function(selectedDates, dateStr, instance){
          const $el = $(instance.input);
          const norm = normalizeFlatpickrValue(instance, dateStr);
          $el.val(norm).trigger('change');
          updateMenuIconColor();
        }
      };
      // Starts
      if ($starts.length){
        try {
          flatpickr($starts[0], opts);
        } catch(e) {}
      }
      // Ends
      if ($ends.length){
        try {
          flatpickr($ends[0], opts);
        } catch(e) {}
      }
    }

    // Events
    // Bind changes for radiogroup
    $wrap.on('change', 'input.nba-display-option', function(){ updateScheduleVisibility(); updateMenuIconColor(); });
    // Toggle bar interactions for Close button (ENABLED/DISABLED)
    $wrap.on('click', '.nba-toggle-bar .toggle-option', function(){
      const $btn = $(this);
      const $bar = $btn.closest('.nba-toggle-bar');
      const target = $bar.data('target');
      const val = $btn.data('value') == 1 ? '1' : '0';
      // Update visual state
      $bar.find('.toggle-option').removeClass('is-active').attr('aria-pressed','false');
      $btn.addClass('is-active').attr('aria-pressed','true');
      // Update hidden input
      if (target){
        $(target).val(val).trigger('change');
      }
      updateCloseVisibility();
    });
    $starts.on('input change', updateMenuIconColor);
    $ends.on('input change', updateMenuIconColor);
    $mobileToggle.on('change', function(){ updateMobileVisibility(); updatePreview(); });
    $actionType.on('change', updateActionFields);
    // Live preview updates for action text/url
    $buttonText.on('input change', updatePreview);
    $linkText.on('input change', updatePreview);
    // Unified Action URL + New Tab
    $actionUrl.on('input paste blur', function(){
      const cleaned = cleanUrlText($actionUrl.val());
      if ($actionUrl.val() !== cleaned) {
        $actionUrl.val(cleaned);
      }
      validateUrl($actionUrl);
      updatePreview();
    });
    $actionNewTab.on('change', function(){ updatePreview(); });

    $msg.on('input', function(){ updateCounters(); updatePreview(); });
    $mobileMsg.on('input', function(){ updateCounters(); updatePreview(); });

    // (moved URL validation above to also trigger preview)

    $fontSize.on('change', updatePreview);
    $fontColour.on('change', updatePreview);
    $linkColour.on('change', updatePreview);
    $bgStyle.on('change', function(){ updateBgVisibility(); updatePreview(); });
    $bgGradStart.on('change', updatePreview);
    $bgGradEnd.on('change', updatePreview);
    $bgDirection.on('input change', function(){
      const v = clamp(parseInt($bgDirection.val()||0,10)||0, 0, 360);
      $bgDirection.val(v);
      updatePreview();
    });
    $btnBg.on('change', updatePreview);
    $btnBgHover.on('change', updatePreview);
    $btnFont.on('change', updatePreview);
    $closeEnable.on('change', updateCloseVisibility);

    function updateBgVisibility(){
      const isGrad = $bgStyle.val() === 'Gradient';
      $('.bg-colour').toggleClass('is-hidden', isGrad)
        .find('input').prop('disabled', isGrad);
      $('.bg-gradient').toggleClass('is-hidden', !isGrad)
        .find('input').prop('disabled', !isGrad);
    }

    // Initialize
    // Activate initial tab only after all variables and functions are defined
    (function initTabsAndState(){
      const initial = (location.hash||'').replace('#','');
      if (initial && $tabBtns.filter('[data-tab="'+initial+'"]').length){
        activateTab(initial);
      } else {
        const $active = $tabBtns.filter('[aria-selected="true"]').first();
        activateTab($active.data('tab'));
      }
      // Ensure UI sync on load
      updateScheduleVisibility();
      updateMenuIconColor();
      updateMobileVisibility();
      updateActionFields();
      updateCounters();
      updateBgVisibility();
      updateCloseVisibility();
      updatePreview();
      initFlatpickr();
      // Initial sync to ensure legacy fields reflect unified UI for saving
      syncUnifiedUrlToLegacy();
    })();
  });
})(jQuery);