function debounce(func, wait = 100) {
	let timeout;
	return function (...args) {
		clearTimeout(timeout);
		timeout = setTimeout(() => func.apply(this, args), wait);
	};
}

const OffsetManager = (() => {
	const selectors = new Set([".fixed-top-adjust", ".et_pb_section--fixed"]);

	function register(selector) {
		selectors.add(selector);
	}

	function getAdminBarHeight() {
		const adminBar = document.querySelector("#wpadminbar");
		return (adminBar && adminBar.offsetHeight > 0) ? adminBar.offsetHeight : 0;
	}

	function getNoticeBarHeight(noticeBar, animate) {
		if (!noticeBar || noticeBar.style.display === "none") return 0;

		const expiryKey = noticeBar.dataset.expiryKey;
		const expiry = parseInt(localStorage.getItem(expiryKey));
		const now = Date.now();

		// Read scheduling attributes from the element
		const scheduleEnabled = noticeBar.dataset.scheduleEnabled === "true";
		const scheduleStart = parseInt(noticeBar.dataset.scheduleStart, 10) * 1000; // convert to ms
		const scheduleEnd = parseInt(noticeBar.dataset.scheduleEnd, 10) * 1000; // convert to ms
		
		if (scheduleEnabled && (now < scheduleStart || now > scheduleEnd)) {
			noticeBar.style.display = "none";
			return 0;
		}
		
		if (expiry && now < expiry) {
			noticeBar.style.display = "none";
			return 0;
		}

		const animateType = noticeBar.dataset.animate;
		const delay = parseInt(noticeBar.dataset.animateDelay || 0);

		if (animate && animateType === "slide-down") {
			noticeBar.style.transition = "transform 0.3s ease, opacity 0.3s ease";
			noticeBar.style.transform = "translateY(-100%)";
			noticeBar.style.opacity = "0";
			setTimeout(() => {
				noticeBar.style.transform = "translateY(0)";
				noticeBar.style.opacity = "1";
			}, delay);
		}

		return noticeBar.offsetHeight;
	}

	function applyOffset(el, offset, animate = false) {
		if (!el) return;
		if (animate) el.style.transition = "top 0.3s ease";
		el.style.top = offset + "px";
	}

	function adjustOffsets(animate = false) {
		const isDesktop = window.innerWidth > 782;
		const noticeBar = document.querySelector(".notice-bar");
		const adminBarHeight = getAdminBarHeight();
		const noticeBarHeight = getNoticeBarHeight(noticeBar, animate);
		const offset = isDesktop ? (noticeBarHeight + adminBarHeight) : noticeBarHeight;

		// Set notice bar position
		if (noticeBar && noticeBar.style.display !== "none") {
			noticeBar.style.top = adminBarHeight + "px";
		}

		// Apply offset to all registered selectors
		selectors.forEach(selector => {
			document.querySelectorAll(selector).forEach(el => {
				const computedStyle = window.getComputedStyle(el);
				const isFixed = computedStyle.position === "fixed";
				const topVal = computedStyle.top;

				// If already offset, don't double apply
				const shouldAdjust =
					isFixed && (topVal === "0px" || topVal === "auto" || topVal === `${adminBarHeight}px`);

				if (shouldAdjust || selector === ".fixed-top-adjust") {
					el.classList.add("fixed-top-adjust");
					applyOffset(el, offset, animate);
				}
			});
		});

		// Adjust body margin to preserve layout
		if (noticeBar && noticeBar.style.display !== "none") {
			if (animate) document.body.style.transition = "margin-top 0.3s ease";
			document.body.style.marginTop = noticeBarHeight + "px";
		}
	}

function bindCloseButton() {
	const noticeBar = document.querySelector(".notice-bar");
	const closeBtn = noticeBar?.querySelector(".notice-close");
	if (!noticeBar || !closeBtn) return;

	closeBtn.addEventListener("click", () => {
		const expiryDuration = parseInt(noticeBar.dataset.expiry) || 0;
		if (expiryDuration) {
			const expiryTime = Date.now() + expiryDuration * 1000;
			localStorage.setItem(noticeBar.dataset.expiryKey, expiryTime);
		}

		// Animate the hiding of the notice bar
		noticeBar.style.transform = "translateY(-100%)";
		noticeBar.style.opacity = "0";

		// Wait for the transition to finish before fully hiding it and removing margin
		setTimeout(() => {
			noticeBar.style.display = "none";
			document.body.style.marginTop = "0px";
			OffsetManager.refresh(); // Recalculate layout without the notice bar
		}, 300); // match the transition duration
	});
}

	return {
		init(animate = false) {
			adjustOffsets(animate);
			bindCloseButton();
		},
		refresh() {
			adjustOffsets(false);
		},
		register,
	};
})();

document.addEventListener("DOMContentLoaded", () => OffsetManager.init(true));
window.addEventListener("resize", debounce(() => OffsetManager.refresh()));