<?php
/**
 * Plugin Name: Make Notice Bar
 * Description: Adds a notice bar accordingly.
 * Version: 0.5.4
 * Author: makeweb
 */

// Security: prevent direct access
defined('ABSPATH') || exit;

// Composer autoload (loads makeUpdate and any other dependencies)
include __DIR__ . '/vendor/autoload.php';

// ===== Native settings (ACF removed) =====
// Defaults for settings
function nba_defaults(): array {
    return [
        'display' => [
            'display' => 'Active',
            'starts' => '',
            'ends' => '',
        ],
        'message' => [
            'message' => '',
            'mobile' => false,
            'mobile_message' => '',
        ],
        'close_button' => [
            'close_button' => true,
            'stop_display_duration' => 'None',
        ],
        'font' => [
            'font_size' => 'M',
            'font_colour' => '#ffffff',
            'link_colour' => '#ffffff',
        ],
        'background' => [
            'style' => 'Colour',
            'colour' => '#1B7AE6',
            'gradient_start' => '#1B7AE6',
            'gradient_end' => '#1B7AE6',
            'direction' => 90,
        ],
        'action_element' => [
            'action_type' => '',
            'link_text' => 'Read more',
            'link_link' => '',
            'button_text' => 'Click Here',
            'button_link' => '',
            'new_tab' => false,
        ],
        'button' => [
            'background_colour' => '',
            'background_colour_hover' => '',
            'font_colour' => '',
        ],
        'custom_css' => '',
    ];
}

function nba_get_settings(): array {
    $saved = get_option('noticebar_settings');
    if (!is_array($saved)) $saved = [];
    return wp_parse_args($saved, nba_defaults());
}

function nba_sanitize_bool($v): bool { return (bool)(is_string($v) ? in_array($v, ['1','on','true'], true) : $v); }
function nba_sanitize_enum($v, array $allowed, $default) { return in_array($v, $allowed, true) ? $v : $default; }
function nba_sanitize_hex($v, $fallback = '') { $c = sanitize_hex_color($v); return $c ? $c : $fallback; }

function nba_sanitize_datetime_local($v): string {
    $v = is_string($v) ? trim($v) : '';
    if ($v === '') return '';
    $tz = new DateTimeZone(get_option('timezone_string') ?: 'UTC');
    $fmt = 'd/m/Y g:i a';
    $dt = DateTime::createFromFormat($fmt, $v, $tz);
    return $dt ? $dt->format($fmt) : '';
}

function nba_sanitize_settings(array $in): array {
    $d = nba_defaults();
    $out = $d;

    // Display
    if (isset($in['display']) && is_array($in['display'])) {
        $out['display']['display'] = nba_sanitize_enum($in['display']['display'] ?? $d['display']['display'], ['Active','Scheduled','Disabled'], 'Active');
        $out['display']['starts'] = nba_sanitize_datetime_local($in['display']['starts'] ?? '');
        $out['display']['ends']   = nba_sanitize_datetime_local($in['display']['ends'] ?? '');
    }

    // Message
    if (isset($in['message']) && is_array($in['message'])) {
        $out['message']['message'] = wp_kses_post($in['message']['message'] ?? '');
        $out['message']['mobile']  = nba_sanitize_bool($in['message']['mobile'] ?? false);
        $out['message']['mobile_message'] = wp_kses_post($in['message']['mobile_message'] ?? '');
    }

    // Close button
    if (isset($in['close_button']) && is_array($in['close_button'])) {
        $out['close_button']['close_button'] = nba_sanitize_bool($in['close_button']['close_button'] ?? true);
        $allowed = ['None','30 minutes','1 hour','1 day','1 week','1 month','6 month','1 year','Forever'];
        $out['close_button']['stop_display_duration'] = nba_sanitize_enum($in['close_button']['stop_display_duration'] ?? 'None', $allowed, 'None');
    }

    // Font
    if (isset($in['font']) && is_array($in['font'])) {
        $out['font']['font_size']   = nba_sanitize_enum($in['font']['font_size'] ?? 'M', ['S','M','L'], 'M');
        $out['font']['font_colour'] = nba_sanitize_hex($in['font']['font_colour'] ?? '#ffffff', '#ffffff');
        $out['font']['link_colour'] = nba_sanitize_hex($in['font']['link_colour'] ?? $out['font']['font_colour'], $out['font']['font_colour']);
    }

    // Background
    if (isset($in['background']) && is_array($in['background'])) {
        $out['background']['style'] = nba_sanitize_enum($in['background']['style'] ?? 'Colour', ['Colour','Gradient'], 'Colour');
        $out['background']['colour'] = nba_sanitize_hex($in['background']['colour'] ?? '#1B7AE6', '#1B7AE6');
        $out['background']['gradient_start'] = nba_sanitize_hex($in['background']['gradient_start'] ?? '#1B7AE6', '#1B7AE6');
        $out['background']['gradient_end']   = nba_sanitize_hex($in['background']['gradient_end']   ?? '#1B7AE6', '#1B7AE6');
        $out['background']['direction'] = max(0, min(360, intval($in['background']['direction'] ?? 90)));
    }

    // Action element
    if (isset($in['action_element']) && is_array($in['action_element'])) {
        $ae = $in['action_element'];
        $out['action_element']['action_type'] = nba_sanitize_enum($ae['action_type'] ?? '', ['', 'Link','Button'], '');
        $out['action_element']['link_text']   = sanitize_text_field($ae['link_text'] ?? 'Read more');
        $out['action_element']['button_text'] = sanitize_text_field($ae['button_text'] ?? 'Click Here');
        $out['action_element']['new_tab']     = nba_sanitize_bool($ae['new_tab'] ?? false);

        // New unified field: action_url (host/path only or full). Enforce https:// prefix and sanitize.
        $action_url = '';
        if (isset($ae['action_url'])) {
            $raw = is_string($ae['action_url']) ? trim($ae['action_url']) : '';
            if ($raw !== '') {
                // If it already has a scheme, strip it and `www.`; we will enforce https
                $raw = preg_replace('~^(https?:\/\/)?~i', '', $raw);
                $raw = preg_replace('~^www\.~i', '', $raw);
                $raw = preg_replace('~\/+$~', '', $raw); // single trailing slash
                $action_url = 'https://' . $raw;
            }
        }

        // Fallback to legacy fields if action_url not provided (for older saves)
        if ($action_url === '') {
            $legacy = (!empty($ae['button_link']) ? $ae['button_link'] : (!empty($ae['link_link']) ? $ae['link_link'] : ''));
            $action_url = is_string($legacy) ? trim($legacy) : '';
        }

        // Sanitize final URL
        $action_url = esc_url_raw($action_url);
        // Map to legacy keys so front-end remains unchanged
        $out['action_element']['button_link'] = $action_url;
        $out['action_element']['link_link']   = $action_url;
        // Also store normalized action_url for future admin loads (not used on front-end)
        $out['action_element']['action_url']  = $action_url;
    }

    // Button styles
    if (isset($in['button']) && is_array($in['button'])) {
        $out['button']['background_colour'] = nba_sanitize_hex($in['button']['background_colour'] ?? '', '');
        $out['button']['background_colour_hover'] = nba_sanitize_hex($in['button']['background_colour_hover'] ?? '', '');
        $out['button']['font_colour'] = nba_sanitize_hex($in['button']['font_colour'] ?? '', '');
    }

    // Custom CSS (strip style tags)
    $css = isset($in['custom_css']) ? (string)$in['custom_css'] : '';
    $css = preg_replace('/<style[^>]*>|<\\/style>/i', '', $css);
    $out['custom_css'] = $css;

    return $out;
}

// Admin menu and assets
add_action('admin_menu', function() {
    add_menu_page('Notice Bar', 'Notice Bar', 'manage_options', 'noticebar', 'nba_render_settings_page', 'dashicons-megaphone', 2.1);
});

add_action('admin_enqueue_scripts', function($hook) {
    if ($hook !== 'toplevel_page_noticebar') return;
    wp_enqueue_style('wp-color-picker');
    // Dashicons for preview device toggle icons
    wp_enqueue_style('dashicons');
    // Flatpickr (modern datetime picker)
    wp_enqueue_style('flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css', [], '4.6.13');
    $base_url = plugin_dir_url(__FILE__) . 'admin/';
    $css_path = __DIR__ . '/admin/noticebar.css';
    $js_path  = __DIR__ . '/admin/noticebar.js';
    $css_ver = file_exists($css_path) ? (string) filemtime($css_path) : '0.4.0';
    $js_ver  = file_exists($js_path) ? (string) filemtime($js_path) : '0.4.0';
    wp_enqueue_style('noticebar-admin-style', $base_url . 'noticebar.css', [], $css_ver);
    // Flatpickr JS
    wp_enqueue_script('flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.js', [], '4.6.13', true);
    wp_enqueue_script('noticebar-admin-scripts', $base_url . 'noticebar.js', ['jquery','wp-color-picker','flatpickr'], $js_ver, true);
    // Localize timezone/link for hints
    $tz_string = get_option('timezone_string');
    if (!$tz_string || !is_string($tz_string)) {
        // Fallback from gmt_offset
        $offset = get_option('gmt_offset');
        $tz_string = ($offset === 0 || $offset === '0' || $offset === 0.0) ? 'UTC' : 'UTC' . (floatval($offset) >= 0 ? '+' : '') . strval($offset);
    }
    wp_localize_script('noticebar-admin-scripts', 'NBA_ADMIN', [
        'timezone' => $tz_string,
        'timezone_settings_url' => admin_url('options-general.php'),
    ]);
});

// Compute if the notice bar is effectively active now (Active OR Scheduled and within window)
function nba_is_effectively_active(): bool {
    $s = nba_get_settings();
    $display = $s['display']['display'] ?? 'Active';
    if ($display === 'Disabled') return false;
    if ($display === 'Active') return true;
    if ($display !== 'Scheduled') return false;

    $starts = trim((string)($s['display']['starts'] ?? ''));
    $ends   = trim((string)($s['display']['ends'] ?? ''));

    $tz = new DateTimeZone(get_option('timezone_string') ?: 'UTC');
    $fmt = 'd/m/Y g:i a';
    $now = new DateTime('now', $tz);
    $start_ok = true;
    $end_ok = true;
    if ($starts !== '') {
        $start_dt = DateTime::createFromFormat($fmt, $starts, $tz);
        if ($start_dt instanceof DateTime) {
            $start_ok = ($now->getTimestamp() >= $start_dt->getTimestamp());
        }
    }
    if ($ends !== '') {
        $end_dt = DateTime::createFromFormat($fmt, $ends, $tz);
        if ($end_dt instanceof DateTime) {
            $end_ok = ($now->getTimestamp() <= $end_dt->getTimestamp());
        }
    }
    return ($start_ok && $end_ok);
}

// Admin menu icon colour based on effective active state
add_action('admin_head', function(){
    if (!current_user_can('manage_options')) return;
    if (nba_is_effectively_active()) {
        echo '<style id="nba_menu_icon_color_boot">#adminmenu a.menu-top.toplevel_page_noticebar div.wp-menu-image:before{color:#00C3AA !important;}</style>';
    }
});

function nba_render_settings_page() {
    if (!current_user_can('manage_options')) return;
    $saved = false;
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['nba_settings_nonce']) && wp_verify_nonce($_POST['nba_settings_nonce'], 'nba_save_settings')) {
        $raw = isset($_POST['noticebar_settings']) && is_array($_POST['noticebar_settings']) ? wp_unslash($_POST['noticebar_settings']) : [];
        $san = nba_sanitize_settings($raw);
        update_option('noticebar_settings', $san);
        $saved = true;
        do_action('litespeed_purge_all');
    }
    $s = nba_get_settings();

    if ($saved) {
        echo '<div class="notice notice-success is-dismissible"><p>Notice Bar settings saved.</p></div>';
    }

    echo '<div class="wrap noticebar-admin">';
    echo '<form method="post" novalidate class="noticebar-form">';
    wp_nonce_field('nba_save_settings', 'nba_settings_nonce');
    // Add referer hidden input to mirror requested markup structure
    if (function_exists('wp_referer_field')) {
        wp_referer_field(); // echoes hidden input
    } else {
        $ref = isset($_SERVER['REQUEST_URI']) ? esc_url_raw($_SERVER['REQUEST_URI']) : '';
        echo '<input type="hidden" name="_wp_http_referer" value="' . esc_attr($ref) . '">';
    }

    // Live Preview body (sits by itself above tabs)
    echo '<div class="noticebar-preview-wrap noticebar-preview is-desktop">'
        . '<div class="noticebar-preview-body">'
        . '<div class="notice-bar" id="nba_preview_bar" style="">'
        . '<div class="notice-wrap">'
        . '<span class="notice-message" id="nba_preview_message"></span>'
        . '<a href="#" class="notice-bar-button is-hidden" id="nba_preview_button">Button</a>'
        . '<a href="#" class="notice-bar-link is-hidden" id="nba_preview_link">Read more</a>'
        . '<button type="button" class="notice-close is-hidden" id="nba_preview_close" aria-label="Close Notice">&times;</button>'
        . '</div>'
        . '</div>'
        . '<style id="nba_preview_style"></style>'
        . '</div>'
        . '</div>';

    // Tabs container – preview header sits at the top of this container
    echo '<div class="noticebar-tabs">';

    // Preview header inside tabs container
    // Read plugin version from header for display in preview title
    $plugin_meta = function_exists('get_file_data') ? get_file_data(__FILE__, ['Version' => 'Version']) : ['Version' => ''];
    $plugin_ver = isset($plugin_meta['Version']) ? trim((string)$plugin_meta['Version']) : '';

    echo '<div class="noticebar-preview-head">'
        . '<div class="noticebar-preview-head-inner">'
        . '<strong class="preview-title">Notice bar'
        . (!empty($plugin_ver) ? ' <span class="preview-version">v' . esc_html($plugin_ver) . '</span>' : '')
        . '</strong>'
        . '<div class="preview-actions">'
        . '<div class="preview-device-toggle is-hidden" role="group" aria-label="Preview device">'
        . '<button type="button" class="device-btn device-desktop" aria-pressed="true" title="Desktop" aria-label="Desktop preview"><span class="dashicons dashicons-desktop"></span></button>'
        . '<button type="button" class="device-btn device-mobile" aria-pressed="false" title="Mobile" aria-label="Mobile preview"><span class="dashicons dashicons-smartphone"></span></button>'
        . '</div>';
    // primary save button aligned to the right inside the header bar
    ob_start();
    submit_button('Save Changes', 'primary button-primary-save', 'submit', false, ['id' => 'nba_save_top']);
    $btn = ob_get_clean();
    echo $btn;
    echo '</div></div>';

    // Tabs navigation
    echo '<ul class="tab-nav" role="tablist">'
        . '<li><button type="button" role="tab" aria-selected="false" data-tab="general">General</button></li>'
        . '<li><button type="button" role="tab" aria-selected="true" data-tab="message">Message</button></li>'
        . '<li><button type="button" role="tab" aria-selected="false" data-tab="style">Style</button></li>'
        . '<li><button type="button" role="tab" aria-selected="false" data-tab="action">Action</button></li>'
        . '<li class="custom-tab"><button type="button" role="tab" aria-label="Custom CSS" aria-selected="false" data-tab="custom">'
        . '<span class="dashicons dashicons-shortcode" aria-hidden="true"></span>'
        . '<span class="screen-reader-text">Custom CSS</span>'
        . '</button></li>'
        . '</ul>';

    // Panels start
    echo '<div class="tab-panels">';

    // General panel (hidden by default to match requested sample where Message is active)
    echo '<div class="tab-panel is-hidden" role="tabpanel" data-tab="general">';
    echo '<table class="form-table"><tbody>';

    // Row 1: Display (radio bar only)
    echo '<tr class="field-row" data-group="display">'
        . '<th scope="row"><span>Display</span></th>'
        . '<td>';
    // Radio-style segmented control for Display
    $current_display = $s['display']['display'];
    echo '<div class="nba-radio-bar" role="radiogroup" aria-label="Display options">';
    foreach (['Active','Scheduled','Disabled'] as $opt) {
        $id = 'nba_display_' . strtolower($opt);
        $checked = checked($current_display, $opt, false);
        echo '<input type="radio" class="nba-display-option" id="' . esc_attr($id) . '" name="noticebar_settings[display][display]" value="' . esc_attr($opt) . '" ' . $checked . ' />';
        echo '<label for="' . esc_attr($id) . '">' . esc_html($opt) . '</label>';
    }
    echo '</div>';
    echo '</td></tr>';

    // Row 2: Scheduling (help text + starts/ends/timezone)
    $is_sched = ($s['display']['display'] === 'Scheduled');
    echo '<tr class="field-row' . ($is_sched ? '' : ' is-hidden') . '" data-group="schedule">'
        . '<th scope="row"><span>Scheduling</span></th>'
        . '<td>';
    // Inline help - visible only when Scheduled
    $help_hidden_class = ($is_sched) ? '' : ' is-hidden';
    echo '<span class="description inline-help dt-help' . $help_hidden_class . '">Choose when to display the notice bar.</span>';

    echo '<div class="noticebar-grid dt-helpers" data-controls="schedule">';
    echo '<div class="col-6 field-starts' . ($is_sched ? '' : ' is-hidden') . '" data-field="starts">'
        . '<label class="screen-reader-text" for="nba_starts">Starts</label>'
        . '<input id="nba_starts" type="text" class="regular-text" placeholder="d/m/Y g:i a" name="noticebar_settings[display][starts]" value="' . esc_attr($s['display']['starts']) . '"' . ($is_sched ? '' : ' disabled') . '>'
        . '</div>';
    echo '<div class="col-6 field-ends' . ($is_sched ? '' : ' is-hidden') . '" data-field="ends">'
        . '<label class="screen-reader-text" for="nba_ends">Ends</label>'
        . '<input id="nba_ends" type="text" class="regular-text" placeholder="d/m/Y g:i a" name="noticebar_settings[display][ends]" value="' . esc_attr($s['display']['ends']) . '"' . ($is_sched ? '' : ' disabled') . '>'
        . '</div>';
    // Timezone hint
    $tz_label = get_option('timezone_string');
    if (!$tz_label || !is_string($tz_label)) {
        $offset = get_option('gmt_offset');
        $tz_label = ($offset === 0 || $offset === '0' || $offset === 0.0) ? 'UTC' : 'UTC' . (floatval($offset) >= 0 ? '+' : '') . strval($offset);
    }
    echo '<div class="col-12 dt-timezone-hint' . ($is_sched ? '' : ' is-hidden') . '"><small class="description">Times use the site timezone: <strong>' . esc_html($tz_label) . '</strong>. <a href="' . esc_url(admin_url('options-general.php')) . '" target="_blank" rel="noopener">Change timezone</a>.</small></div>';
    echo '</div>';

    echo '</td></tr>';

    // Row 3: Close button (segmented enable/disable toggle)
    $close_enabled = !empty($s['close_button']['close_button']);
    echo '<tr class="field-row" data-group="close">'
        . '<th scope="row"><span>Close button</span></th>'
        . '<td>'
        . '<div class="nba-toggle-bar" role="group" aria-label="Close button toggle" data-target="#nba_close_enable">'
        . '<button type="button" class="toggle-option' . ($close_enabled ? ' is-active' : '') . '" data-value="1" aria-pressed="' . ($close_enabled ? 'true' : 'false') . '">ENABLED</button>'
        . '<button type="button" class="toggle-option' . (!$close_enabled ? ' is-active' : '') . '" data-value="0" aria-pressed="' . (!$close_enabled ? 'true' : 'false') . '">DISABLED</button>'
        . '</div>'
        . '<input type="hidden" id="nba_close_enable" name="noticebar_settings[close_button][close_button]" value="' . ($close_enabled ? '1' : '0') . '">'
        . '</td>'
        . '</tr>';

    // Row 4: Stay closed duration (only visible if close enabled)
    $durations = ['None','30 minutes','1 hour','1 day','1 week','1 month','6 month','1 year','Forever'];
    $duration_value = $s['close_button']['stop_display_duration'] ?? 'None';
    echo '<tr class="field-row close-duration-row' . ($close_enabled ? '' : ' is-hidden') . '" data-group="close">'
        . '<th scope="row"><label for="nba_close_duration">Stay closed duration</label></th>'
        . '<td>'
        . '<select id="nba_close_duration" name="noticebar_settings[close_button][stop_display_duration]"' . ($close_enabled ? '' : ' disabled') . '>';
    foreach ($durations as $d) {
        printf('<option value="%s" %s>%s</option>', esc_attr($d), selected($duration_value, $d, false), esc_html($d));
    }
    echo '</select>'
        . '</td>'
        . '</tr>';

    echo '</tbody></table>';
    echo '</div>'; // end General panel

    // Message panel
    echo '<div class="tab-panel" role="tabpanel" data-tab="message">';
    echo '<table class="form-table"><tbody>';
    echo '<tr class="field-row" data-group="message"><th scope="row"><label for="nba_message">Message</label></th><td>';
    echo '<p><textarea id="nba_message" rows="3" class="large-text" name="noticebar_settings[message][message]" aria-describedby="nba_message_counter">' . esc_textarea($s['message']['message']) . '</textarea>';
    echo ' <span id="nba_message_counter" class="counter" aria-live="polite"></span></p>';
    echo '<label><input id="nba_mobile_toggle" type="checkbox" name="noticebar_settings[message][mobile]" ' . checked($s['message']['mobile'], true, false) . '> Use different text on mobile</label>';
    echo '<p class="mobile-text field-mobile-message"><label class="screen-reader-text" for="nba_mobile_message">Mobile message</label>';
    echo '<textarea id="nba_mobile_message" rows="2" class="large-text" name="noticebar_settings[message][mobile_message]" aria-describedby="nba_mobile_message_counter">' . esc_textarea($s['message']['mobile_message']) . '</textarea> ';
    echo '<span id="nba_mobile_message_counter" class="counter" aria-live="polite"></span></p>';
    echo '</td></tr>';
    echo '</tbody></table>';
    echo '</div>'; // end Message panel

    // Close Button panel removed (fields moved into General tab)

    // Style panel (combines Font and Background)
    echo '<div class="tab-panel is-hidden" role="tabpanel" data-tab="style">';
    echo '<table class="form-table"><tbody>';
    // Font Size row
    echo '<tr class="field-row" data-group="font"><th scope="row"><label for="nba_font_size">Font Size</label></th><td>';
    echo '<select id="nba_font_size" name="noticebar_settings[font][font_size]">';
    foreach (['S'=>'Small','M'=>'Medium','L'=>'Large'] as $val=>$label) printf('<option value="%s" %s>%s</option>', esc_attr($val), selected($s['font']['font_size'], $val, false), esc_html($label));
    echo '</select>';
    echo '</td></tr>';
    // Font Colour row
    echo '<tr class="field-row" data-group="font"><th scope="row"><label for="nba_font_colour">Font Colour</label></th><td>';
    echo '<input id="nba_font_colour" type="text" class="nba-color" name="noticebar_settings[font][font_colour]" value="' . esc_attr($s['font']['font_colour']) . '">';
    echo '</td></tr>';
    // Link Colour row
    echo '<tr class="field-row" data-group="font"><th scope="row"><label for="nba_link_colour">Link Colour</label></th><td>';
    echo '<input id="nba_link_colour" type="text" class="nba-color" name="noticebar_settings[font][link_colour]" value="' . esc_attr($s['font']['link_colour']) . '">';
    echo '</td></tr>';
    // Background Style row
    echo '<tr class="field-row" data-group="background"><th scope="row"><label for="nba_bg_style">Background Style</label></th><td>';
    echo '<select id="nba_bg_style" name="noticebar_settings[background][style]"><option value="Colour" ' . selected($s['background']['style'],'Colour', false) . '>Colour</option><option value="Gradient" ' . selected($s['background']['style'],'Gradient', false) . '>Gradient</option></select>';
    echo '</td></tr>';
    // Background Colour row (colour only)
    echo '<tr class="field-row bg-colour-row bg-colour" data-group="background"><th scope="row"><label for="nba_bg_colour">Background Colour</label></th><td>';
    echo '<input id="nba_bg_colour" type="text" class="nba-color" name="noticebar_settings[background][colour]" value="' . esc_attr($s['background']['colour']) . '">';
    echo '</td></tr>';
    // Gradient Start row
    echo '<tr class="field-row bg-gradient-row bg-gradient" data-group="background"><th scope="row"><label for="nba_bg_grad_start">Gradient Start</label></th><td>';
    echo '<input id="nba_bg_grad_start" type="text" class="nba-color" name="noticebar_settings[background][gradient_start]" value="' . esc_attr($s['background']['gradient_start']) . '">';
    echo '</td></tr>';
    // Gradient End row
    echo '<tr class="field-row bg-gradient-row bg-gradient" data-group="background"><th scope="row"><label for="nba_bg_grad_end">Gradient End</label></th><td>';
    echo '<input id="nba_bg_grad_end" type="text" class="nba-color" name="noticebar_settings[background][gradient_end]" value="' . esc_attr($s['background']['gradient_end']) . '">';
    echo '</td></tr>';
    // Gradient Direction row
    echo '<tr class="field-row bg-gradient-row bg-gradient" data-group="background"><th scope="row"><label for="nba_bg_direction">Gradient Direction</label></th><td>';
    echo '<input id="nba_bg_direction" type="number" min="0" max="360" name="noticebar_settings[background][direction]" value="' . esc_attr($s['background']['direction']) . '">';
    echo '</td></tr>';
    echo '</tbody></table>';
    echo '</div>'; // end Style panel

    // Action panel
    echo '<div class="tab-panel is-hidden" role="tabpanel" data-tab="action">';
    echo '<table class="form-table"><tbody>';
    // Action Type row
    echo '<tr class="field-row" data-group="action"><th scope="row"><label for="nba_action_type">Action Type</label></th><td>';
    echo '<select id="nba_action_type" name="noticebar_settings[action_element][action_type]"><option value="" ' . selected($s['action_element']['action_type'],'', false) . '>None</option><option value="Link" ' . selected($s['action_element']['action_type'],'Link', false) . '>Link</option><option value="Button" ' . selected($s['action_element']['action_type'],'Button', false) . '>Button</option></select>';
    echo '</td></tr>';

    // Link Text row
    echo '<tr class="field-row action-link-row" data-group="action"><th scope="row"><label for="nba_link_text">Link Text</label></th><td>';
    echo '<input id="nba_link_text" type="text" class="regular-text" name="noticebar_settings[action_element][link_text]" value="' . esc_attr($s['action_element']['link_text']) . '">';
    echo '</td></tr>';

    // Button Text row
    echo '<tr class="field-row action-button-row" data-group="action"><th scope="row"><label for="nba_button_text">Button Text</label></th><td>';
    echo '<input id="nba_button_text" type="text" class="regular-text" name="noticebar_settings[action_element][button_text]" value="' . esc_attr($s['action_element']['button_text']) . '">';
    echo '</td></tr>';
    // Button Styles rows
    echo '<tr class="field-row action-button-row" data-group="action"><th scope="row"><label for="nba_btn_bg">Button Background</label></th><td>';
    echo '<input id="nba_btn_bg" type="text" class="nba-color" name="noticebar_settings[button][background_colour]" value="' . esc_attr($s['button']['background_colour']) . '">';
    echo '</td></tr>';
    echo '<tr class="field-row action-button-row" data-group="action"><th scope="row"><label for="nba_btn_bg_hover">Button Hover Background</label></th><td>';
    echo '<input id="nba_btn_bg_hover" type="text" class="nba-color" name="noticebar_settings[button][background_colour_hover]" value="' . esc_attr($s['button']['background_colour_hover']) . '">';
    echo '</td></tr>';
    echo '<tr class="field-row action-button-row" data-group="action"><th scope="row"><label for="nba_btn_font">Button Font Colour</label></th><td>';
    echo '<input id="nba_btn_font" type="text" class="nba-color" name="noticebar_settings[button][font_colour]" value="' . esc_attr($s['button']['font_colour']) . '">';
    echo '</td></tr>';

    // Unified Action URL + New Tab (integrated UI)
    $initial_url_full = (string)($s['action_element']['action_url'] ?? '');
    if ($initial_url_full === '') {
        $initial_url_full = (string)($s['action_element']['button_link'] ?? '') ?: (string)($s['action_element']['link_link'] ?? '');
    }
    // Display without scheme and www.
    $display_url = $initial_url_full;
    if ($display_url !== '') {
        $display_url = preg_replace('~^(https?:\/\/)?~i', '', $display_url);
        $display_url = preg_replace('~^www\.~i', '', $display_url);
        $display_url = preg_replace('~\/$~', '', $display_url);
    }
    $new_tab_checked = !empty($s['action_element']['new_tab']);
    echo '<tr class="field-row action-url-row" data-group="action"><th scope="row"><label for="nba_action_url">Action URL</label></th><td class="field-url">';
    echo '<div class="link-field-container">';
    echo '  <div class="input-group">';
    echo '    <span class="prefix">https://</span>';
    echo '    <input type="text" id="nba_action_url" class="url-input" name="noticebar_settings[action_element][action_url]" value="' . esc_attr($display_url) . '" placeholder="example.com/page">';
    echo '    <div class="integrated-toggle-wrapper">';
    echo '      <span class="toggle-label-text">Open in new tab</span>';
    echo '      <label class="switch">';
    echo '        <input type="checkbox" id="nba_action_new_tab" name="noticebar_settings[action_element][new_tab]" ' . checked($new_tab_checked, true, false) . ' />';
    echo '        <span class="slider round"></span>';
    echo '      </label>';
    echo '    </div>';
    echo '  </div>';
    echo '  <div class="inline-help url-hint" aria-live="polite"></div>';
    echo '</div>';
    echo '</td></tr>';

    echo '</tbody></table>';
    echo '</div>'; // end Action panel

    // (Button Styles panel removed; fields moved under Action and shown conditionally)

    // Custom CSS panel
    echo '<div class="tab-panel is-hidden" role="tabpanel" data-tab="custom">';
    echo '<table class="form-table"><tbody>';
    echo '<tr class="field-row" data-group="custom"><th scope="row"><label for="nba_custom_css">Custom CSS</label></th><td>';
    echo '<p><textarea id="nba_custom_css" rows="5" class="large-text code" name="noticebar_settings[custom_css]">' . esc_textarea($s['custom_css']) . '</textarea></p>';
    echo '</td></tr>';
    echo '</tbody></table>';
    echo '</div>'; // end Custom CSS panel

    echo '</div>'; // end tab-panels
    echo '</div>'; // end noticebar-tabs

    echo '</form></div>';
}

// One-time migration from ACF (if values exist and option empty)
add_action('admin_init', function() {
    $has = get_option('noticebar_settings');
    $migrated = get_option('noticebar_migrated_acf');
    if (!is_array($has) && function_exists('get_field') && !$migrated) {
        $data = [
            'message' => (array) get_field('message', 'noticebar_settings'),
            'close_button' => (array) get_field('close_button', 'noticebar_settings'),
            'font' => (array) get_field('font', 'noticebar_settings'),
            'background' => (array) get_field('background', 'noticebar_settings'),
            'display' => (array) get_field('Display', 'noticebar_settings'),
            'action_element' => (array) get_field('action_element', 'noticebar_settings'),
            'button' => (array) get_field('button', 'noticebar_settings'),
            'custom_css' => (string) get_field('custom_css', 'noticebar_settings'),
        ];
        $san = nba_sanitize_settings($data);
        update_option('noticebar_settings', $san);
        update_option('noticebar_migrated_acf', 1);
        add_action('admin_notices', function(){
            echo '<div class="notice notice-success is-dismissible"><p>Notice Bar settings migrated from ACF.</p></div>';
        });
    }
});

add_action('wp_enqueue_scripts', function () {
    wp_register_script('notice-bar', 'data:text/javascript,', [], null, true);
    wp_enqueue_script('notice-bar');

    ob_start();
    include plugin_dir_path(__FILE__) . 'notice-bar.js';
    $script = ob_get_clean();

    wp_add_inline_script('notice-bar', $script);
    wp_enqueue_style('notice-bar', plugin_dir_url(__FILE__) . 'notice-bar.css');
});

add_action('wp_footer', 'nba_add_notice_bar');
function nba_add_notice_bar() {
    $s = nba_get_settings();
    $message_group = isset($s['message']) ? $s['message'] : [];
    $desktop_message = strip_wrapping_p($message_group['message'] ?? '');
    $mobile_message = strip_wrapping_p($message_group['mobile_message'] ?? '');
    $use_mobile = !empty($message_group['mobile']);

    $close_button_settings = isset($s['close_button']) ? $s['close_button'] : [];
    $enable_close_button = !empty($close_button_settings['close_button']);
    $expiry_label = $close_button_settings['stop_display_duration'] ?? 'None';

    $font_group = isset($s['font']) ? $s['font'] : [];
    $background_group = isset($s['background']) ? $s['background'] : [];
    $schedule_group = isset($s['display']) ? $s['display'] : [];
    $action_element = isset($s['action_element']) ? $s['action_element'] : [];
    $button_styles = isset($s['button']) ? $s['button'] : [];

    if (($schedule_group['display'] ?? '') === 'Disabled') return;

    $schedule_enabled = ($schedule_group['display'] ?? '') === 'Scheduled';
    $start_timestamp = $end_timestamp = 0;

    if ($schedule_enabled) {
        $tz = new DateTimeZone(get_option('timezone_string') ?: 'UTC');
        $acf_format = 'd/m/Y g:i a';
        $start_time_obj = !empty($schedule_group['starts']) ? DateTime::createFromFormat($acf_format, $schedule_group['starts'], $tz) : false;
        $end_time_obj = !empty($schedule_group['ends']) ? DateTime::createFromFormat($acf_format, $schedule_group['ends'], $tz) : false;
        $start_timestamp = $start_time_obj ? $start_time_obj->getTimestamp() : 0;
        $end_timestamp   = $end_time_obj ? $end_time_obj->getTimestamp() : 0;
    }

    if (!$desktop_message && (!$use_mobile || !$mobile_message)) return;

    $expiry_map = [
        'None' => 0, '30 minutes' => 1800, '1 hour' => 3600, '1 day' => 86400,
        '1 week' => 604800, '1 month' => 2628000, '6 month' => 15724800, '1 year' => 31536000, 'Forever' => 999999999,
    ];
    $expiry_seconds = $expiry_map[$expiry_label] ?? 0;
    $notice_id = substr(md5(strip_tags($desktop_message . $mobile_message)), 0, 8);
    $expiry_key = 'notice_bar_dismissed_' . $notice_id;

    $font_size = $font_group['font_size'] ?? 'M';
    switch ($font_size) {
        case 'S':
            $font_size_css = 'font-size: 0.875rem;';
            break;
        case 'L':
            $font_size_css = 'font-size: 1.25rem;';
            break;
        default:
            $font_size_css = 'font-size: 1rem;';
            break;
    }
    $font_color = sanitize_hex_color($font_group['font_colour'] ?? '#fff');
    $font_color_css = "color: {$font_color};";

    $bg_style = $background_group['style'] ?? 'Colour';
    $background_css = '';
    if ($bg_style === 'Colour' && !empty($background_group['colour'])) {
        $background_css = 'background-color: ' . sanitize_hex_color($background_group['colour']) . ';';
    } elseif ($bg_style === 'Gradient') {
        $start = sanitize_hex_color($background_group['gradient_start'] ?? '#1B7AE6');
        $end = sanitize_hex_color($background_group['gradient_end'] ?? '#1B7AE6');
        $direction = intval($background_group['direction'] ?? 90);
        $background_css = "background: linear-gradient({$direction}deg, {$start}, {$end});";
    }

    $inline_styles = esc_attr(trim("{$font_size_css} {$font_color_css} {$background_css}"));
    $action_element_html = $hover_css = '';
    $action_type = $action_element['action_type'] ?? '';

    if ($action_type === 'Button') {
        [$action_element_html, $hover_css] = nba_generate_action_button($action_element, $button_styles);
    }

    $link_html = '';
    if ($action_type === 'Link') {
        $link_html = nba_generate_action_link($action_element);
    }

    if ($use_mobile && $mobile_message) {
        if ($link_html) {
            $mobile_message  .= $link_html;
            $desktop_message .= $link_html;
        }
        $message_content = sprintf(
            '<span class="mobile-message">%s</span><span class="desktop-message">%s</span>',
            wp_kses_post($mobile_message),
            wp_kses_post($desktop_message)
        );
    } else {
        if ($link_html) {
            $desktop_message .= $link_html;
        }
        $message_content = wp_kses_post($desktop_message);
    }

    $close_button_html = $enable_close_button ? '<button class="notice-close" aria-label="Close Notice">&times;</button>' : '';

    printf(
        '<div class="notice-bar" data-animate="%s" data-animate-delay="%d" data-expiry="%d" data-expiry-key="%s" data-schedule-enabled="%s" data-schedule-start="%s" data-schedule-end="%s" style="%s"> <div class="notice-wrap"><span class="notice-message">%s</span>%s%s </div></div>',
        'slide-down', 0, esc_attr($expiry_seconds), esc_attr($expiry_key),
        esc_attr($schedule_enabled ? 'true' : 'false'), esc_attr($start_timestamp), esc_attr($end_timestamp),
        $inline_styles, $message_content, $action_element_html, $close_button_html
    );

    $custom_css = isset($s['custom_css']) ? (string)$s['custom_css'] : '';
    $custom_css = preg_replace('/<style[^>]*>|<\\/style>/i', '', $custom_css); // remove existing tags

    if (!empty($hover_css) || !empty($custom_css)) {
        echo '<style class="noticebar_custom_css">' . $hover_css . $custom_css . '</style>';
    }
}

function nba_generate_action_button(array $action_element, array $button_styles): array {
    $button_text = esc_html($action_element['button_text'] ?? 'Click Here');
    $button_link = esc_url($action_element['button_link'] ?? '#');
    $new_tab_attr = !empty($action_element['new_tab']) ? 'target="_blank" rel="noopener noreferrer"' : '';

    $bg_color        = $button_styles['background_colour'] ?? '';
    $bg_color_hover  = $button_styles['background_colour_hover'] ?? '';
    $font_color      = $button_styles['font_colour'] ?? '';

    $style_parts = [];
    if ($bg_color) $style_parts[] = 'background-color: ' . esc_attr($bg_color);
    if ($font_color) $style_parts[] = 'color: ' . esc_attr($font_color);
    $inline_style = $style_parts ? 'style="' . implode('; ', $style_parts) . '"' : '';

    $attributes = trim(implode(' ', array_filter([$new_tab_attr, $inline_style])));

    $button_html = sprintf(
        '<a href="%s" class="notice-bar-button" %s>%s</a>',
        $button_link,
        $attributes,
        $button_text
    );
    $hover_css = '';
    if ($bg_color_hover) {
        $hover_css = sprintf(
            '.notice-bar-button:hover { background-color: %s !important; }',
            esc_attr($bg_color_hover)
        );
    }
    return [$button_html, $hover_css];
}

function nba_generate_action_link(array $action_element): string {
    $link_text = esc_html($action_element['link_text'] ?? 'Read more');
    $link_link = esc_url($action_element['link_link'] ?? '#');
    $new_tab = !empty($action_element['new_tab']) ? ' target="_blank" rel="noopener noreferrer"' : '';

    $font_styles = nba_get_settings()['font'];
    $link_colour = $font_styles['link_colour'] ?? '';
    $link_style = $link_colour ? ' style="color: ' . esc_attr($link_colour) . ';"' : '';

    return sprintf(
        ' <a href="%s" class="notice-bar-link"%s%s>%s</a>',
        $link_link,
        $new_tab,
        $link_style,
        $link_text
    );
}

function strip_wrapping_p($content) {
    $content = trim($content);
    if (preg_match('/^<p>(.*?)<\/p>$/s', $content, $matches)) {
        return $matches[1];
    }
    return $content;
}

// (ACF admin enqueue removed; using admin_enqueue_scripts above)


////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if (class_exists('makeUpdate')) {
    error_log('makeUpdate');
    try {
        $updater = new makeUpdate("9553", "ftz_mwv4prd8WTZ.gpz", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}else{
    error_log('makeUpdaten');
}