<?php
/**
 * Plugin Name: Make Testimonials (CPT + Shortcodes + Divi)
 * Description: Adds a beginner-friendly Testimonials custom post type with categories, simple custom fields, and Grid/Slider shortcodes + Divi modules.
 * Version: 1.0.34
 * Author: MakeWeb
 * Author URI: https://makeweb.com.au
 * Text Domain: wp-testimonials
 */

if (!defined('ABSPATH')) { exit; }

// Define constants
if (!defined('WPTEST_BASE')) {
    define('WPTEST_BASE', plugin_basename(__FILE__));
}
if (!defined('WPTEST_DIR')) {
    define('WPTEST_DIR', plugin_dir_path(__FILE__));
}
if (!defined('WPTEST_URL')) {
    define('WPTEST_URL', plugin_dir_url(__FILE__));
}

class WP_Testimonials_Plugin {
    public function __construct()
    {
        // Register CPT/tax very early on init so Divi’s builder render (AJAX) sees the post type
        add_action('init', [$this, 'register_cpt_tax'], 0);
        // Register post meta early as well
        add_action('init', [$this, 'register_post_meta'], 0);
        add_action('wp_ajax_wpt_toggle_featured', [$this, 'ajax_toggle_featured']);
        add_action('add_meta_boxes', [$this, 'register_metabox']);
        add_action('save_post', [$this, 'save_metabox']);

        add_action('wp', [$this, 'disable_single']);

        add_action('admin_enqueue_scripts', [$this, 'admin_assets']);
        add_action('wp_enqueue_scripts', [$this, 'register_front_assets']);

        add_shortcode('testimonials', [$this, 'shortcode']);
        add_shortcode('testimonials_grid', [$this, 'shortcode_grid']);
        add_shortcode('testimonials_slider', [$this, 'shortcode_slider']);

        // Admin list columns
        add_filter('manage_edit-testimonial_columns', [$this, 'columns']);
        add_action('manage_testimonial_posts_custom_column', [$this, 'column_content'], 10, 2);

        // Divi modules are loaded using includes on the et_builder_ready hook (declared outside the class).
    }

    public function register_cpt_tax() {
        $labels = [
            'name' => __('Testimonials', 'wp-testimonials'),
            'singular_name' => __('Testimonial', 'wp-testimonials'),
            'add_new_item' => __('Add New Testimonial', 'wp-testimonials'),
            'edit_item' => __('Edit Testimonial', 'wp-testimonials'),
            'new_item' => __('New Testimonial', 'wp-testimonials'),
            'view_item' => __('View Testimonial', 'wp-testimonials'),
            'search_items' => __('Search Testimonials', 'wp-testimonials'),
            'not_found' => __('No testimonials found', 'wp-testimonials'),
            'menu_name' => __('Testimonials', 'wp-testimonials'),
        ];
        $args = [
            'labels' => $labels,
            'public' => true,
            'publicly_queryable' => false, // no single
            'exclude_from_search' => true,
            'show_ui' => true,
            'show_in_menu' => true,
            'menu_icon' => 'dashicons-format-chat',
            'supports' => ['title'],
            'has_archive' => false,
            'rewrite' => false,
            'show_in_rest' => true,
        ];
        register_post_type('testimonial', $args);

        // Categories taxonomy
        $tax_labels = [
            'name' => __('Testimonial Categories', 'wp-testimonials'),
            'singular_name' => __('Testimonial Category', 'wp-testimonials'),
        ];
        register_taxonomy('testimonial_category', ['testimonial'], [
            'hierarchical' => true,
            'labels' => $tax_labels,
            'show_ui' => true,
            'show_admin_column' => true,
            'query_var' => true,
            'rewrite' => ['slug' => 'testimonial-category'],
            'show_in_rest' => true,
        ]);
    }

    public function register_post_meta() {
        $meta_args = [
            'single' => true,
            'type' => 'string',
            'show_in_rest' => true,
            'auth_callback' => function() { return current_user_can('edit_posts'); }
        ];
        register_post_meta('testimonial', '_wpt_text', $meta_args);
        register_post_meta('testimonial', '_wpt_author_name', $meta_args);
        register_post_meta('testimonial', '_wpt_author_role', $meta_args);
        register_post_meta('testimonial', '_wpt_website', $meta_args);
        register_post_meta('testimonial', '_wpt_website_target', $meta_args);
        register_post_meta('testimonial', '_wpt_website_rel', $meta_args);
        register_post_meta('testimonial', '_wpt_author_photo_id', array_merge($meta_args, ['type' => 'integer']));
        register_post_meta('testimonial', '_wpt_rating', array_merge($meta_args, ['type' => 'number']));
        register_post_meta('testimonial', '_wpt_featured', array_merge($meta_args, ['type' => 'boolean']));
    }

    public function register_metabox() {
        add_meta_box('wpt_details', __('Testimonial Details', 'wp-testimonials'), [$this, 'metabox_html'], 'testimonial', 'normal', 'high');
    }

    private function get_meta($post_id) {
        return [
            'text' => get_post_meta($post_id, '_wpt_text', true),
            'author_name' => get_post_meta($post_id, '_wpt_author_name', true),
            'author_role' => get_post_meta($post_id, '_wpt_author_role', true),
            'website' => get_post_meta($post_id, '_wpt_website', true),
            'website_target' => get_post_meta($post_id, '_wpt_website_target', true),
            'website_rel' => get_post_meta($post_id, '_wpt_website_rel', true),
            'author_photo_id' => (int) get_post_meta($post_id, '_wpt_author_photo_id', true),
            'rating' => (float) get_post_meta($post_id, '_wpt_rating', true),
            'featured' => (bool) get_post_meta($post_id, '_wpt_featured', true),
        ];
    }

    public function metabox_html($post) {
        wp_nonce_field('wpt_save', 'wpt_nonce');
        $m = $this->get_meta($post->ID);
        $photo_url = '';
        if (!empty($m['author_photo_id'])) {
            $src = wp_get_attachment_image_src($m['author_photo_id'], 'thumbnail');
            if ($src && !empty($src[0])) { $photo_url = esc_url($src[0]); }
        }
        ?>
        <style>
            /* Layout mimicking front-end: bubble on top, author row below */
            .wpt-admin-card { border:1px solid #e2e4e7; background:#fff; border-radius:8px; padding:16px; }
            .wpt-admin-bubble label { font-weight:600; display:block; margin:0 0 6px; }
            .wpt-admin-author { display:flex; align-items:center; gap:12px; margin-top:16px; }
            .wpt-photo-picker { width:64px; height:64px; border-radius:50%; background:#f3f4f6; border:2px dashed #c3c4c7; display:flex; align-items:center; justify-content:center; cursor:pointer; overflow:hidden; position:relative; }
            .wpt-photo-picker img { width:100%; height:100%; object-fit:cover; border-radius:50%; }
            .wpt-photo-picker .wpt-plus { font-size:22px; color:#7a7a7a; }
            .wpt-author-fields { flex:1; display:grid; grid-template-columns: 1fr auto; gap:8px 10px; align-items:center; }
            .wpt-author-fields .widefat { width:100%; }
            .wpt-row { margin-top:12px; display:flex; gap:12px; align-items:center; }
            .wpt-star-picker { display:inline-flex; gap:4px; cursor:pointer; user-select:none; }
            .wpt-star-picker .star { position:relative; font-size:20px; line-height:1; color:#ffb400; opacity:.3; transition:opacity .15s ease-in-out; }
            .wpt-star-picker .star.fill { opacity:1; }
            .wpt-star-picker .star.half { opacity:1; }
            .wpt-star-picker .star.half::before { content:'★'; position:absolute; left:0; top:0; width:0.5em; overflow:hidden; color:#ffb400; opacity:1; }
            /* Link icon button inside Role/Company field */
            .wpt-link-btn { appearance:none; -webkit-appearance:none; background:transparent; border:none; padding:4px; margin:0; cursor:pointer; color:#8c8f94; display:flex; align-items:center; justify-content:center; border-radius:6px; text-decoration:none; }
            .wpt-link-btn:hover { background:#f0f0f1; }
            .wpt-link-btn:focus { outline:2px solid #2271b1; outline-offset:2px; }
            .wpt-link-btn .dashicons { font-size:18px; width:18px; height:18px; line-height:18px; }
            .wpt-link-btn.has-link { color:#2271b1; }
            /* Help text */
            .wpt-help { color:#666; font-size:12px; margin-top:6px; }
            /* Modern popover for website editing */
            .wpt-pop { position:relative; }
            .wpt-pop .wpt-pop-panel { position:absolute; top:100%; left:0; z-index:1000; background:#fff; border:1px solid #e5e7eb; padding:14px; border-radius:12px; box-shadow:0 12px 35px rgba(0,0,0,.14); margin-top:8px; width:min(460px, 92vw); display:none; }
            .wpt-pop .wpt-pop-panel:before{content:""; position:absolute; top:-6px; left:36px; width:12px; height:12px; background:#fff; border-left:1px solid #e5e7eb; border-top:1px solid #e5e7eb; transform:rotate(45deg)}
            .wpt-pop .wpt-pop-head{ font-weight:600; display:flex; align-items:center; gap:8px; margin-bottom:10px; }
            .wpt-pop .wpt-pop-label{ font-size:12px; font-weight:600; color:#4b5563; display:block; margin:8px 0 4px; }
            .wpt-pop .wpt-pop-row{ display:flex; gap:12px; margin-top:6px; }
            .wpt-pop .wpt-pop-col{ flex:1; min-width:0; }
            .wpt-pop .wpt-pop-check{ display:inline-flex; align-items:center; gap:6px; margin:4px 10px 0 0; }
            .wpt-pop .wpt-pop-actions{ display:flex; justify-content:space-between; align-items:center; gap:10px; margin-top:12px; }
            .wpt-pop .wpt-pop-panel input[type=url],
            .wpt-pop .wpt-pop-panel select{ width:100%; }
            .wpt-pop.open .wpt-pop-panel { display:block; }
            /* Rating row: show Clear button on hover only */
            .wpt-rating { position:relative; }
            .wpt-rating-clear { opacity:0; transition:opacity .15s ease-in-out; pointer-events:none; }
            .wpt-rating:hover .wpt-rating-clear { opacity:1; pointer-events:auto; }
            .wpt-inline-hide { display:none !important; }
        </style>

        <div class="wpt-admin-card">
            <div class="wpt-admin-bubble">
                <label for="wpt_text"><?php _e('Testimonial Text', 'wp-testimonials'); ?></label>
                <?php
                $settings = [
                    'textarea_name' => 'wpt_text',
                    'media_buttons' => false,
                    'teeny' => true,
                    'drag_drop_upload' => false,
                    'quicktags' => false,
                    'textarea_rows' => 5,
                ];
                wp_editor($m['text'], 'wpt_text_editor', $settings);
                ?>
                <p class="wpt-help"><?php _e('Keep it short and clear. Formatting is simplified for consistency.', 'wp-testimonials'); ?></p>
            </div>

            <div class="wpt-admin-author">
                <div class="wpt-photo-picker" id="wpt_photo_picker" title="<?php esc_attr_e('Click to add/change photo', 'wp-testimonials'); ?>">
                    <?php if ($photo_url): ?>
                        <img src="<?php echo $photo_url; ?>" alt="" />
                    <?php else: ?>
                        <span class="wpt-plus">+</span>
                    <?php endif; ?>
                </div>
                <input type="hidden" id="wpt_author_photo_id" name="wpt_author_photo_id" value="<?php echo (int)$m['author_photo_id']; ?>" />

                <div class="wpt-author-fields">
                    <div style="grid-column:1/-1;">
                        <label for="wpt_author_name" style="font-weight:600; display:block; margin-bottom:4px;"><?php _e('Person’s Name', 'wp-testimonials'); ?></label>
                        <input type="text" id="wpt_author_name" name="wpt_author_name" class="widefat" value="<?php echo esc_attr($m['author_name']); ?>" placeholder="<?php esc_attr_e('Person’s Name', 'wp-testimonials'); ?>" />
                    </div>

                    <div class="wpt-pop" id="wpt_role_wrap" style="grid-column:1/-1; position:relative;">
                        <label for="wpt_author_role" style="font-weight:600; display:block; margin-bottom:4px;"><?php _e('Role / Company', 'wp-testimonials'); ?></label>
                        <div style="position:relative;">
                            <input type="text" id="wpt_author_role" name="wpt_author_role" class="widefat" value="<?php echo esc_attr($m['author_role']); ?>" placeholder="<?php esc_attr_e('Role / Company', 'wp-testimonials'); ?>" />
                            <?php $has_link = !empty($m['website']); ?>
                            <button type="button" id="wpt_link_toggle" class="wpt-link-btn<?php echo $has_link ? ' has-link' : ''; ?>" title="<?php esc_attr_e('Add/edit website link', 'wp-testimonials'); ?>" style="position:absolute; right:8px; top:50%; transform:translateY(-50%);">
                                <span class="dashicons dashicons-admin-links"></span>
                            </button>
                        </div>
                        <div class="wpt-pop-panel" id="wpt_link_pop">
                            <div class="wpt-pop-head"><span class="dashicons dashicons-admin-links"></span> <?php _e('Website link', 'wp-testimonials'); ?></div>
                            <label for="wpt_website" class="wpt-pop-label"><?php _e('URL', 'wp-testimonials'); ?></label>
                            <input type="url" id="wpt_website" name="wpt_website" value="<?php echo esc_url($m['website']); ?>" placeholder="https://example.com" />
                            <div class="wpt-pop-row">
                                <div class="wpt-pop-col">
                                    <label for="wpt_website_target" class="wpt-pop-label"><?php _e('Target', 'wp-testimonials'); ?></label>
                                    <?php $tgt = $m['website_target'] ?: '_blank'; ?>
                                    <select id="wpt_website_target" name="wpt_website_target" class="widefat">
                                        <option value="_blank" <?php selected($tgt,'_blank'); ?>><?php _e('New tab (_blank)','wp-testimonials'); ?></option>
                                        <option value="_self" <?php selected($tgt,'_self'); ?>><?php _e('Same tab (_self)','wp-testimonials'); ?></option>
                                        <option value="_parent" <?php selected($tgt,'_parent'); ?>>_parent</option>
                                        <option value="_top" <?php selected($tgt,'_top'); ?>>_top</option>
                                    </select>
                                </div>
                                <div class="wpt-pop-col">
                                    <span class="wpt-pop-label"><?php _e('Rel attributes', 'wp-testimonials'); ?></span>
                                    <?php $rels = array_filter(array_map('trim', explode(' ', (string)$m['website_rel']))); $rels = array_fill_keys($rels, true); ?>
                                    <label class="wpt-pop-check"><input type="checkbox" name="wpt_rel[]" value="nofollow" <?php checked(isset($rels['nofollow'])); ?>/> nofollow</label>
                                    <label class="wpt-pop-check"><input type="checkbox" name="wpt_rel[]" value="sponsored" <?php checked(isset($rels['sponsored'])); ?>/> sponsored</label>
                                    <label class="wpt-pop-check"><input type="checkbox" name="wpt_rel[]" value="ugc" <?php checked(isset($rels['ugc'])); ?>/> ugc</label>
                                    <label class="wpt-pop-check"><input type="checkbox" name="wpt_rel[]" value="noreferrer" <?php checked(isset($rels['noreferrer'])); ?>/> noreferrer</label>
                                    <label class="wpt-pop-check"><input type="checkbox" name="wpt_rel[]" value="noopener" <?php checked(isset($rels['noopener'])); ?>/> noopener</label>
                                </div>
                            </div>
                            <div class="wpt-pop-actions">
                                <button type="button" class="button button-link-delete" id="wpt_link_remove"><?php _e('Remove link', 'wp-testimonials'); ?></button>
                                <button type="button" class="button button-primary" id="wpt_link_close"><?php _e('Done', 'wp-testimonials'); ?></button>
                            </div>
                        </div>
                    </div>

                    <!-- Stars moved to bottom of card -->
                    <div class="wpt-row wpt-rating" style="grid-column:1/-1; justify-content:space-between; align-items:center;">
                        <div class="wpt-star-picker" id="wpt_star_picker" aria-label="<?php esc_attr_e('Rating', 'wp-testimonials'); ?>" title="<?php esc_attr_e('Click to set rating (supports half stars).', 'wp-testimonials'); ?>">
                            <?php for($i=1;$i<=5;$i++): ?>
                                <span class="star" data-val="<?php echo $i; ?>">☆</span>
                            <?php endfor; ?>
                            <input type="hidden" id="wpt_rating" name="wpt_rating" value="<?php echo esc_attr($m['rating']); ?>" />
                        </div>
                        <button type="button" class="button wpt-rating-clear" id="wpt_rating_clear"><?php _e('Clear', 'wp-testimonials'); ?></button>
                    </div>
                </div>
            </div>
            <p class="wpt-help"><?php _e('Tip: Click the stars to set a rating. Click the photo to add an image. Use the link icon in Role/Company to add a website.', 'wp-testimonials'); ?></p>
        </div>

        <script>
        jQuery(function($){
            // Media picker for photo
            function openMedia(){
                const frame = wp.media({ title: '<?php echo esc_js(__('Select Photo', 'wp-testimonials')); ?>', button: { text: '<?php echo esc_js(__('Use this', 'wp-testimonials')); ?>' }, multiple: false });
                frame.on('select', function(){
                    const at = frame.state().get('selection').first().toJSON();
                    $('#wpt_author_photo_id').val(at.id);
                    const url = (at.sizes && at.sizes.thumbnail) ? at.sizes.thumbnail.url : at.url;
                    $('#wpt_photo_picker').html('<img src="'+url+'" alt="" />');
                });
                frame.open();
            }
            $('#wpt_photo_picker').on('click', function(e){ e.preventDefault(); openMedia(); });

            // Star picker with half-star support + clear
            const $picker = $('#wpt_star_picker');
            const $stars = $picker.find('.star');
            function paintStars(val){
                val = parseFloat(val)||0;
                const full = Math.floor(val);
                const half = (val - full) >= 0.5 ? 1 : 0;
                $stars.removeClass('fill half');
                $stars.each(function(){
                    const i = parseInt($(this).data('val'));
                    const $s = $(this);
                    if(i <= full){
                        $s.addClass('fill').text('★');
                    } else {
                        $s.text('☆');
                    }
                });
                if(half && full < 5){
                    var $h = $stars.eq(full);
                    $h.addClass('half').text('☆');
                }
            }
            function valFromEvent(e, $star){
                const rect = $star[0].getBoundingClientRect();
                const x = (e.clientX || (e.originalEvent && e.originalEvent.touches && e.originalEvent.touches[0] && e.originalEvent.touches[0].clientX) || rect.left) - rect.left;
                const half = x < rect.width/2 ? 0.5 : 1.0;
                const base = parseInt($star.data('val')) - 1; // 0-based base
                return Math.min(5, Math.max(0.5, base + half));
            }
            $stars.on('mousemove', function(e){ paintStars(valFromEvent(e, $(this))); });
            $picker.on('mouseleave', function(){ paintStars($('#wpt_rating').val()); });
            $stars.on('click', function(e){
                let v = valFromEvent(e, $(this));
                const current = parseFloat($('#wpt_rating').val())||0;
                if(v === current){ // toggle down to 0 if clicking same value again
                    v = 0;
                }
                $('#wpt_rating').val(v);
                paintStars(v);
            });
            $('#wpt_rating_clear').on('click', function(){ $('#wpt_rating').val('0'); paintStars(0); });
            paintStars($('#wpt_rating').val());

            // Link popover + inline icon state
            const $popWrap = $('#wpt_role_wrap');
            const $linkToggle = $('#wpt_link_toggle');
            const $website = $('#wpt_website');
            const $target = $('#wpt_website_target');
            function refreshLinkState(){
                var url = ($website.val()||'').trim();
                var has = url !== '';
                $linkToggle.toggleClass('has-link', has);
                // Keep the icon only; no text/underline/hostname shown in the field
            }
            // default target to _blank if not set
            if(!$target.val()){ $target.val('_blank'); }
            refreshLinkState();
            $website.on('input change', refreshLinkState);

            $linkToggle.on('click', function(e){ e.preventDefault(); $popWrap.toggleClass('open'); if($popWrap.hasClass('open')){ $website.focus(); } });
            $('#wpt_link_close').on('click', function(){ $popWrap.removeClass('open'); refreshLinkState(); });
            $('#wpt_link_remove').on('click', function(){
                $website.val('');
                $('input[name="wpt_rel[]"]').prop('checked', false);
                $target.val('_blank');
                refreshLinkState();
                $popWrap.removeClass('open');
            });
            $(document).on('click', function(e){ if(!$(e.target).closest('#wpt_role_wrap, #wpt_link_toggle').length){ $popWrap.removeClass('open'); } });
        });
        </script>
        <?php
    }

    public function save_metabox($post_id) {
        if (!isset($_POST['wpt_nonce']) || !wp_verify_nonce($_POST['wpt_nonce'], 'wpt_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;
        if (get_post_type($post_id) !== 'testimonial') return;

        $text = isset($_POST['wpt_text']) ? wp_kses_post($_POST['wpt_text']) : '';
        $author_name = isset($_POST['wpt_author_name']) ? sanitize_text_field($_POST['wpt_author_name']) : '';
        $author_role = isset($_POST['wpt_author_role']) ? sanitize_text_field($_POST['wpt_author_role']) : '';
        $website = isset($_POST['wpt_website']) ? esc_url_raw($_POST['wpt_website']) : '';
        $target = isset($_POST['wpt_website_target']) ? sanitize_text_field($_POST['wpt_website_target']) : '';
        $allowed_targets = ['_blank','_self','_parent','_top'];
        if (!in_array($target, $allowed_targets, true)) { $target = '_blank'; }
        $rels_in = isset($_POST['wpt_rel']) && is_array($_POST['wpt_rel']) ? array_map('sanitize_text_field', (array)$_POST['wpt_rel']) : [];
        $allowed_rels = ['nofollow','sponsored','ugc','noreferrer','noopener'];
        $rels = array_values(array_unique(array_intersect($rels_in, $allowed_rels)));
        $rel_str = implode(' ', $rels);
        $photo_id = isset($_POST['wpt_author_photo_id']) ? (int) $_POST['wpt_author_photo_id'] : 0;
        $rating_raw = isset($_POST['wpt_rating']) ? (float) $_POST['wpt_rating'] : 0;
        // Clamp 0..5 and snap to nearest 0.5
        $rating = max(0, min(5, round($rating_raw * 2) / 2));

        update_post_meta($post_id, '_wpt_text', $text);
        update_post_meta($post_id, '_wpt_author_name', $author_name);
        update_post_meta($post_id, '_wpt_author_role', $author_role);
        update_post_meta($post_id, '_wpt_website', $website);
        update_post_meta($post_id, '_wpt_website_target', $target);
        update_post_meta($post_id, '_wpt_website_rel', $rel_str);
        update_post_meta($post_id, '_wpt_author_photo_id', $photo_id);
        update_post_meta($post_id, '_wpt_rating', $rating);
        // _wpt_featured is managed via the list table toggle to avoid accidental changes here.

        // Auto-generate/update the post title based on provided fields
        $current_title = get_post_field('post_title', $post_id);
        $new_title = '';
        if (!empty($author_name)) {
            $new_title = $author_name;
        } else {
            $plain = trim(wp_strip_all_tags($text));
            if ($plain !== '') {
                // Truncate to ~60 chars without breaking words
                if (strlen($plain) > 60) {
                    $trunc = substr($plain, 0, 57);
                    $space = strrpos($trunc, ' ');
                    $plain = ($space !== false ? substr($trunc, 0, $space) : $trunc) . '…';
                }
                $new_title = $plain;
            }
        }
        if ($new_title === '') {
            $new_title = 'Testimonial #' . $post_id;
        }
        if ($new_title !== $current_title) {
            // Prevent infinite loop
            remove_action('save_post', [$this, 'save_metabox']);
            wp_update_post([
                'ID' => $post_id,
                'post_title' => $new_title,
            ]);
            add_action('save_post', [$this, 'save_metabox']);
        }
    }

    public function admin_assets($hook) {
        if (in_array($hook, ['post.php','post-new.php'])) {
            wp_enqueue_media();
            $screen = get_current_screen();
            if ($screen && $screen->post_type === 'testimonial') {
                // Hide the title and slug boxes for a simpler editing interface
                wp_register_style('wpt-admin-edit-style', false);
                wp_add_inline_style('wpt-admin-edit-style', '#titlediv,#edit-slug-box{display:none!important}');
                wp_enqueue_style('wpt-admin-edit-style');
            }
        }
        // List screen assets for Featured toggle
        if ($hook === 'edit.php') {
            $screen = get_current_screen();
            if ($screen && $screen->post_type === 'testimonial') {
                wp_register_script('wpt-admin-list', false, ['jquery'], '1.0.0', true);
                $nonce = wp_create_nonce('wpt_toggle_featured');
                $ajax = admin_url('admin-ajax.php');
                $inline = "jQuery(function($){\n  $(document).on('change','.wpt-featured-toggle',function(){\n    var cb=$(this);\n    var id=cb.data('post');\n    var val=cb.is(':checked')?1:0;\n    cb.prop('disabled',true);\n    $.post('{$ajax}',{ action:'wpt_toggle_featured', nonce:'{$nonce}', post_id:id, value:val }, function(resp){\n      cb.prop('disabled',false);\n      if(!resp||resp.success!==true){\n        cb.prop('checked',!val);\n        alert('Failed to update.');\n      }\n    }).fail(function(){ cb.prop('disabled',false); cb.prop('checked',!val); alert('Failed to update.'); });\n  });\n});";
                wp_add_inline_script('wpt-admin-list', $inline);
                wp_enqueue_script('wpt-admin-list');
                $css = '.column-wpt_featured{width:100px}.wpt-toggle-wrap{display:flex;align-items:center;gap:6px}.wpt-toggle{position:relative;display:inline-block;width:40px;height:20px}.wpt-toggle input{display:none}.wpt-slider-knob{position:absolute;cursor:pointer;top:0;left:0;right:0;bottom:0;background:#ccc;transition:.2s;border-radius:999px}.wpt-slider-knob:before{position:absolute;content:"";height:16px;width:16px;left:2px;top:2px;background:white;transition:.2s;border-radius:50%}.wpt-toggle input:checked + .wpt-slider-knob{background:#4caf50}.wpt-toggle input:checked + .wpt-slider-knob:before{transform:translateX(20px)}';
                wp_register_style('wpt-admin-list-style', false);
                wp_add_inline_style('wpt-admin-list-style', $css);
                wp_enqueue_style('wpt-admin-list-style');
            }
        }
    }

    public function register_front_assets() {
        // Registered but not enqueued until shortcode is used
        wp_register_style('wpt-front', WPTEST_URL . 'assets/front.css', [], '1.0.0');
        wp_register_script('wpt-slider', WPTEST_URL . 'assets/slider.js', [], '1.0.0', true);
        wp_register_script('wpt-grid', WPTEST_URL . 'assets/grid.js', [], '1.0.0', true);
        wp_register_script('wpt-bubble', WPTEST_URL . 'assets/bubble.js', [], '1.0.0', true);
    }

    public function shortcode_grid($atts) {
        // Grid-specific defaults, remove slider-only options
        $atts = shortcode_atts([
            'category' => '',
            'limit' => '12',
            'show_rating' => 'true',
            'show_photo' => 'true',
            'gap' => '16px',
            'card_bg' => '#ffffff',
            'text_color' => '#222222',
            'accent_color' => '#ffb400',
            'bubble' => 'bottom-left',
            'grid_style' => 'standard',
            'target_width' => '360px',
            'border_width' => '0',
            'orderby' => 'date',
            'order' => 'DESC',
        ], $atts, 'testimonials_grid');
        $atts['display'] = 'grid';
        return $this->shortcode($atts);
    }

    public function shortcode_slider($atts) {
        // Slider-specific defaults, remove grid-only options
        $atts = shortcode_atts([
            'category' => '',
            'limit' => '12',
            'show_rating' => 'true',
            'show_photo' => 'true',
            'autoplay' => '4000',
            'arrows' => 'show',
            'dots' => 'true',
            'gap' => '16px',
            'card_bg' => '#ffffff',
            'text_color' => '#222222',
            'accent_color' => '#ffb400',
            'bubble' => 'bottom-left',
            'card_max_width' => '1440px',
            'border_width' => '0',
            'orderby' => 'date',
            'order' => 'DESC',
        ], $atts, 'testimonials_slider');
        $atts['display'] = 'slider';
        return $this->shortcode($atts);
    }

    public function shortcode($atts) {
        $defaults = [
            'display' => 'grid', // grid|slider
            'category' => '',
            'limit' => '12',
            'target_width' => '360px', // grid responsiveness is driven by target width
            'show_rating' => 'true',
            'show_photo' => 'true',
            'autoplay' => '4000', // ms, slider only
            'arrows' => 'show', // hide|show|hover for slider
            'dots' => 'true',
            'gap' => '16px',
            'card_bg' => '#ffffff',
            'text_color' => '#222222',
            'accent_color' => '#ffb400',
            'bubble' => 'bottom-left',
            'grid_style' => 'standard', // standard|equal|masonry (grid only)
            'card_max_width' => '1440px', // slider only; default max width for card
            'border_width' => '0',
            'orderby' => 'date',
            'order' => 'DESC',
        ];
        $a = shortcode_atts($defaults, $atts, 'testimonials');

        $args = [
            'post_type' => 'testimonial',
            'posts_per_page' => (int) $a['limit'],
            'orderby' => sanitize_key($a['orderby']),
            'order' => $a['order'] === 'ASC' ? 'ASC' : 'DESC',
        ];
        if (!empty($a['category'])) {
            $args['tax_query'] = [[
                'taxonomy' => 'testimonial_category',
                'field' => 'slug',
                'terms' => array_map('sanitize_title', array_map('trim', explode(',', $a['category'])))
            ]];
        }
        $q = new WP_Query($args);
        if (!$q->have_posts()) return '';

        // Determine display and grid style early for asset enqueues
        $is_slider = ($a['display'] === 'slider');
        $grid_style = in_array(isset($a['grid_style']) ? $a['grid_style'] : 'standard', ['standard','equal','masonry'], true) ? $a['grid_style'] : 'standard';

        wp_enqueue_style('wpt-front');
        if ($a['display'] === 'slider') {
            wp_enqueue_script('wpt-slider');
        } else {
            if ($grid_style === 'masonry') {
                wp_enqueue_script('wpt-grid');
            }
        }
        // Ensure bubble tail offset reacts to current border width
        wp_enqueue_script('wpt-bubble');

        // Build styles
        $uid = 'wpt-' . wp_generate_password(6, false);
        $gap = esc_attr($a['gap']);
        $card_bg = esc_attr($a['card_bg']);
        $text_color = esc_attr($a['text_color']);
        $accent_color = esc_attr($a['accent_color']);

        ob_start();
        $bubble = in_array(isset($a['bubble']) ? $a['bubble'] : 'bottom-left', ['none','bottom-left','bottom-center','bottom-right'], true) ? $a['bubble'] : 'bottom-left';
        $grid_style = in_array(isset($a['grid_style']) ? $a['grid_style'] : 'standard', ['standard','equal','masonry'], true) ? $a['grid_style'] : 'standard';
        $card_max = isset($a['card_max_width']) ? trim($a['card_max_width']) : '';
        $card_max_norm = strtolower(str_replace(' ', '', $card_max));
        $target_width = isset($a['target_width']) ? trim($a['target_width']) : '360px';
        $is_slider = ($a['display'] === 'slider');
        ?>
        <?php
        // Build inline CSS variables. When rendered via Divi module (design_source=divi),
        // avoid setting color-related vars so Divi design settings and theme defaults take over.
        $style_vars = [
            '--wpt-gap: ' . $gap,
            '--wpt-target: ' . esc_attr($target_width),
        ];
        if ($is_slider && $card_max !== '') {
            $style_vars[] = '--wpt-card-max: ' . esc_attr($card_max);
        }
        // Border width is always set so bubble tail offset can align in all modes (including Divi)
        $bw = isset($a['border_width']) ? trim((string)$a['border_width']) : '0';
        if ($bw === '') { $bw = '0'; }
        $style_vars[] = '--wpt-bq-border: ' . esc_attr($bw);
        if (!isset($a['design_source']) || $a['design_source'] !== 'divi') {
            // Non‑Divi usage: provide sensible defaults
            $style_vars[] = '--wpt-card-bg: ' . $card_bg;
            $style_vars[] = '--wpt-text: ' . $text_color;
            $style_vars[] = '--wpt-accent: ' . $accent_color;
            // Provide comfortable default radius/padding/shadow
            $style_vars[] = '--wpt-radius: 12px';
            $style_vars[] = '--wpt-padding: 20px 24px';
            $style_vars[] = '--wpt-shadow: 0 1px 3px rgba(0,0,0,.08)';
            $style_vars[] = '--wpt-author-text-size: 14px';
            $style_vars[] = '--wpt-stars-size: 14px';
        }
        $style_attr = implode('; ', $style_vars);
        // Build wrapper classes and mark Divi-designed instances
        $wrap_classes = 'wpt-wrap wpt-' . ($is_slider ? 'slider' : 'grid');
        if (isset($a['design_source']) && $a['design_source'] === 'divi') {
            $wrap_classes .= ' wpt-design-divi';
        }
        ?>
        <div id="<?php echo esc_attr($uid); ?>" class="<?php echo esc_attr($wrap_classes); ?>" data-grid-style="<?php echo esc_attr($grid_style); ?>" data-autoplay="<?php echo (int)$a['autoplay']; ?>" data-arrows="<?php echo esc_attr($a['arrows']); ?>" data-dots="<?php echo $a['dots']==='true'?'1':'0'; ?>" style="<?php echo esc_attr($style_attr); ?>">
            <style>
                <?php if(!$is_slider): ?>
                /* Standard and Equal grid respond to target width */
                #<?php echo esc_js($uid); ?>.wpt-grid[data-grid-style="standard"] .wpt-items,
                #<?php echo esc_js($uid); ?>.wpt-grid[data-grid-style="equal"] .wpt-items { grid-template-columns: repeat(auto-fit, minmax(min(100%, var(--wpt-target)), 1fr)); }
                /* Masonry uses CSS columns sized by target width */
                #<?php echo esc_js($uid); ?>.wpt-grid[data-grid-style="masonry"] .wpt-items { column-width: var(--wpt-target); column-gap: var(--wpt-gap); }
                <?php endif; ?>
            </style>
            <div class="wpt-items">
                <?php while($q->have_posts()): $q->the_post();
                    $m = [
                        'author_name' => get_post_meta(get_the_ID(), '_wpt_author_name', true),
                        'author_role' => get_post_meta(get_the_ID(), '_wpt_author_role', true),
                        'website' => get_post_meta(get_the_ID(), '_wpt_website', true),
                        'website_target' => get_post_meta(get_the_ID(), '_wpt_website_target', true),
                        'website_rel' => get_post_meta(get_the_ID(), '_wpt_website_rel', true),
                        'author_photo_id' => (int) get_post_meta(get_the_ID(), '_wpt_author_photo_id', true),
                        'rating' => (float) get_post_meta(get_the_ID(), '_wpt_rating', true),
                    ];
                    $photo = $m['author_photo_id'] ? wp_get_attachment_image_url($m['author_photo_id'], 'thumbnail') : '';
                    $t_text = get_post_meta(get_the_ID(), '_wpt_text', true);
                ?>
                <div class="<?php echo $is_slider ? 'wpt-slide' : 'wpt-item'; ?>">
                  <figure class="wpt-card">
                    <blockquote class="wpt-bubble bubble-<?php echo esc_attr($bubble); ?>"><?php echo wp_kses_post($t_text !== '' ? $t_text : get_the_content()); ?></blockquote>
                    <figcaption class="wpt-author">
                        <?php if($a['show_photo']==='true' && $photo): ?>
                            <img class="wpt-photo" src="<?php echo esc_url($photo); ?>" alt=""/>
                        <?php endif; ?>
                        <div class="wpt-author-text">
                            <?php if(!empty($m['author_name'])): ?>
                            <div class="wpt-name"><?php echo esc_html($m['author_name']); ?></div>
                            <?php endif; ?>
                            <?php 
                                $role_html = '';
                                if(!empty($m['author_role'])){
                                    $role_text = esc_html($m['author_role']);
                                    if(!empty($m['website'])){
                                        $tgt = get_post_meta(get_the_ID(), '_wpt_website_target', true);
                                        $tgt = in_array($tgt, ['_blank','_self','_parent','_top'], true) ? $tgt : '_blank';
                                        $rel_raw = trim((string) get_post_meta(get_the_ID(), '_wpt_website_rel', true));
                                        $rels_arr = array_filter(array_map('trim', preg_split('/\s+/', $rel_raw)));
                                        if ($tgt === '_blank' && !in_array('noopener', $rels_arr, true)) { $rels_arr[] = 'noopener'; }
                                        $rel_attr = implode(' ', array_unique($rels_arr));
                                        $role_html = '<a href="'.esc_url($m['website']).'" target="'.esc_attr($tgt).'"'.($rel_attr!==''?' rel="'.esc_attr($rel_attr).'"':'').'>'.$role_text.'</a>';
                                    } else {
                                        $role_html = $role_text;
                                    }
                                    echo '<div class="wpt-role">'.$role_html.'</div>';
                                }
                            ?>
                            <?php 
                            $r = (float)$m['rating'];
                            if ($a['show_rating']==='true' && $r > 0):
                                if ($r < 0) $r = 0; if ($r > 5) $r = 5;
                                $full = floor($r);
                                $half = ($r - $full) >= 0.5 ? 1 : 0;
                                $empty = 5 - $full - $half;
                            ?>
                                <div class="wpt-stars" aria-label="<?php echo esc_attr($r); ?> out of 5">
                                    <?php for($i=0;$i<$full;$i++): ?><span class="star full">★</span><?php endfor; ?>
                                    <?php if($half): ?><span class="star half">☆</span><?php endif; ?>
                                    <?php for($i=0;$i<$empty;$i++): ?><span class="star empty">☆</span><?php endfor; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </figcaption>
                  </figure>
                </div>
                <?php endwhile; wp_reset_postdata(); ?>
            </div>
            <?php if($a['display']==='slider'): ?>
                <button class="wpt-prev" type="button" aria-label="Previous">‹</button>
                <button class="wpt-next" type="button" aria-label="Next">›</button>
                <div class="wpt-dots"></div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    public function disable_single() {
        if (is_singular('testimonial')) {
            global $wp_query; $wp_query->set_404(); status_header(404); nocache_headers();
            include get_404_template(); exit;
        }
    }

    public function columns($cols) {
        // Build a clean set of columns:
        // - Keep checkbox and Title
        // - Add Featured toggle after Title
        // - Keep Testimonial Categories taxonomy column (WordPress key: taxonomy-testimonial_category)
        // - Add Rating
        // - Remove Date and any default/duplicate Categories or Thumbnail columns
        $new = [];
        if (isset($cols['cb'])) { $new['cb'] = $cols['cb']; }
        $new['title'] = isset($cols['title']) ? $cols['title'] : __('Title');
        $new['wpt_featured'] = __('Featured', 'wp-testimonials');
        if (isset($cols['taxonomy-testimonial_category'])) {
            $new['taxonomy-testimonial_category'] = $cols['taxonomy-testimonial_category'];
        }
        $new['wpt_rating'] = __('Rating', 'wp-testimonials');
        return $new;
    }

    public function column_content($column, $post_id) {
        if ($column === 'wpt_cat') {
            $terms = get_the_terms($post_id, 'testimonial_category');
            if (empty($terms) || is_wp_error($terms)) { echo '—'; return; }
            echo esc_html(implode(', ', wp_list_pluck($terms, 'name')));
        } elseif ($column === 'wpt_rating') {
            $r = (int) get_post_meta($post_id, '_wpt_rating', true);
            echo $r ? str_repeat('★', $r) : '—';
        } elseif ($column === 'wpt_featured') {
            $is = (bool) get_post_meta($post_id, '_wpt_featured', true);
            echo '<label class="wpt-toggle"><input type="checkbox" class="wpt-featured-toggle" data-post="'.(int)$post_id.'" '.checked($is, true, false).' /><span class="wpt-slider-knob"></span></label>';
        }
    }

    public function ajax_toggle_featured() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpt_toggle_featured')) {
            wp_send_json_error(['message' => 'Invalid nonce'], 403);
        }
        $post_id = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;
        if (!$post_id || get_post_type($post_id) !== 'testimonial') {
            wp_send_json_error(['message' => 'Invalid post'], 400);
        }
        if (!current_user_can('edit_post', $post_id)) {
            wp_send_json_error(['message' => 'No permission'], 403);
        }
        $val = isset($_POST['value']) ? (int) $_POST['value'] : 0;
        update_post_meta($post_id, '_wpt_featured', $val ? 1 : 0);
        wp_send_json_success(['value' => $val ? 1 : 0]);
    }

    public function register_divi_module() {
        new WPT_Divi_Testimonials_Grid_Module();
        new WPT_Divi_Testimonials_Slider_Module();
    }
}

// Instantiate plugin and expose a global reference for internal rendering helpers
$GLOBALS['wpt_plugin'] = new WP_Testimonials_Plugin();

// Public helper functions so Divi modules (and others) can render without relying on do_shortcode
if (!function_exists('wpt_render_testimonials_grid')) {
    function wpt_render_testimonials_grid($atts = []) {
        if (isset($GLOBALS['wpt_plugin']) && $GLOBALS['wpt_plugin'] instanceof WP_Testimonials_Plugin) {
            return $GLOBALS['wpt_plugin']->shortcode_grid((array)$atts);
        }
        // Fallback to shortcode execution if instance not available
        return do_shortcode('[testimonials_grid ' . http_build_query((array)$atts, '', ' ') . ']');
    }
}
if (!function_exists('wpt_render_testimonials_slider')) {
    function wpt_render_testimonials_slider($atts = []) {
        if (isset($GLOBALS['wpt_plugin']) && $GLOBALS['wpt_plugin'] instanceof WP_Testimonials_Plugin) {
            return $GLOBALS['wpt_plugin']->shortcode_slider((array)$atts);
        }
        return do_shortcode('[testimonials_slider ' . http_build_query((array)$atts, '', ' ') . ']');
    }
}

// Load Divi modules using include files on Divi's ready hook, as recommended
add_action('et_builder_ready', function () {
    // Each file defines and instantiates its module class
    if (defined('WPTEST_DIR')) {
        $grid = WPTEST_DIR . 'includes/module-testimonials-grid.php';
        $slider = WPTEST_DIR . 'includes/module-testimonials-slider.php';
        if (file_exists($grid)) { include_once $grid; }
        if (file_exists($slider)) { include_once $slider; }
    }
});


////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if (class_exists('makeUpdate')) {
    try {
        $updater = new makeUpdate("9551", "yva!baj0ybh6bmj4BVR", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}