<?php

namespace Optimus\Divi\Modules\GoogleMapWithStyle;

use Optimus\Divi\Modules\BaseModule as DiviModule;

class Module extends DiviModule
{
    public $name = 'Google Map with Style';

    public function init()
    {
        parent::init();
    }

    public function fields()
    {
        return [
            'address' => [
                'label' => 'Address',
                'type' => 'text',
            ],
            'zoom' => [
                'label' => 'Map Zoom',
                'type' => 'range',
                'default' => 7,
                'range_settings' => [
                    'min' => '1',
                    'max' => '19',
                    'step' => '1',
                ],
                'description' => __('Zoom level (1:World 19:Individual property) ', 'et_builder'),
            ],
            'marker_src' => [
                'label' => __('Marker URL', 'et_builder'),
                'type' => 'upload',
                'option_category' => 'basic_option',
                'upload_button_text' => __('Upload an image', 'et_builder'),
                'choose_text' => __('Choose an Image', 'et_builder'),
                'update_text' => __('Set As Image', 'et_builder'),
                'description' => __('Upload your desired image, or type in the URL to the image you would like to display.', 'et_builder'),
            ],
            'marker_size' => [
                'label' => __('Marker Size', 'et_builder'),
                'type' => 'range',
                'default' => 64,
                'default_unit' => 'px',
                'range_settings' => [
                    'min' => '10',
                    'max' => '100',
                    'step' => '1',
                ],
            ],
            'height' => [
                'label' => 'Height',
                'type' => 'text',
                'description' => __('Map Height in px (will default to 500px)', 'et_builder'),
            ],
            'controls' => [
                'label' => __('Show Controls', 'et_builder'),
                'type' => 'yes_no_button',
                'options' => [
                    'on' => __('Yes', 'et_builder'),
                    'off' => __('No', 'et_builder'),
                ],
            ],
            'marker_click_type' => [
                'label' => __('Marker Click Action', 'et_builder'),
                'type' => 'select',
                'options' => [
                    '' => __('None', 'et_builder'),
                    'popup' => __('Popup with Info Text', 'et_builder'),
                    'open_url' => __('Open URL', 'et_builder'),
                    'open_place_name' => __('Open Place by Name', 'et_builder'),
                    'open_place_id' => __('Open Place by ID', 'et_builder'),
                ],
                'default' => '',
                'description' => __('Choose what should happen when the marker is clicked.', 'et_builder'),
            ],
            'info_content' => [
                'label' => __('Popup Text Content', 'et_builder'),
                'type' => 'textarea',
                'show_if' => [
                    'marker_click_type' => 'popup',
                ],
                'description' => __('Text that will show in the popup.', 'et_builder'),
            ],
            'custom_url' => [
                'label' => __('URL to Open', 'et_builder'),
                'type' => 'text',
                'show_if' => [
                    'marker_click_type' => 'open_url',
                ],
                'description' => __('Enter the URL that should open on marker click.', 'et_builder'),
            ],
            'place_name' => [
                'label' => __('Place Name', 'et_builder'),
                'type' => 'text',
                'show_if' => [
                    'marker_click_type' => 'open_place_name',
                ],
                'description' => __('Enter the name of the place to open in Google Maps.', 'et_builder'),
            ],
            'place_id' => [
                'label' => __('Place ID', 'et_builder'),
                'type' => 'text',
                'show_if' => [
                    'marker_click_type' => 'open_place_id',
                ],
                'description' => __(
                    'Enter the Google Place ID. You can find Place IDs here: <a href="https://developers.google.com/maps/documentation/places/web-service/place-id" target="_blank">Google Place ID Finder</a>',
                    'et_builder'
                ),
            ],                
            'disabled_on' => [
                'label' => __('Disable on', 'et_builder'),
                'type' => 'multiple_checkboxes',
                'options' => [
                    'phone' => __('Phone', 'et_builder'),
                    'tablet' => __('Tablet', 'et_builder'),
                    'desktop' => __('Desktop', 'et_builder'),
                ],
                'additional_att' => 'disable_on',
                'option_category' => 'configuration',
                'description' => __('This will disable the module on selected devices', 'et_builder'),
            ],
            'admin_label' => [
                'label' => __('Admin Label', 'et_builder'),
                'type' => 'text',
                'description' => __('This will change the label of the module in the builder for easy identification.', 'et_builder'),
            ],
            'module_id' => [
                'label' => __('CSS ID', 'et_builder'),
                'type' => 'text',
                'option_category' => 'configuration',
                'tab_slug' => 'custom_css',
                'option_class' => 'et_pb_custom_css_regular',
            ],
            'module_class' => [
                'label' => __('CSS Class', 'et_builder'),
                'type' => 'text',
                'option_category' => 'configuration',
                'tab_slug' => 'custom_css',
                'option_class' => 'et_pb_custom_css_regular',
            ],
            'style' => [
                'label' => __('Google Maps Style', 'et_builder'),
                'type' => 'textarea',
                'description' => __('Custom google maps style for your map. (style ideas <a href="https://snazzymaps.com/">https://snazzymaps.com</a>) (must be base64 encoded <a href="https://codepen.io/unetic/full/bOgVGM">SNAZY4DIVI</a>)', 'et_builder'),
            ],
        ];
    }

    function isJson($str) {
        $json = json_decode($str);
        return $json && $str != $json;
    }

    public function shortcode_callback($atts, $content, $function_name)
    {
        $module_id = $this->shortcode_atts['module_id'];
        $module_class = $this->shortcode_atts['module_class'];
        $address = $this->shortcode_atts['address'];
        $zoom = $this->shortcode_atts['zoom'];
        $marker_src = $this->shortcode_atts['marker_src'];
        $marker_size = $this->props['marker_size'];
        $marker_size = preg_replace('/px$/', '', $marker_size);
        $height = $this->shortcode_atts['height'];
        $controls = $this->shortcode_atts['controls'];
        $marker_click_type = $this->shortcode_atts['marker_click_type'];
        $info_content = $this->shortcode_atts['info_content'];
        $info_content = html_entity_decode($info_content);
        $custom_url = $this->shortcode_atts['custom_url'];
        $place_name = $this->shortcode_atts['place_name'];
        $place_id = $this->shortcode_atts['place_id'];
        
        $snazzy_style = $this->shortcode_atts['style'];

        $module_class = \ET_Builder_Element::add_module_order_class($module_class, $function_name);

        $apikey = et_pb_get_google_api_key();

        $snazzy_style = base64_decode($snazzy_style);

        if(!$this->isJson($snazzy_style)){
            $snazzy_style = '[]';
        }
        
        $info_fields = '';
        $info_fields .= '<div class="address">' . esc_html($address ?: 'East Gosford') . '</div>';
        $info_fields .= '<div class="zoom">' . esc_html($zoom ?: '7') . '</div>';
        $info_fields .= '<div class="marker_src">' . esc_html($marker_src) . '</div>';
        $info_fields .= '<div class="marker_size">' . esc_html($marker_size ?: '64') . '</div>';
        $info_fields .= '<div class="controls">' . ($controls === 'off' ? 'true' : 'false') . '</div>';
        $info_fields .= '<div class="snazzy">' . esc_html($snazzy_style) . '</div>';
        $info_fields .= '<div class="marker_click">' . esc_html($marker_click_type) . '</div>';
        
        if ($marker_click_type === 'popup' && $info_content) {
            $info_fields .= '<div class="info_content">' . do_shortcode($info_content) . '</div>';
        }
        
        if ($marker_click_type === 'open_url' && $custom_url) {
            $info_fields .= '<div class="custom_url">' . esc_url($custom_url) . '</div>';
        }
        
        if ($marker_click_type === 'open_place_name' && $place_name) {
            $info_fields .= '<div class="place_name">' . esc_html($place_name) . '</div>';
        }
        
        if ($marker_click_type === 'open_place_id' && $place_id) {
            $info_fields .= '<div class="place_id">' . esc_html($place_id) . '</div>';
        }
        $output = sprintf(
            '<div %1$s class="block maps%2$s">
                <div class="content">
                    <div class="map_canvas" style="%3$s">
                        <div class="infotext" style="display:none;">%4$s</div>
                    </div>
                </div>
            </div>',
            ('' !== $module_id ? sprintf(' id="%1$s"', esc_attr($module_id)) : ''),
            ('' !== $module_class ? ' ' . esc_attr($module_class) : ''),
            ('' !== $height ? 'height:' . esc_attr($height) . 'px;' : 'height: 500px;'),
            $info_fields
        );

        if (!is_admin()) {
            wp_enqueue_script('google_js', "https://maps.googleapis.com/maps/api/js?v=3.exp&key=$apikey", '', '');
            wp_enqueue_script('load_maps', $this->assetUrl('loadMaps.js'));
            $style = $snazzy_style ? $snazzy_style : '[{"featureType":"all","elementType":"all","stylers":[{"visibility":"on"}]}]';
            wp_localize_script('load_maps', 'snazzy_style', $style);
        }

        return $output;
    }
}
