<?php
/**
 * Portfolio Pack Settings Page
 *
 * @package portfolio_pack
 * 
 * Note: A more robust version of this implementation is archived in the 'archive' directory.
 * The archived version includes functionality for creating and editing field groups through the ACF UI.
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Register the Portfolio Pack settings page
 */
function pp_register_settings_page() {
    add_submenu_page(
        'edit.php?post_type=portfolio',
        'Portfolio Pack Settings',
        'Settings',
        'manage_options',
        'portfolio-pack-settings',
        'pp_render_settings_page'
    );
}
add_action('admin_menu', 'pp_register_settings_page');

/**
 * Register settings
 */
function pp_register_settings() {
    register_setting('portfolio_pack_settings', 'portfolio_pack_modules');
}
add_action('admin_init', 'pp_register_settings');

/**
 * Render the settings page
 */
function pp_render_settings_page() {
    // Get available modules
    $available_modules = pp_get_available_modules();

    // Get enabled modules
    $enabled_modules = get_option('portfolio_pack_modules', array());

    // If option doesn't exist yet, enable all modules by default
    if (empty($enabled_modules)) {
        $enabled_modules = array_keys($available_modules);
        update_option('portfolio_pack_modules', $enabled_modules);
    }

    // Process ACF field import if form was submitted
    if (isset($_POST['pp_import_acf_fields']) && isset($_FILES['acf_json_file'])) {
        pp_process_acf_field_import();
    }

    // Process example field group import if form was submitted
    if (isset($_POST['pp_import_example_fields']) && isset($_POST['example_field_group'])) {
        pp_process_example_field_group_import();
    }

    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

        <?php settings_errors('portfolio_pack_acf_import'); ?>

        <h2 class="nav-tab-wrapper">
            <a href="#modules-tab" class="nav-tab nav-tab-active">Modules</a>
            <a href="#acf-import-tab" class="nav-tab">Portfolio Fields Import</a>
        </h2>

        <div id="modules-tab" class="tab-content">
            <form method="post" action="options.php">
                <?php settings_fields('portfolio_pack_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enabled Modules</th>
                        <td>
                            <?php foreach ($available_modules as $module_id => $module_name) : ?>
                                <label>
                                    <input type="checkbox" name="portfolio_pack_modules[]" value="<?php echo esc_attr($module_id); ?>" 
                                        <?php checked(in_array($module_id, $enabled_modules)); ?>>
                                    <?php echo esc_html($module_name); ?>
                                </label><br>
                            <?php endforeach; ?>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>

        <div id="acf-import-tab" class="tab-content" style="display: none;">
            <h2>Import Portfolio Detail Fields from JSON</h2>
            <p>Upload a JSON file to import fields into the Portfolio Details field group.</p>

            <h3>Select Example Field Group</h3>
            <p>Choose from pre-defined example field groups:</p>

            <form method="post">
                <input type="hidden" name="acf_field_group_key" value="group_66f38f34bb31f">
                <table class="form-table">
                    <tr>
                        <th scope="row">Example Field Groups</th>
                        <td>
                            <select name="example_field_group" required>
                                <option value="">Select an example field group</option>
                                <?php
                                $example_field_groups = pp_get_example_field_groups();
                                foreach ($example_field_groups as $file => $name) :
                                ?>
                                    <option value="<?php echo esc_attr($file); ?>"><?php echo esc_html($name); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">Select a pre-defined field group to import.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Import Example Fields', 'secondary', 'pp_import_example_fields'); ?>
            </form>

            <h3>Or Upload Custom JSON File</h3>
            <p>Upload your own JSON file containing ACF fields:</p>

            <form method="post" enctype="multipart/form-data">
                <input type="hidden" name="acf_field_group_key" value="group_66f38f34bb31f">
                <table class="form-table">
                    <tr>
                        <th scope="row">JSON File</th>
                        <td>
                            <input type="file" name="acf_json_file" required accept=".json">
                            <p class="description">Upload a JSON file containing ACF fields.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Import Fields', 'primary', 'pp_import_acf_fields'); ?>
            </form>
        </div>
    </div>

    <script>
    jQuery(document).ready(function($) {
        // Check if we need to show the ACF import tab (after form submission)
        var showAcfTab = <?php echo (isset($_POST['pp_import_acf_fields']) || isset($_POST['pp_import_example_fields'])) ? 'true' : 'false'; ?>;

        if (showAcfTab) {
            // Hide all tab content
            $('.tab-content').hide();

            // Remove active class from all tabs
            $('.nav-tab').removeClass('nav-tab-active');

            // Add active class to ACF import tab
            $('a[href="#acf-import-tab"]').addClass('nav-tab-active');

            // Show the ACF import tab content
            $('#acf-import-tab').show();
        }

        // Tab navigation
        $('.nav-tab').on('click', function(e) {
            e.preventDefault();

            // Hide all tab content
            $('.tab-content').hide();

            // Remove active class from all tabs
            $('.nav-tab').removeClass('nav-tab-active');

            // Add active class to clicked tab
            $(this).addClass('nav-tab-active');

            // Show the corresponding tab content
            $($(this).attr('href')).show();
        });
    });
    </script>
    <?php
}

/**
 * Get available modules
 * 
 * @return array Array of module IDs and names
 */
function pp_get_available_modules() {
    return array(
        'portfolio-specs' => 'Portfolio Specifications',
        // Add more modules here as they are created
    );
}

/**
 * Get available example field groups
 * 
 * @return array Array of example field group files and names
 */
function pp_get_example_field_groups() {
    $example_field_groups = array();
    $example_dir = PORTFOLIO_PACK_PLUGIN_DIR . 'includes/example-field-groups';

    // Check if directory exists
    if (!is_dir($example_dir)) {
        return $example_field_groups;
    }

    // Get all JSON files in the directory
    $files = glob($example_dir . '/*.json');

    foreach ($files as $file) {
        $filename = basename($file);
        $name = str_replace(array('-', '_', '.json'), array(' ', ' ', ''), $filename);
        $name = ucwords($name);
        $example_field_groups[$filename] = $name;
    }

    return $example_field_groups;
}

/**
 * Process example field group import
 */
function pp_process_example_field_group_import() {
    // Check if we have the required data
    if (!isset($_POST['example_field_group']) || empty($_POST['example_field_group'])) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'missing_example_field_group',
            __('Please select an example field group.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Get the example field group filename
    $filename = sanitize_text_field($_POST['example_field_group']);
    $file_path = PORTFOLIO_PACK_PLUGIN_DIR . 'includes/example-field-groups/' . $filename;

    // Check if file exists
    if (!file_exists($file_path)) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'file_not_found',
            __('Example field group file not found.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Read the file contents
    $json_string = file_get_contents($file_path);
    if ($json_string === false) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'file_read_error',
            __('Error reading file. Please try again.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Get the field group key
    $field_group_key = 'group_66f38f34bb31f';

    // Import the fields
    if (function_exists('pp_import_acf_fields_from_json')) {
        try {
            $result = pp_import_acf_fields_from_json($json_string, $field_group_key);

            if (is_wp_error($result)) {
                add_settings_error(
                    'portfolio_pack_acf_import',
                    'import_error',
                    $result->get_error_message(),
                    'error'
                );
            } else {
                add_settings_error(
                    'portfolio_pack_acf_import',
                    'import_success',
                    __('Example fields imported successfully.', 'portfolio_pack'),
                    'success'
                );

                // Check module files after import for any changes or corruption
                $modules_dir = PORTFOLIO_PACK_PLUGIN_DIR . 'includes/modules';
                if (is_dir($modules_dir)) {
                    $module_dirs = glob($modules_dir . '/*', GLOB_ONLYDIR);
                    foreach ($module_dirs as $module_dir) {
                        // Check if module file exists
                        $module_file = $module_dir . '/module.php';
                    }
                }
            }
        } catch (Exception $e) {
            add_settings_error(
                'portfolio_pack_acf_import',
                'import_exception',
                __('Exception during import: ', 'portfolio_pack') . $e->getMessage(),
                'error'
            );
        }
    } else {
        add_settings_error(
            'portfolio_pack_acf_import',
            'function_not_found',
            __('Import function not found. Please check if the helper functions are loaded.', 'portfolio_pack'),
            'error'
        );
    }
}

/**
 * Check if a module is enabled
 * 
 * @param string $module_id The module ID to check
 * @return bool Whether the module is enabled
 */
function pp_is_module_enabled($module_id) {
    $enabled_modules = get_option('portfolio_pack_modules', array());

    // If option doesn't exist yet, enable all modules by default
    if (empty($enabled_modules)) {
        $available_modules = pp_get_available_modules();
        $enabled_modules = array_keys($available_modules);
        update_option('portfolio_pack_modules', $enabled_modules);
    }

    return in_array($module_id, $enabled_modules);
}

/**
 * Process ACF field import from JSON file
 * This simplified version only imports fields into the Portfolio Details field group
 */
function pp_process_acf_field_import() {
    // Check if a file was uploaded
    if (!isset($_FILES['acf_json_file']) || $_FILES['acf_json_file']['error'] !== UPLOAD_ERR_OK) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'file_upload_error',
            __('Error uploading file. Please try again.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Get the field group key (hardcoded to Portfolio Details field group)
    $field_group_key = 'group_66f38f34bb31f';

    // Get the uploaded file
    $file = $_FILES['acf_json_file'];

    // Check file type
    $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    if ($file_ext !== 'json') {
        add_settings_error(
            'portfolio_pack_acf_import',
            'invalid_file_type',
            __('Please upload a JSON file.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Read the file contents
    $json_string = file_get_contents($file['tmp_name']);
    if ($json_string === false) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'file_read_error',
            __('Error reading file. Please try again.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Decode JSON
    $json_data = json_decode($json_string, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'invalid_json',
            __('Invalid JSON format.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Check if JSON contains fields
    if (!isset($json_data['fields']) || !is_array($json_data['fields'])) {
        add_settings_error(
            'portfolio_pack_acf_import',
            'no_fields',
            __('No fields found in JSON.', 'portfolio_pack'),
            'error'
        );
        return;
    }

    // Import fields directly using ACF functions
    if (function_exists('acf_get_field_group') && function_exists('acf_update_field')) {
        // Get the field group
        $field_group = acf_get_field_group($field_group_key);
        if (!$field_group) {
            add_settings_error(
                'portfolio_pack_acf_import',
                'group_not_found',
                __('Portfolio Details field group not found.', 'portfolio_pack'),
                'error'
            );
            return;
        }

        // Import fields
        foreach ($json_data['fields'] as $field) {
            // Make sure the field has a key
            if (!isset($field['key'])) {
                $field['key'] = 'field_' . uniqid();
            }

            // Set the parent field group
            $field['parent'] = $field_group_key;

            // Update the field
            acf_update_field($field);
        }

        add_settings_error(
            'portfolio_pack_acf_import',
            'import_success',
            __('Fields imported successfully.', 'portfolio_pack'),
            'success'
        );
    } else {
        add_settings_error(
            'portfolio_pack_acf_import',
            'acf_not_active',
            __('Advanced Custom Fields is not active.', 'portfolio_pack'),
            'error'
        );
    }
}
