<?php
/**
 * ACF Field Handler Class
 *
 * @package portfolio_pack
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Class PP_ACF_Field_Handler
 *
 * Handles the addition of fields to ACF field groups
 */
class PP_ACF_Field_Handler {
    /**
     * Singleton instance
     *
     * @var PP_ACF_Field_Handler
     */
    private static $instance = null;

    /**
     * Get the singleton instance
     *
     * @return PP_ACF_Field_Handler
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        // Private constructor to enforce singleton pattern
    }

    /**
     * Add a field to an ACF field group
     *
     * @param array $field The field data
     * @return array|bool The field data with ID on success, false on failure
     */
    public function add_field($field) {
        // Validate and prepare the field
        $field = wp_parse_args($field, array(
            'key' => '',
            'label' => '',
            'name' => '',
            'type' => 'text',
            'parent' => 0, // can be ID or group key
            'menu_order' => 0,
            'instructions' => '',
            'required' => 0,
            'id' => '',
            'class' => '',
            'conditional_logic' => 0,
            '_valid' => 0,
            'prefix' => 'acf'
        ));


        // Generate field key if not provided
        if (empty($field['key'])) {
            $field['key'] = 'field_' . uniqid();
        }

        // Ensure we have a valid parent
        if (empty($field['parent'])) {
            return false;
        }

        // If parent is a string (assuming it's a field group key), get its ID
        if (!is_numeric($field['parent'])) {
            $parent_group = get_posts(array(
                'post_type' => 'acf-field-group',
                'name' =>  $field['parent'],
                'post_status' => 'any',
                'posts_per_page' => 1
            ));


            if (!empty($parent_group)) {
                $field['parent'] = $parent_group[0]->ID;
            } else {
                // Try direct key match
                $parent_group = get_posts(array(
                    'post_type' => 'acf-field-group',
                    'posts_per_page' => 1,
                    'meta_key' => '_acf_key',
                    'meta_value' => $field['parent']
                ));

                if (!empty($parent_group)) {
                    $field['parent'] = $parent_group[0]->ID;
                } else {
                    return false; // Parent group not found
                }
            }
        }

        // Create post
        $post_data = array(
            'post_status' => 'publish',
            'post_type' => 'acf-field',
            'post_title' => $field['label'],
            'post_name' => $field['name'],
            'post_excerpt' => $field['name'],
            'post_parent' => $field['parent'],
            'menu_order' => $field['menu_order'],
            'post_content' => maybe_serialize($field)
        );

        // Insert the field post
        $field_id = wp_insert_post($post_data);

        if (!$field_id || is_wp_error($field_id)) {
            return false;
        }

        // Update field with ID
        $field['ID'] = $field_id;

        // Clear any field group cache
        wp_cache_delete("acf_get_field_group_posts");

        return $field;
    }

}

// Function wrapper for backward compatibility
function acf_add_field($field) {
    return PP_ACF_Field_Handler::get_instance()->add_field($field);
}
