<?php
/**
 * ACF Choice Sync Class
 *
 * @package portfolio_pack
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Class to handle ACF choice synchronization
 */
class ACF_Choice_Sync {
    /**
     * Instance of the class
     *
     * @var ACF_Choice_Sync
     */
    private static $instance = null;

    /**
     * Constructor
     */
    private function __construct() {
        // Add custom setting to ACF field for JSON Sync
        add_action('acf/render_field_settings', array($this, 'render_field_settings'));

        // Load terms from JSON if the setting is enabled
        add_filter('acf/load_field', array($this, 'load_terms_from_json'));

        // Hook into ACF save_post
        add_action('acf/save_post', array($this, 'save_post'));

    }

    /**
     * Get instance of the class
     *
     * @return ACF_Choice_Sync
     */
    public static function get_instance() {
        if (self::$instance === null) {
            // Check if another instance exists before creating a new one
            global $acf_choice_sync_instance;

            if (isset($acf_choice_sync_instance) && $acf_choice_sync_instance instanceof ACF_Choice_Sync) {
                self::$instance = $acf_choice_sync_instance;

            } else {
                self::$instance = new self();
                $acf_choice_sync_instance = self::$instance;

            }
        }

        return self::$instance;
    }

    /**
     * Add a custom setting to ACF field for JSON Sync (only for select fields)
     *
     * @param array $field The field array
     */
    public function render_field_settings($field) {
        // Restrict to select fields
        if ($field['type'] !== 'select') {
            return; // Exit if not a select field
        }

        acf_render_field_setting($field, array(
            'label'        => __('Sync Choices', 'portfolio_pack'),
            'instructions' => 'Enable to sync this field choices for quick select on other posts.',
            'name'         => 'json_choice_sync',
            'type'         => 'true_false',
            'ui'           => 1,
        ), true); // Must pass true for global settings
    }

    /**
     * Save terms to JSON if the setting is enabled
     *
     * @param array $field The field array
     * @param int $post_id The post ID
     */
    public function save_terms_to_json($field, $post_id) {
        if (!isset($field['json_choice_sync']) || !$field['json_choice_sync']) {
            return; // Exit if JSON sync is not enabled
        }

        $post_type = get_post_type($post_id);
        $field_name = $field['name'];

        // Get all posts of the specified type
        $args = array(
            'post_type' => $post_type,
            'posts_per_page' => -1,
            'fields' => 'ids',
        );
        $posts = get_posts($args);

        $terms = array();

        // Loop through each post and get the field values
        foreach ($posts as $post_id) {
            $previous_terms = get_field($field_name, $post_id);
            if ($previous_terms) {
                if (is_array($previous_terms)) {
                    $terms = array_merge($terms, $previous_terms);
                } else {
                    $terms[] = $previous_terms;
                }
            }
        }

        // Remove duplicates and save to JSON in acf_select_choices folder
        $terms = array_unique($terms);

        // Use wp_upload_dir instead of get_stylesheet_directory for better compatibility
        $upload_dir = wp_upload_dir();
        $json_directory = $upload_dir['basedir'] . '/acf_select_choices';

        // Create the directory if it doesn't exist
        if (!file_exists($json_directory)) {
            wp_mkdir_p($json_directory);
        }

        $json_file_path = $json_directory . '/' . $field_name . '.json';

        // Use wp_json_encode instead of json_encode for better compatibility
        $json_data = wp_json_encode($terms);

        // Check if json_encode was successful
        if ($json_data === false) {
            return;
        }

        // Write to file
        $result = file_put_contents($json_file_path, $json_data);
    }

    /**
     * Load terms from JSON if the setting is enabled
     *
     * @param array $field The field array
     * @return array The modified field array
     */
    public function load_terms_from_json($field) {
        if (!isset($field['json_choice_sync']) || !$field['json_choice_sync']) {
            return $field; // Exit if JSON sync is not enabled
        }

        $field['choices'] = array();

        // Use wp_upload_dir instead of get_stylesheet_directory for better compatibility
        $upload_dir = wp_upload_dir();
        $json_file_path = $upload_dir['basedir'] . '/acf_select_choices/' . $field['name'] . '.json';

        // Check if the file exists and read its contents
        if (file_exists($json_file_path)) {
            $json_data = file_get_contents($json_file_path);

            // Check if file_get_contents was successful
            if ($json_data === false) {
                return $field;
            }

            // Decode JSON
            $terms = json_decode($json_data, true);

            // Check if json_decode was successful
            if ($terms === null) {
                return $field;
            }

            if (is_array($terms)) {
                foreach ($terms as $term) {
                    $field['choices'][$term] = $term; // Add choices
                }
            }
        }

        return $field;
    }

    /**
     * Hook into ACF save_post
     *
     * @param int $post_id The post ID
     */
    public function save_post($post_id) {
        $fields = get_field_objects($post_id);
        if ($fields) {
            foreach ($fields as $field) {
                $this->save_terms_to_json($field, $post_id);
            }
        }
    }
}

// Initialize the class
add_action('init', function() {
    ACF_Choice_Sync::get_instance();
});
