<?php
/**
 * ACF Helper Functions
 *
 * @package portfolio_pack
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Create an empty ACF field group that can be edited in the ACF UI
 *
 * @param string $key     The field group key
 * @param string $title   The field group title
 * @param array  $location The location rules for the field group
 * @param bool   $active   Whether the field group is active or not
 * @return int|WP_Error   The post ID of the created field group or WP_Error
 */
function pp_create_acf_field_group($key, $title, $location = array(), $active = true) {
    // Check if ACF is active
    if (!function_exists('acf_get_field_group')) {
        return new WP_Error('acf_not_active', __('Advanced Custom Fields is not active.', 'portfolio_pack'));
    }

    // Check if field group already exists
    $existing_group = acf_get_field_group($key);
    if ($existing_group) {
        return $existing_group['ID'];
    }

    // Set default location if not provided
    if (empty($location)) {
        $location = array(
            array(
                array(
                    'param' => 'post_type',
                    'operator' => '==',
                    'value' => 'portfolio',
                ),
            ),
        );
    }

    // Create field group
    $field_group = array(
        'key' => $key,
        'title' => $title,
        'fields' => array(),
        'location' => $location,
        'menu_order' => 0,
        'position' => 'normal',
        'style' => 'default',
        'label_placement' => 'top',
        'instruction_placement' => 'label',
        'hide_on_screen' => '',
        'active' => $active,
        'description' => '',
        'show_in_rest' => 1,
    );

    // Check if acf_import_field_group function exists or if acf_add_local_field_group exists
    $use_import_field_group = function_exists('acf_import_field_group');
    $use_add_local_field_group = function_exists('acf_add_local_field_group');

    if (!$use_import_field_group && !$use_add_local_field_group) {
        return new WP_Error('acf_field_group_functions_not_available', __('ACF field group functions are not available.', 'portfolio_pack'));
    }

    // Save field group
    if ($use_import_field_group) {
        return acf_import_field_group($field_group);
    } else {
        // When using acf_add_local_field_group, we need to return something meaningful
        // since it doesn't return an ID like acf_import_field_group does
        acf_add_local_field_group($field_group);

        // Try to get the field group to confirm it was added
        if (function_exists('acf_get_field_group')) {
            $added_group = acf_get_field_group($key);
            if ($added_group) {
                return isset($added_group['ID']) ? $added_group['ID'] : true;
            }
        }

        // Return true if we can't confirm, but no error occurred
        return true;
    }
}

/**
 * Add a field to an ACF field group
 *
 * @param array $field       The field data
 * @param string $group_key  The field group key
 * @return bool|WP_Error     True on success, WP_Error on failure
 */
function pp_add_field_to_field_group($field, $group_key) {
    // Check if ACF is active
    if (!function_exists('acf_get_field_group')) {
        return new WP_Error('acf_not_active', __('Advanced Custom Fields is not active.', 'portfolio_pack'));
    }

    // Make sure the field has a key
    if (!isset($field['key'])) {
        $field['key'] = 'field_' . uniqid();
    }

    // Set the parent field group
    $field['parent'] = $group_key;

    // Try to use our custom field handler
    if (class_exists('PP_ACF_Field_Handler')) {
        $field_handler = PP_ACF_Field_Handler::get_instance();
        $result = $field_handler->add_field($field);

        if ($result) {
            return true;
        } else {
            return new WP_Error('field_add_error', __('Error adding field using custom handler.', 'portfolio_pack'));
        }
    }

    // If all else fails, try a direct database approach
    // Get the field group
    $field_group = acf_get_field_group($group_key);
    if (!$field_group) {
        return new WP_Error('group_not_found', __('Field group not found.', 'portfolio_pack'));
    }

    // Get existing fields
    $fields = acf_get_fields($field_group);
    if (!is_array($fields)) {
        $fields = array();
    }

    // Check if field already exists
    $field_exists = false;
    foreach ($fields as $i => $existing_field) {
        if ($existing_field['key'] === $field['key']) {
            // Update existing field
            $fields[$i] = $field;
            $field_exists = true;
            break;
        }
    }

    // Add new field if it doesn't exist
    if (!$field_exists) {
        $fields[] = $field;
    }

    // Update field group with new fields
    $field_group['fields'] = $fields;

    // Save field group
    if (function_exists('acf_update_field_group')) {
        acf_update_field_group($field_group);
        return true;
    }

    return new WP_Error('update_methods_not_available', __('No methods available to update fields.', 'portfolio_pack'));
}

/**
 * Import fields from JSON into an existing ACF field group
 *
 * @param string $json_string The JSON string containing the fields
 * @param string $group_key   The field group key to import into
 * @return bool|WP_Error      True on success, WP_Error on failure
 */
function pp_import_acf_fields_from_json($json_string, $group_key) {
    // Check if ACF is active
    if (!function_exists('acf_get_field_group')) {
        return new WP_Error('acf_not_active', __('Advanced Custom Fields is not active.', 'portfolio_pack'));
    }

    // Decode JSON
    $json_data = json_decode($json_string, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return new WP_Error('invalid_json', __('Invalid JSON format.', 'portfolio_pack'));
    }

    // Check if JSON contains fields
    if (!isset($json_data['fields']) || !is_array($json_data['fields'])) {
        return new WP_Error('no_fields', __('No fields found in JSON.', 'portfolio_pack'));
    }

    // Get existing field group
    $field_group = acf_get_field_group($group_key);
    if (!$field_group) {
        return new WP_Error('group_not_found', __('Field group not found.', 'portfolio_pack'));
    }

    // Get existing fields - handle potential null return
    $existing_fields = acf_get_fields($field_group);

    // Import new fields
    if (!empty($json_data['fields']) && is_array($json_data['fields'])) {
        foreach ($json_data['fields'] as $field) {
            // Use our custom function to add the field to the field group
            $result = pp_add_field_to_field_group($field, $field_group['key']);

            if (is_wp_error($result)) {
                // Continue with next field instead of crashing
                continue;
            }
        }
    }

    return true;
}
