<?php
/**
 * Plugin Name: Portfolio Pack
 * Plugin URI: https://example.com/portfolio-pack
 * Description: A WordPress plugin for managing and displaying portfolio items.
 * Version: 1.0.2
 * Author: Portfolio Pack Team
 * Author URI: https://example.com
 * Text Domain: portfolio-pack
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @package portfolio_pack
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('PORTFOLIO_PACK_VERSION', '1.0.0');
define('PORTFOLIO_PACK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PORTFOLIO_PACK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PORTFOLIO_PACK_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Plugin activation hook
 */
function portfolio_pack_activate() {
    // Flush rewrite rules after registering custom post types
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'portfolio_pack_activate');

/**
 * Plugin deactivation hook
 */
function portfolio_pack_deactivate() {
    // Flush rewrite rules to remove our custom post types
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'portfolio_pack_deactivate');

/**
 * Load plugin textdomain
 */
function portfolio_pack_load_textdomain() {
    load_plugin_textdomain('portfolio-pack', false, dirname(PORTFOLIO_PACK_PLUGIN_BASENAME) . '/languages');
}
add_action('plugins_loaded', 'portfolio_pack_load_textdomain');

// Include post types
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/post-types/portfolio.php';

// Include helpers
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/helpers/acf-helpers.php';
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/helpers/acf-add-field.php';

// Include ACF fields
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/acf-fields/portfolio-details-fields.php';
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/acf-fields/portfolio-fields.php';

// Include settings page
require_once PORTFOLIO_PACK_PLUGIN_DIR . 'includes/admin/settings-page.php';

/**
 * Check if ACF plugin is active
 */
function portfolio_pack_is_acf_active() {
    return class_exists('ACF');
}

/**
 * Admin notice for missing ACF dependency
 */
function portfolio_pack_acf_missing_notice() {
    ?>
    <div class="notice notice-error">
        <p><?php _e('Portfolio Pack requires Advanced Custom Fields plugin to be installed and activated.', 'portfolio-pack'); ?></p>
    </div>
    <?php
}

/**
 * Check for plugin dependencies
 */
function portfolio_pack_check_dependencies() {
    if (!portfolio_pack_is_acf_active()) {
        add_action('admin_notices', 'portfolio_pack_acf_missing_notice');
        return false;
    }
    return true;
}
add_action('admin_init', 'portfolio_pack_check_dependencies');

// Load enabled modules
function portfolio_pack_load_modules() {
    // Check dependencies before loading modules
    if (!portfolio_pack_check_dependencies()) {
        return;
    }
    
    // Get available modules
    $modules_dir = PORTFOLIO_PACK_PLUGIN_DIR . 'includes/modules';

    // Check if modules directory exists
    if (!is_dir($modules_dir)) {
        return;
    }

    // Get all module directories
    $module_dirs = glob($modules_dir . '/*', GLOB_ONLYDIR);

    foreach ($module_dirs as $module_dir) {
        $module_id = basename($module_dir);
        // Check if module is enabled
        if (function_exists('pp_is_module_enabled') && !pp_is_module_enabled($module_id)) {
            continue;
        }

        // Check if module init file exists
        $module_file = $module_dir . '/module.php';
        if (file_exists($module_file)) {
            include_once($module_file);
        }
    }
}

// Load modules when WordPress is ready
add_action('init', 'portfolio_pack_load_modules', 20);

// Add settings link on plugin page
function portfolio_pack_settings_link($links) {
    $settings_link = '<a href="admin.php?page=portfolio-pack-settings">' . __('Settings', 'portfolio-pack') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . PORTFOLIO_PACK_PLUGIN_BASENAME, 'portfolio_pack_settings_link');
