/**
 * Service Area Tooltips - Admin Script
 * Adds "Copy Shortcode" buttons to the repeater rows in the admin interface
 * with a popup shortcode builder
 */
jQuery(document).ready(function($) {
    // Get default values from settings (passed via wp_localize_script)
    const defaultHighlight = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.highlight : 'none';
    const defaultHeadingColor = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.headingColor : '#000000';
    const defaultBgColor = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.bgColor : '#ffffff';
    const defaultSuburbColor = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.suburbColor : '#000000';
    const defaultWidth = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.width : '250';
    const defaultFontSize = typeof serviceAreaDefaults !== 'undefined' ? serviceAreaDefaults.fontSize : '14';
    
    // Add modal HTML to the page
    $('body').append(`
        <div id="shortcode-builder-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.7); z-index:99999;">
            <div style="position:relative; width:600px; max-width:90%; margin:50px auto; background:#fff; padding:20px; border-radius:5px; box-shadow:0 0 10px rgba(0,0,0,0.3);">
                <button id="close-shortcode-modal" style="position:absolute; top:10px; right:10px; background:none; border:none; font-size:20px; cursor:pointer;">&times;</button>
                <h2>Shortcode Builder</h2>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Display Text (optional):</label>
                    <input type="text" id="shortcode-display-text" style="width:100%;" placeholder="Leave empty to use area name">
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Highlight Style:</label>
                    <select id="shortcode-highlight" style="width:100%;">
                        <option value="none" ${defaultHighlight === 'none' ? 'selected' : ''}>None</option>
                        <option value="underline" ${defaultHighlight === 'underline' ? 'selected' : ''}>Underline</option>
                        <option value="icon" ${defaultHighlight === 'icon' ? 'selected' : ''}>Question Mark Icon</option>
                    </select>
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Heading Color:</label>
                    <input type="color" id="shortcode-heading-color" value="${defaultHeadingColor}">
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Background Color:</label>
                    <input type="color" id="shortcode-bg-color" value="${defaultBgColor}">
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Suburb Text Color:</label>
                    <input type="color" id="shortcode-suburb-color" value="${defaultSuburbColor}">
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Tooltip Width:</label>
                    <div style="display:flex; align-items:center;">
                        <input type="text" id="shortcode-width" value="${defaultWidth}" style="flex-grow:1;">
                        <span style="margin-left:5px; padding:5px; background:#f5f5f5; border:1px solid #ddd; border-radius:3px;">px</span>
                    </div>
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Font Size:</label>
                    <div style="display:flex; align-items:center;">
                        <input type="text" id="shortcode-font-size" value="${defaultFontSize}" style="flex-grow:1;">
                        <span style="margin-left:5px; padding:5px; background:#f5f5f5; border:1px solid #ddd; border-radius:3px;">px</span>
                    </div>
                </div>
                <div style="margin-top:20px;">
                    <div style="background:#f5f5f5; padding:10px; border-radius:3px; margin-bottom:15px;">
                        <label style="display:block; margin-bottom:5px;">Generated Shortcode:</label>
                        <code id="generated-shortcode" style="display:block; padding:10px; background:#fff; border:1px solid #ddd;"></code>
                    </div>
                    <button id="copy-generated-shortcode" class="button button-primary" style="width:100%;">Copy Shortcode & Close</button>
                </div>
            </div>
        </div>
    `);

    // Current row ID for the modal
    let currentRowId = 0;
    let currentAreaName = '';

    // Function to update the generated shortcode
    function updateGeneratedShortcode() {
        const displayText = $('#shortcode-display-text').val();
        const highlight = $('#shortcode-highlight').val();
        const headingColor = $('#shortcode-heading-color').val();
        const bgColor = $('#shortcode-bg-color').val();
        const suburbColor = $('#shortcode-suburb-color').val();
        const width = $('#shortcode-width').val();
        const fontSize = $('#shortcode-font-size').val();

        // Start with the basic shortcode
        let shortcode = `[service_area id="${currentRowId}"`;
        
        // Add optional parameters if they're not the defaults
        if (highlight !== defaultHighlight) {
            shortcode += ` highlight="${highlight}"`;
        }
        
        if (headingColor !== defaultHeadingColor) {
            shortcode += ` heading_color="${headingColor}"`;
        }
        
        if (bgColor !== defaultBgColor) {
            shortcode += ` bg_color="${bgColor}"`;
        }
        
        if (suburbColor !== defaultSuburbColor) {
            shortcode += ` suburb_color="${suburbColor}"`;
        }
        
        if (width !== defaultWidth) {
            shortcode += ` width="${width}px"`;
        }
        
        if (fontSize !== defaultFontSize) {
            shortcode += ` font_size="${fontSize}px"`;
        }
        
        // Close the opening tag
        shortcode += ']';
        
        // If display text is provided, add it and the closing tag
        if (displayText) {
            shortcode += displayText + '[/service_area]';
        }
        // Otherwise, we're done (shorthand format)
        
        // Update the display
        $('#generated-shortcode').text(shortcode);
    }

    // Function to open the modal
    function openShortcodeModal(rowId, areaName) {
        currentRowId = rowId;
        currentAreaName = areaName;
        
        // Reset form fields to defaults from settings
        $('#shortcode-display-text').val('');
        $('#shortcode-highlight').val(defaultHighlight);
        $('#shortcode-heading-color').val(defaultHeadingColor);
        $('#shortcode-bg-color').val(defaultBgColor);
        $('#shortcode-suburb-color').val(defaultSuburbColor);
        $('#shortcode-width').val(defaultWidth);
        $('#shortcode-font-size').val(defaultFontSize);
        
        // Update the generated shortcode
        updateGeneratedShortcode();
        
        // Show the modal
        $('#shortcode-builder-modal').show();
    }

    // Close modal when clicking the close button or outside the modal
    $('#close-shortcode-modal').on('click', function() {
        $('#shortcode-builder-modal').hide();
    });
    
    $('#shortcode-builder-modal').on('click', function(e) {
        if (e.target === this) {
            $(this).hide();
        }
    });

    // Update shortcode when form fields change
    $('#shortcode-display-text, #shortcode-highlight, #shortcode-heading-color, #shortcode-bg-color, #shortcode-suburb-color, #shortcode-width, #shortcode-font-size').on('change input', updateGeneratedShortcode);

    // Copy generated shortcode
    $('#copy-generated-shortcode').on('click', function() {
        const shortcode = $('#generated-shortcode').text();
        
        // Copy to clipboard
        navigator.clipboard.writeText(shortcode).then(function() {
            // Visual feedback
            const originalText = $('#copy-generated-shortcode').text();
            $('#copy-generated-shortcode').text('Copied!');
            
            setTimeout(function() {
                $('#copy-generated-shortcode').text(originalText);
                $('#shortcode-builder-modal').hide();
            }, 1000);
        }).catch(function() {
            // Fallback for older browsers
            const textarea = document.createElement('textarea');
            textarea.value = shortcode;
            textarea.style.position = 'fixed';
            document.body.appendChild(textarea);
            textarea.focus();
            textarea.select();
            
            try {
                document.execCommand('copy');
                const originalText = $('#copy-generated-shortcode').text();
                $('#copy-generated-shortcode').text('Copied!');
                
                setTimeout(function() {
                    $('#copy-generated-shortcode').text(originalText);
                    $('#shortcode-builder-modal').hide();
                }, 1000);
            } catch (err) {
                console.error('Failed to copy shortcode', err);
                alert('Failed to copy shortcode. Please try again.');
            }
            
            document.body.removeChild(textarea);
        });
    });

    // Function to add copy buttons to each repeater row
    function addCopyButtons() {
        // Target each repeater row
        $('.acf-field-repeater[data-name="areas"] .acf-row:not(.acf-clone)').each(function(index) {
            const $row = $(this);
            const rowId = index + 1; // Row IDs start from 1
            
            // Check if button already exists
            if ($row.find('.copy-shortcode-btn').length === 0) {
                // Create button
                const $button = $('<button>', {
                    'class': 'button button-secondary copy-shortcode-btn',
                    'text': 'Generate Shortcode',
                    'data-id': rowId,
                    'style': 'margin: 10px 0;'
                });
                
                // Get the area field
                const $areaField = $row.find('.acf-field[data-name="area"]');
                const areaName = $areaField.find('input[type="text"]').val();
                
                // Add button before the area field
                $areaField.prepend($button);
                
                // Add click event
                $button.on('click', function(e) {
                    e.preventDefault();
                    
                    // Get the current area name (it might have changed)
                    const currentAreaName = $areaField.find('input[type="text"]').val();
                    
                    // Open the shortcode builder modal
                    openShortcodeModal(rowId, currentAreaName);
                });
            }
        });
    }
    
    // Initial run
    addCopyButtons();
    
    // Re-run when new rows are added or when layout changes
    $(document).on('click', '.acf-repeater .acf-actions .acf-button', function() {
        setTimeout(addCopyButtons, 500);
    });
    
    // Also run when ACF fields are ready
    acf.add_action('ready', addCopyButtons);
    acf.add_action('append', addCopyButtons);
    acf.add_action('load', addCopyButtons);
    
    // Bulk Suburb Entry Functionality
    function addBulkSuburbEntryButtons() {
        // Target each suburbs field in repeater rows
        $('.acf-field-repeater[data-name="areas"] .acf-row:not(.acf-clone)').each(function() {
            const $row = $(this);
            const $suburbsField = $row.find('.acf-field[data-name="suburbs"]');
            
            // Check if bulk entry button already exists
            if ($suburbsField.length > 0 && $suburbsField.find('.bulk-suburb-entry-btn').length === 0) {
                // Create bulk entry button
                const $bulkButton = $('<button>', {
                    'class': 'button button-secondary bulk-suburb-entry-btn',
                    'text': 'Bulk Add Suburbs',
                    'style': 'margin: 5px 0; margin-right: 10px;'
                });
                
                // Insert button before the select field
                $suburbsField.find('.acf-input').prepend($bulkButton);
                
                // Add click event for bulk entry
                $bulkButton.on('click', function(e) {
                    e.preventDefault();
                    openBulkSuburbModal($suburbsField);
                });
            }
        });
    }
    
    // Add bulk suburb modal HTML
    $('body').append(`
        <div id="bulk-suburb-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.7); z-index:99999;">
            <div style="position:relative; width:500px; max-width:90%; margin:50px auto; background:#fff; padding:20px; border-radius:5px; box-shadow:0 0 10px rgba(0,0,0,0.3);">
                <button id="close-bulk-suburb-modal" style="position:absolute; top:10px; right:10px; background:none; border:none; font-size:20px; cursor:pointer;">&times;</button>
                <h2>Bulk Add Suburbs</h2>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">Enter suburbs (one per line or comma-separated):</label>
                    <textarea id="bulk-suburb-textarea" rows="10" style="width:100%; resize:vertical;" placeholder="Example:
Manly
Dee Why
Brookvale

Or comma-separated:
Manly, Dee Why, Brookvale"></textarea>
                </div>
                <div style="margin-bottom:15px;">
                    <label style="display:block; margin-bottom:5px;">
                        <input type="checkbox" id="clear-existing-suburbs"> Clear existing suburbs before adding new ones
                    </label>
                </div>
                <div style="margin-top:20px;">
                    <button id="add-bulk-suburbs" class="button button-primary" style="margin-right:10px;">Add Suburbs</button>
                    <button id="cancel-bulk-suburbs" class="button">Cancel</button>
                </div>
            </div>
        </div>
    `);
    
    let currentSuburbsField = null;
    
    // Function to open bulk suburb modal
    function openBulkSuburbModal($field) {
        currentSuburbsField = $field;
        $('#bulk-suburb-textarea').val('');
        $('#clear-existing-suburbs').prop('checked', false);
        $('#bulk-suburb-modal').show();
    }
    
    // Close bulk suburb modal
    $('#close-bulk-suburb-modal, #cancel-bulk-suburbs').on('click', function() {
        $('#bulk-suburb-modal').hide();
        currentSuburbsField = null;
    });
    
    $('#bulk-suburb-modal').on('click', function(e) {
        if (e.target === this) {
            $(this).hide();
            currentSuburbsField = null;
        }
    });
    
    // Add bulk suburbs functionality
    $('#add-bulk-suburbs').on('click', function() {
        if (!currentSuburbsField) return;
        
        const textareaValue = $('#bulk-suburb-textarea').val().trim();
        if (!textareaValue) {
            alert('Please enter some suburbs to add.');
            return;
        }
        
        // Parse suburbs from textarea (support both line breaks and commas)
        let suburbs = [];
        if (textareaValue.includes('\n')) {
            // Line-separated
            suburbs = textareaValue.split('\n').map(s => s.trim()).filter(s => s.length > 0);
        } else {
            // Comma-separated
            suburbs = textareaValue.split(',').map(s => s.trim()).filter(s => s.length > 0);
        }
        
        if (suburbs.length === 0) {
            alert('No valid suburbs found. Please check your input.');
            return;
        }
        
        // Get the ACF select field
        const $select = currentSuburbsField.find('select');
        if ($select.length === 0) {
            alert('Could not find the suburbs select field.');
            return;
        }
        
        // Clear existing values if requested
        if ($('#clear-existing-suburbs').prop('checked')) {
            $select.val(null).trigger('change');
        }
        
        // Get current selected values
        let currentValues = $select.val() || [];
        if (!Array.isArray(currentValues)) {
            currentValues = currentValues ? [currentValues] : [];
        }
        
        // Add new suburbs
        suburbs.forEach(function(suburb) {
            // Check if option already exists
            let optionExists = false;
            let existingValue = null;
            $select.find('option').each(function() {
                if ($(this).val().toLowerCase() === suburb.toLowerCase()) {
                    optionExists = true;
                    existingValue = $(this).val();
                    return false; // break
                }
            });
            
            if (optionExists) {
                // Option exists, just make sure it's selected
                if (!currentValues.includes(existingValue)) {
                    currentValues.push(existingValue);
                }
            } else {
                // Option doesn't exist, create it and add to values
                const $newOption = $('<option></option>')
                    .attr('value', suburb)
                    .text(suburb);
                $select.append($newOption);
                
                if (!currentValues.includes(suburb)) {
                    currentValues.push(suburb);
                }
            }
        });
        
        // Update the select field with new values
        $select.val(currentValues).trigger('change');
        
        // Show success message
        const addedCount = suburbs.length;
        alert(`Successfully added ${addedCount} suburb${addedCount !== 1 ? 's' : ''}!`);
        
        // Close modal
        $('#bulk-suburb-modal').hide();
        currentSuburbsField = null;
    });
    
    // Add bulk suburb buttons when page loads and when rows are added
    addBulkSuburbEntryButtons();
    
    // Re-run when new rows are added
    $(document).on('click', '.acf-repeater .acf-actions .acf-button', function() {
        setTimeout(addBulkSuburbEntryButtons, 500);
    });
    
    // Also run when ACF fields are ready
    acf.add_action('ready', addBulkSuburbEntryButtons);
    acf.add_action('append', addBulkSuburbEntryButtons);
    acf.add_action('load', addBulkSuburbEntryButtons);
});