<?php
/**
 * Plugin Name: Service Area Tooltips
 * Plugin URI: https://makelabs.com.au/
 * Description: Adds a shortcode [service_area area="Area Name"] that displays a tooltip with suburb microdata for SEO.
 * Version: 1.0.12
 * Author: makeweb
 * Author URI: https://makelabs.com.au/
 * Text Domain: service-area-tooltips
 * Domain Path: /languages
 * Requires at least: 6.8
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// Register ACF fields and options page
add_action( 'acf/include_fields', function() {
    if ( ! function_exists( 'acf_add_local_field_group' ) ) {
        return;
    }

    acf_add_options_page(array(
        'menu_slug' => 'service-areas',
        'page_title' => 'Service Areas',
        'active' => true,
        'menu_title' => 'Service Areas',
        'capability' => 'edit_posts',
        'position' => '',
        'icon_url' => 'dashicons-admin-site-alt2',
        'redirect' => true,
        'post_id' => 'options',
        'autoload' => false,
        'update_button' => 'Update',
        'updated_message' => 'Areas Updated',
        'acfe_autosync' => array(),
    ));

    // Combined Service Areas field group with tabs
    acf_add_local_field_group( array(
        'key' => 'group_service_areas_combined',
        'title' => 'Service Areas',
        'fields' => array(
            // Tab: Areas
            array(
                'key' => 'field_tab_areas',
                'label' => 'Areas',
                'name' => 'tab_areas',
                'type' => 'tab',
                'placement' => 'top',
            ),
            array(
                'key' => 'field_672abf8cbf04c',
                'label' => 'Areas',
                'name' => 'areas',
                'type' => 'repeater',
                'instructions' => 'Add service areas and their associated suburbs. <br/> Each area can be displayed on your site using the shortcode [service_area id="ROW_NUMBER"]. Use the "Copy Shortcode" button to quickly generate the shortcode.',
                'layout' => 'block',
                'button_label' => 'Add Area',
                'sub_fields' => array(
                    array(
                        'key' => 'field_672ac031bf04d',
                        'label' => 'Area',
                        'name' => 'area',
                        'type' => 'text',
                        'instructions' => 'Enter the name of the service area (e.g., "Northern Beaches", "Eastern Suburbs"). This will be displayed in the tooltip and can be used as the default display text in the shortcode.',
                        'required' => 1,
                    ),
                    array(
                        'key' => 'field_67f5d3e929032',
                        'label' => 'Popup heading',
                        'name' => 'popup_heading',
                        'type' => 'text',
                        'instructions' => 'Custom heading for the tooltip popup. You can use {area} as a merge tag to include the area name in your heading. If left empty, a default heading will be generated.',
                        'placeholder' => 'We work in the following areas in {area}',
                    ),
                    array(
                        'key' => 'field_672ac037bf04e',
                        'label' => 'Suburb(s)',
                        'name' => 'suburbs',
                        'type' => 'select',
                        'instructions' => 'Select or enter suburbs that belong to this service area. These will be displayed in the tooltip and included in the microdata for SEO.',
                        'multiple' => 1,
                        'allow_custom' => 1,
                        'ui' => 1,
                        'return_format' => 'value',
                    )
                )
            ),
            
            // Tab: How to Use
            array(
                'key' => 'field_tab_how_to_use',
                'label' => 'How to Use',
                'name' => 'tab_how_to_use',
                'type' => 'tab',
                'placement' => 'top',
            ),
            array(
                'key' => 'field_shortcode_instructions',
                'label' => 'How to Use the Shortcode',
                'name' => 'shortcode_instructions',
                'type' => 'message',
                'message' => '<h3>Basic Usage</h3>
                <p>Use the shortcode <code>[service_area id="1"]</code> or <code>[service_area id="1"]Custom Text[/service_area]</code> to display a tooltip with suburb information.</p>
                
                <h3>Available Parameters</h3>
                <ul>
                    <li><strong>id</strong> (required) - The row number of the area in the repeater (starts from 1)</li>
                    <li><strong>area</strong> (deprecated) - The name of the area (kept for backward compatibility)</li>
                    <li><strong>highlight</strong> - How to highlight the area text (options: none, underline, icon) - default: none</li>
                    <li><strong>heading_color</strong> - Colour of the tooltip heading - default: #000000</li>
                    <li><strong>bg_color</strong> - Background colour of the tooltip - default: #ffffff</li>
                    <li><strong>suburb_color</strong> - Colour of the suburb text - default: #000000</li>
                    <li><strong>width</strong> - Width of the tooltip - default: 250px</li>
                    <li><strong>font_size</strong> - Font size of the tooltip content - default: 14px</li>
                </ul>
                
                <h3>Content Between Tags</h3>
                <p>You can now include content between the opening and closing shortcode tags. This content will be displayed instead of the area name:</p>
                <p><code>[service_area id="1"]Sydney Region[/service_area]</code></p>
                <p>If no content is provided, the area name will be displayed as before.</p>
                
                <h3>Examples</h3>
                <p><code>[service_area id="1"]</code> - Basic usage with row ID</p>
                <p><code>[service_area id="1"]Northern Beaches[/service_area]</code> - With custom display text</p>
                <p><code>[service_area id="1" highlight="underline" heading_color="#0066cc" bg_color="#f5f5f5"]Custom Text[/service_area]</code> - With styling and custom text</p>
                <p><code>[service_area id="1" highlight="icon" width="300px"]</code> - With icon and custom width</p>
                <p><code>[service_area id="1" font_size="16px"]</code> - With custom font size</p>
                
                <h3>Copy Shortcode Button</h3>
                <p>Each area in the repeater now has a "Copy Shortcode" button that will copy the shortcode with the correct ID to your clipboard.</p>',
                'new_lines' => 'wpautop',
                'esc_html' => 0,
            ),
            
            // Tab: Settings
            array(
                'key' => 'field_tab_settings',
                'label' => 'Settings',
                'name' => 'tab_settings',
                'type' => 'tab',
                'placement' => 'top',
            ),
            array(
                'key' => 'field_highlight_default',
                'label' => 'Default Highlight Style',
                'name' => 'highlight_default',
                'type' => 'select',
                'instructions' => 'Choose how to highlight the area text by default. This can be overridden in the shortcode.',
                'choices' => array(
                    'none' => 'None',
                    'underline' => 'Underline',
                    'icon' => 'Question Mark Icon',
                ),
                'default_value' => 'none',
                'return_format' => 'value',
                'multiple' => 0,
                'ui' => 1,
            ),
            array(
                'key' => 'field_heading_color_default',
                'label' => 'Default Heading Colour',
                'name' => 'heading_color_default',
                'type' => 'color_picker',
                'instructions' => 'Choose the default colour for tooltip headings. This can be overridden in the shortcode.',
                'default_value' => '#000000',
            ),
            array(
                'key' => 'field_bg_color_default',
                'label' => 'Default Background Colour',
                'name' => 'bg_color_default',
                'type' => 'color_picker',
                'instructions' => 'Choose the default background colour for tooltips. This can be overridden in the shortcode.',
                'default_value' => '#ffffff',
            ),
            array(
                'key' => 'field_suburb_color_default',
                'label' => 'Default Suburb Text Colour',
                'name' => 'suburb_color_default',
                'type' => 'color_picker',
                'instructions' => 'Choose the default colour for suburb text in tooltips. This can be overridden in the shortcode.',
                'default_value' => '#000000',
            ),
            array(
                'key' => 'field_width_default',
                'label' => 'Default Tooltip Width',
                'name' => 'width_default',
                'type' => 'text',
                'instructions' => 'Set the default width for tooltips (numeric value only). "px" will be automatically appended. This can be overridden in the shortcode.',
                'default_value' => '250',
                'append' => 'px',
            ),
            array(
                'key' => 'field_font_size_default',
                'label' => 'Default Font Size',
                'name' => 'font_size_default',
                'type' => 'text',
                'instructions' => 'Set the default font size for tooltip content (numeric value only). "px" will be automatically appended. This can be overridden in the shortcode.',
                'default_value' => '14',
                'append' => 'px',
            ),
        ),
        'location' => array(
            array(
                array(
                    'param' => 'options_page',
                    'operator' => '==',
                    'value' => 'service-areas',
                )
            )
        ),
        'menu_order' => 0,
        'position' => 'normal',
        'style' => 'default',
        'label_placement' => 'top',
        'instruction_placement' => 'label',
        'active' => true,
    ));
});

// Enqueue frontend assets
add_action('wp_enqueue_scripts', function() {
    wp_register_style('service-area-tooltips', plugin_dir_url(__FILE__) . 'assets/style.css');
    wp_register_script('service-area-tooltips', plugin_dir_url(__FILE__) . 'assets/script.js', [], false, true);
});

// Enqueue admin assets
add_action('admin_enqueue_scripts', function($hook) {
    // Only load on the Service Areas options page
    if ($hook != 'toplevel_page_service-areas') {
        return;
    }
    
    wp_enqueue_script('service-area-tooltips-admin', plugin_dir_url(__FILE__) . 'assets/admin-script.js', ['jquery', 'acf-input'], false, true);
    
    // Pass default values from settings to JavaScript
    $highlight_default = get_field('highlight_default', 'option') ?: 'none';
    $heading_color_default = get_field('heading_color_default', 'option') ?: '#000000';
    $bg_color_default = get_field('bg_color_default', 'option') ?: '#ffffff';
    $suburb_color_default = get_field('suburb_color_default', 'option') ?: '#000000';
    $width_default = get_field('width_default', 'option') ?: '250';
    $font_size_default = get_field('font_size_default', 'option') ?: '14';
    
    wp_localize_script('service-area-tooltips-admin', 'serviceAreaDefaults', [
        'highlight' => $highlight_default,
        'headingColor' => $heading_color_default,
        'bgColor' => $bg_color_default,
        'suburbColor' => $suburb_color_default,
        'width' => $width_default,
        'fontSize' => $font_size_default
    ]);
});

// Shortcode
add_shortcode('service_area', function($atts, $content = null) {
    wp_enqueue_style('service-area-tooltips');
    wp_enqueue_script('service-area-tooltips');

    // Get default values from settings
    $highlight_default = get_field('highlight_default', 'option') ?: 'none';
    $heading_color_default = get_field('heading_color_default', 'option') ?: '#000000';
    $bg_color_default = get_field('bg_color_default', 'option') ?: '#ffffff';
    $suburb_color_default = get_field('suburb_color_default', 'option') ?: '#000000';
    $width_default = (get_field('width_default', 'option') ?: '250') . 'px';
    $font_size_default = (get_field('font_size_default', 'option') ?: '14') . 'px';

    $atts = shortcode_atts([
        'id' => '',
        'area' => '', // Keep for backward compatibility
        'highlight' => $highlight_default,
        'heading_color' => $heading_color_default,
        'bg_color' => $bg_color_default,
        'suburb_color' => $suburb_color_default,
        'width' => $width_default,
        'font_size' => $font_size_default,
    ], $atts);

    // Check if we have an ID or area name
    $row_id = intval($atts['id']);
    $target_area = strtolower(trim($atts['area']));
    
    // If neither ID nor area is provided, return empty
    if (!$row_id && !$target_area) return '';

    // Get all areas from the repeater
    $areas = get_field('areas', 'option');
    
    // If no areas found, return empty
    if (!$areas || !is_array($areas)) return '';
    
    // Find the target row
    $target_row = null;
    
    if ($row_id > 0) {
        // Find by row ID (1-based index)
        if (isset($areas[$row_id - 1])) {
            $target_row = $areas[$row_id - 1];
        }
    } else if ($target_area) {
        // Find by area name (backward compatibility)
        foreach ($areas as $row) {
            if (strtolower($row['area']) === $target_area) {
                $target_row = $row;
                break;
            }
        }
    }
    
    // If target row found, display the tooltip
    if ($target_row) {
        $area = $target_row['area'];
        $popup_heading = $target_row['popup_heading'];
        $suburbs = $target_row['suburbs'];
        
        // Use content between tags if provided, otherwise use area name
        $display_text = !empty($content) ? $content : $area;
        
        // Process popup heading - replace {area} merge tag or use default
        if (!empty($popup_heading)) {
            $tooltip_heading = str_replace('{area}', $area, $popup_heading);
        } else {
            $tooltip_heading = "We work in the following areas in $area";
        }

        // Highlight class
        $highlight_class = 'highlight-' . $atts['highlight'];

        ob_start(); ?>
        <span class="service-area-tooltip-wrapper <?php echo esc_attr($highlight_class); ?>" data-tooltip-trigger>
            <span class="area-text"><?php echo esc_html($display_text); ?></span>
            <?php if ($atts['highlight'] === 'icon'): ?>
                <span class="tooltip-icon" aria-hidden="true">[?]</span>
            <?php endif; ?>
            <span class="tooltip-content" style="
                background:<?php echo esc_attr($atts['bg_color']); ?>;
                color:<?php echo esc_attr($atts['suburb_color']); ?>;
                width:<?php echo esc_attr($atts['width']); ?>;
                font-size:<?php echo esc_attr($atts['font_size']); ?>;
            ">
                <span itemscope itemtype="https://schema.org/Place">
                    <meta itemprop="name" content="<?php echo esc_attr($area); ?>">
                    <span class="tooltip-heading" style="color:<?php echo esc_attr($atts['heading_color']); ?>; display: block; margin-bottom: 5px;">
                        <?php echo esc_html($tooltip_heading); ?>
                    </span>
                    <span class="suburbs-list">
                    <?php if (!empty($suburbs)) : ?>
                        <?php foreach ($suburbs as $index => $suburb): ?>
                            <span itemprop="containsPlace" itemscope itemtype="https://schema.org/Place">
                                <meta itemprop="name" content="<?php echo esc_attr($suburb); ?>">
                                <span class="suburb-text" style="color:<?php echo esc_attr($atts['suburb_color']); ?>;"><?php echo esc_html($suburb); ?></span><?php if ($index < count($suburbs) - 1) echo ', '; ?>
                            </span>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <span class="no-suburbs-message">No suburbs listed.</span>
                    <?php endif; ?>
                    </span>
                </span>
            </span>
        </span>
        <?php
        return ob_get_clean();
    }

    return '';
});


////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if (class_exists('makeUpdate')) {
    try {
        $updater = new makeUpdate("9545", "yat.rkc3PVM9ykf4uqj", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}