<?php
/**
 * Plugin Name: Site Tutorials Manager
 * Description: Adds a Tutorials admin page to manage and view instructional videos.
 * Version: 1.0.7
 * Author: <a href="https://makeweb.com.au/">MakeWeb</a>
 * Author URI: https://makeweb.com.au/
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// Register the Tutorials menu page
add_action( 'admin_menu', function() {
	add_menu_page(
		'Tutorials',
		'Tutorials',
		'manage_options',
		'site-tutorials',
		'stm_render_tutorials_page',
		'dashicons-video-alt3',
		25
	);
});

// Add plugin action links
add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), function($links) {
	$admin_link = '<a href="' . admin_url('admin.php?page=site-tutorials&admin_mode=true') . '">Admin Mode</a>';
	array_unshift($links, $admin_link);
	return $links;
});

// Enqueue scripts & styles
add_action( 'admin_enqueue_scripts', function($hook) {
	if ($hook !== 'toplevel_page_site-tutorials') return;
	wp_enqueue_style('stm-admin', plugin_dir_url(__FILE__) . 'stm-admin.css', [], '1.3.1');
	wp_enqueue_script('stm-admin', plugin_dir_url(__FILE__) . 'stm-admin.js', ['jquery'], '1.3.1', true);
});

// Register the tutorials option
add_action('admin_init', function() {
	if (!get_option('stm_tutorials')) {
		update_option('stm_tutorials', []);
	}
});

// Helper function to get the embed URL
function stm_get_embed_url($url) {
	if (strpos($url, 'vimeo.com/') !== false) {
		if (preg_match('~/([0-9]+)$~', $url, $m)) {
			// Added &autoplay=1 for instant play
			return 'https://player.vimeo.com/video/' . $m[1] . '?dnt=1&autoplay=1';
		}
	} elseif (strpos($url, 'youtube.com/watch') !== false || strpos($url, 'youtu.be/') !== false) {
		if (preg_match('~(v=|youtu\.be/)([A-Za-z0-9_-]+)~', $url, $m)) {
			// Added &autoplay=1 for instant play
			return 'https://www.youtube.com/embed/' . $m[2] . '?modestbranding=1&rel=0&autoplay=1';
		}
	}
	return $url; // fallback
}

// Render Tutorials Page
function stm_render_tutorials_page() {
	$tutorials = get_option('stm_tutorials', []);
	
	// Check for Admin Mode via query string
	$is_admin_mode = isset($_GET['admin_mode']) && $_GET['admin_mode'] === 'true';

	// Handle SAVE/UPDATE tutorial
	if (isset($_POST['stm_save_tutorial'])) {
		check_admin_referer('stm_save_tutorial');
		$id = isset($_POST['tutorial_id']) && $_POST['tutorial_id'] ? intval($_POST['tutorial_id']) : time();
		$tutorials[$id] = [
			'title' => sanitize_text_field($_POST['title']),
			'description' => sanitize_textarea_field($_POST['description']),
			'video_url' => esc_url_raw($_POST['video_url'])
		];
		update_option('stm_tutorials', $tutorials);
		// Redirect back to admin mode after save to maintain view
		$redirect_url = admin_url('admin.php?page=site-tutorials&admin_mode=true');
		wp_safe_redirect($redirect_url);
		exit;
	}

	// Handle DELETE tutorial
	if ($is_admin_mode && isset($_GET['delete'])) {
		$id = intval($_GET['delete']);
		unset($tutorials[$id]);
		update_option('stm_tutorials', $tutorials);
		// Redirect back to admin mode after delete
		$redirect_url = admin_url('admin.php?page=site-tutorials&admin_mode=true');
		wp_safe_redirect($redirect_url);
		exit;
	}

	$filter = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';

	// LIST VIEW
	?>
	<div class="wrap stm-admin-page stm-list-view <?php echo $is_admin_mode ? 'stm-admin-mode' : 'stm-user-mode'; ?>">
		<h1 class="wp-heading-inline">Video Tutorials</h1>

		<form method="get" class="stm-search-form">
			<input type="hidden" name="page" value="site-tutorials">
			<?php if ($is_admin_mode): ?>
				<input type="hidden" name="admin_mode" value="true">
			<?php endif; ?>
			<input type="search" id="stm-search-input" name="s" value="<?php echo esc_attr($filter); ?>" placeholder="Type to filter tutorials by title or description..." autocomplete="off">
			<?php if ($filter): ?>
				<a href="?page=site-tutorials<?php echo $is_admin_mode ? '&admin_mode=true' : ''; ?>" class="button button-link">Clear Filter</a>
			<?php endif; ?>
		</form>

		<div class="stm-table-container">
			<table class="wp-list-table widefat fixed striped stm-tutorials-table">
				<thead>
					<tr>
						<th class="column-title">Title</th>
						<th class="column-description">What You Will Learn</th>
						<th class="column-video">Actions</th>
					</tr>
				</thead>
				<tbody id="stm-tutorial-list">
				<?php
				$found = false;
				foreach ($tutorials as $id => $tutorial) {
					// Apply server-side filter only if explicitly set in the URL (e.g., after an explicit search or on load)
					if ($filter && stripos($tutorial['title'], $filter) === false && stripos($tutorial['description'], $filter) === false) continue;
					$found = true;
					
					$embed_url = stm_get_embed_url($tutorial['video_url']);
					?>
					<tr data-video-id="<?php echo $id; ?>" 
						data-title="<?php echo esc_attr($tutorial['title']); ?>" 
						data-description="<?php echo esc_attr($tutorial['description']); ?>" 
						data-original-url="<?php echo esc_attr($tutorial['video_url']); ?>">
						<td class="column-title"><strong><?php echo esc_html($tutorial['title']); ?></strong></td>
						<td class="column-description"><?php echo esc_html($tutorial['description']); ?></td>
						<td class="column-video">
							<button class="button button-primary play-video" data-video-url="<?php echo esc_url($embed_url); ?>"><span class="dashicons dashicons-controls-play"></span> Play Video</button>
							<?php if ($is_admin_mode): ?>
								<div class="row-actions">
									<span class="edit"><button class="button-link stm-edit-tutorial" data-id="<?php echo $id; ?>">Edit</button> | </span>
									<span class="delete"><a href="?page=site-tutorials&admin_mode=true&delete=<?php echo $id; ?>" onclick="return confirm('Are you sure you want to delete the tutorial: <?php echo esc_attr($tutorial['title']); ?>? This cannot be undone.');">Delete</a></span>
								</div>
							<?php endif; ?>
						</td>
					</tr>
					<?php
				}
				if (!$found) echo '<tr><td colspan="3" class="no-items">No tutorials have been added yet.</td></tr>';
				?>
				</tbody>
			</table>
			
			<?php if ($is_admin_mode): ?>
				<button id="stm-add-tutorial" class="page-title-action button button-primary">Add New Tutorial</button>
			<?php endif; ?>
		</div>
	</div>

	<div id="stm-video-modal" class="stm-modal stm-video-modal">
		<div class="stm-modal-content">
			<span class="stm-close">&times;</span>
			<div class="stm-video-wrapper">
				<iframe id="stm-video-frame" src="" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen></iframe>
			</div>
		</div>
	</div>

	<?php if ($is_admin_mode): ?>
	<div id="stm-edit-modal" class="stm-modal stm-edit-modal">
		<div class="stm-modal-content stm-form-content">
			<span class="stm-close">&times;</span>
			<h2 id="stm-form-title"></h2>
			<form method="post" id="stm-tutorial-form">
				<?php wp_nonce_field('stm_save_tutorial'); ?>
				<input type="hidden" name="tutorial_id" id="tutorial_id" value="">
				<table class="form-table">
					<tr class="stm-form-row">
						<th><label for="title">Tutorial Title</label></th>
						<td><input type="text" id="title" name="title" class="regular-text" placeholder="e.g., How to Edit a Page" required></td>
					</tr>
					<tr class="stm-form-row">
						<th><label for="description">Short Description</label></th>
						<td><textarea id="description" name="description" rows="4" class="large-text" placeholder="A brief summary of what this video teaches."></textarea></td>
					</tr>
					<tr class="stm-form-row">
						<th><label for="video_url">Video URL (YouTube/Vimeo)</label></th>
						<td><input type="url" id="video_url" name="video_url" class="regular-text" placeholder="e.g., https://www.youtube.com/watch?v=ABC1234" required></td>
					</tr>
				</table>
				<p class="submit">
					<button type="submit" name="stm_save_tutorial" class="button-primary button-hero">Save Tutorial</button>
				</p>
			</form>
		</div>
	</div>
	<?php endif; ?>
	<?php
}

////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if (class_exists('makeUpdate')) {
    try {
        $updater = new makeUpdate("9547", "tkt6rkp4auz*RHA_mab", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}