### Slipstream 3rd Party Integration Guide

Slipstream is designed to be extensible, allowing 3rd party plugins to register their own components (post types, settings, widgets) to appear within the Slipstream "Owner" dashboard.

#### The `slipstream_init` Hook

The recommended way to register components is via the `slipstream_init` hook. This hook provides the `Slipstream_Registry` instance.

```php
add_action( 'slipstream_init', function( $registry ) {
    // Your registration logic here
} );
```

---

#### 1. Registering an Existing Post Type

If your plugin already registers a Custom Post Type (CPT), you can easily "opt-in" to Slipstream. This will:
1. Grant the necessary capabilities to the **Owner** role.
2. Add a panel to the Slipstream dashboard.
3. Enable the **Draft/Published** toggle in the post list table.

```php
add_action( 'slipstream_init', function( $registry ) {
    $registry->register_post_type_for_owner( 'your_post_type', array(
        'name'        => 'Testimonials',
        'description' => 'Manage customer testimonials.',
        'icon'        => '<svg ...>...</svg>', // SVG string
        'icon_bg'     => '#10b981',             // Background color for icon
        'priority'    => 30,                    // Order in dashboard
    ) );
} );
```

---

#### 2. Registering a Custom Dashboard Panel

If you want to add a link to a custom settings page or any other URL:

```php
add_action( 'slipstream_init', function( $registry ) {
    $registry->register( 'panels', 'my_custom_panel', array(
        'name'        => 'My Settings',
        'description' => 'Configure your 3rd party plugin.',
        'icon'        => '<svg ...>...</svg>',
        'url'         => admin_url( 'admin.php?page=my-plugin-settings' ),
        'group'       => 'settings', // 'post_types', 'settings', 'user', or custom
        'priority'    => 50,
    ) );
} );
```

---

#### 3. Registering Dashboard Widgets

Widgets appear on the main dashboard of the Slipstream shell.

```php
add_action( 'slipstream_init', function( $registry ) {
    $registry->register( 'widgets', 'my_widget_slug', array(
        'title'    => 'Quick Stats',
        'content'  => function() {
             echo '<div class="p-4">Your custom HTML content here.</div>';
        },
        'width'    => 'half', // 'full' or 'half'
        'priority' => 10,
    ) );
} );
```

---

#### 4. Managing Permissions (Owner Role)

Slipstream uses an **Owner** role. You can dynamically grant capabilities to this role:

```php
add_action( 'slipstream_init', function( $registry ) {
    // Grant specific capability
    $registry->add_owner_capability( 'manage_my_plugin_options' );
    
    // Grant ALL capabilities for a post type (edit, publish, delete, etc.)
    $registry->add_owner_capability_all( 'my_post_type' );
    
    // Grant VIEW ONLY capabilities for a post type
    $registry->add_owner_capability_view( 'my_readonly_post_type' );
} );
```

---

#### 5. List Table Column Customization

You can add or remove columns from the post list table within the Slipstream shell:

```php
add_action( 'slipstream_init', function( $registry ) {
    // Remove columns
    $registry->remove_post_type_columns( 'my_post_type', array( 'date', 'author' ) );
    
    // Add columns
    $registry->add_post_type_columns( 
        'my_post_type', 
        array( 'my_custom_col' => 'Label' ), 
        function( $column_key, $post_id ) {
            if ( $column_key === 'my_custom_col' ) {
                echo get_post_meta( $post_id, 'some_key', true );
            }
        } 
    );
} );
```

---

#### 6. Registering Merge Tags

Merge tags allow users to insert dynamic data into content (e.g., in Divi modules or emails).

```php
add_action( 'slipstream_init', function( $registry ) {
    $registry->register( 'merge_tags', 'my_plugin_tag', array(
        'name'     => 'My Tag',
        'group'    => 'business_info', // or a custom group
        'callback' => function() {
            return get_option( 'my_plugin_value' );
        }
    ) );
} );
```

---

#### Best Practices

- **Use SVG Icons**: For the best look in the dashboard, use Tailwind-friendly SVGs (e.g., from Heroicons).
- **Check for Existence**: Always check if Slipstream is active before using its registry:
  ```php
  if ( class_exists( 'Slipstream_Registry' ) ) {
      // Register here
  }
  ```
- **Priority**: Use the `priority` argument to control where your panels and widgets appear.
