### Slipstream Framework: Technical Specification & AI Generation Guide

Slipstream is a "Headless-Lite" admin framework for WordPress. It reskins the WordPress backend into a modern, app-like interface using Tailwind CSS, specifically optimized for site owners (the `owner` role) while retaining full control for developers.

---

### 1. Core Architecture

#### The "Owner" Role
- Specialized role that bypasses the standard WP Dashboard.
- Redirected to `index.php` (hijacked by `Slipstream_Router`).
- Standard WP menus, admin bar, and notices are hidden.
- Interface is built from "Panels" registered in the `Slipstream_Registry`.

#### The Registry (`Slipstream_Registry`)
The singleton hub for all registrations. Key methods:
- `$registry->register( 'panels', $slug, $args )`: Adds a primary dashboard navigation item.
- `$registry->register( 'widgets', $slug, $args )`: Adds a dashboard module.
- `$registry->register( 'merge_tags', $slug, $args )`: Adds a dynamic placeholder.
- `$registry->register_content_field( $slug, $args )`: Registers a field that can be assigned to pages in Developer Config.
- `$registry->register_callback( $slug, $args )`: Registers a universal or context-specific callback function.
- `$registry->register_admin_config( $slug, $args )`: Adds a tab to the "Developer Config" page.
- `$registry->add_owner_capability( $cap )`: Grants WP capabilities to the Owner role.

---

### 2. Package Structure (The Blueprint)

All Slipstream features are delivered as "Packages". 
Location: `wp-content/slipstream-packages/{slug}/`

**Essential Files:**
1. `manifest.json`: Metadata for the package manager.
   ```json
   {
     "slug": "my-package",
     "name": "My Package",
     "description": "Short description.",
     "version": "1.0.0",
     "requires_core": "2.2.0"
   }
   ```
2. `index.php`: Entry point. Should only include other files.
3. `panels.php`: UI registrations (Dashboard Panels, Admin Config).
4. `settings.php`: Form handling and UI for settings screens.
5. `functions.php`: Core logic.

---

### 3. Registration Examples (The "Code to Generate")

#### Registering a Dashboard Panel
```php
$registry->register( 'panels', 'my-feature', [
    'name'        => 'My Feature',
    'description' => 'Manage your custom data.',
    'icon'        => '<svg>...</svg>', // SVG string (24x24). DASHICONS ARE NOT SUPPORTED.
    'icon_bg'     => '#6366f1',        // Background for the icon tile
    'icon_color'  => '#ffffff',
    'url'         => admin_url( 'admin.php?page=my-feature' ),
    'group'       => 'content',        // 'user', 'settings', or 'content'
    'priority'    => 20,
    'style'       => 'default',        // 'default', 'card', 'hero', 'toggle'
] );
```

#### Registering Developer Config (Admin Only)
These appear in **Slipstream > Developer Config**.
```php
$registry->register_admin_config( 'my-package', [
    'title'    => 'Integration Settings',
    'icon'     => '<svg>...</svg>',
    'callback' => 'my_package_render_admin_options'
] );

function my_package_render_admin_options() {
    $val = get_option( 'my_option', '' );
    ?>
    <div class="space-y-6">
        <div>
            <h2 class="text-lg font-bold text-slate-900 mb-1">Configuration</h2>
            <p class="text-sm text-slate-500">Sensitive or technical settings here.</p>
        </div>
        <div class="ss-field p-4 bg-slate-50 rounded-2xl border border-slate-100">
            <label class="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-2 ml-1">API Key</label>
            <input type="text" name="my_package_api_key" value="<?php echo esc_attr($val); ?>" class="w-full bg-white border-slate-200 rounded-xl p-3 text-sm font-medium">
        </div>
    </div>
    <?php
}
```

#### Handling Saves
Developer Config saves are unified. Hook into:
```php
add_action( 'slipstream_save_developer_config', function( $post_data ) {
    if ( isset( $post_data['my_package_api_key'] ) ) {
        update_option( 'my_option', sanitize_text_field( $post_data['my_package_api_key'] ) );
    }
} );
```

#### Assignable Content Fields
Register fields that can be mapped to specific pages by the developer. Slipstream automatically creates content panels for those pages and handles the UI rendering/saving using native custom fields.

**Registration:**
```php
$registry->register_content_field( 'my-package', [
    'name'        => 'Hero Title',
    'description' => 'Main headline for this page.',
    'merge_tag'   => 'hero_title', // Use as {{hero_title}}
    'type'        => 'text'       // 'text' (default) or 'textarea'
] );
```

**Available Field Types:**
- `text`: Standard single-line text input.
- `textarea`: Multi-line text area (default for the editor UI rendering).

**Benefits:**
1. **Dynamic Panels**: Pages with assigned fields automatically appear in the "Pages" group on the Owner Dashboard.
2. **Contextual Resolution**: `{{hero_title}}` will resolve to the value set on the specific page being viewed.
3. **Integrated UI**: Developers map fields to pages directly within the package's tab in **Slipstream > Developer Config**.
4. **Reliability**: Uses native `get_post_meta` and `update_post_meta` directly, bypassing complex ACF logic for better performance and stability.

---

### 4. UI/UX Standards (Tailwind)

Always use these design patterns for consistency:
- **Containers**: `bg-white rounded-3xl shadow-sm border border-slate-100 p-8`
- **Typography**:
  - Headers: `text-2xl font-bold text-slate-900`
  - Labels: `text-xs font-bold text-slate-400 uppercase tracking-wider`
- **Inputs**: `bg-slate-100 border-none rounded-2xl text-sm font-bold text-slate-700 p-4`
- **Buttons**:
  - Primary: `bg-indigo-600 text-white px-6 py-2 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 transition-all hover:bg-indigo-700 active:scale-95`
- **Colors**: Indigo (`#6366f1`), Slate (`#64748b`), Rose (`#f43f5e`) for errors.

---

### 5. AI Prompt Strategy

To have an AI generate a package, use a prompt like this:

> "Create a Slipstream package called '{Name}' with slug '{slug}'. 
> 1. Include a manifest.json (v1.0.0).
> 2. Register a Dashboard Panel in the 'content' group with an appropriate Lucide-style SVG icon.
> 3. Register a Developer Config tab with a text field for '{Setting Name}'.
> 4. Implement the save handler for the config field.
> 5. Use Slipstream Tailwind standards (rounded-3xl, slate text, indigo accents) for any UI."

---

### 6. Specialized Features

#### 6.1 Merge Tags
Merge tags are dynamic placeholders that can be used in page content, Divi layouts, or widgets.
- **Registration**: `$registry->register('merge_tags', 'slug', $args)`.
- **Syntax Options**:
  - `{{tag_slug}}`: Standard syntax for most text content.
  - `##tag_slug##`: Safe syntax for URLs or when Divi's sanitization strips curly braces.
- **Data Sources**:
  - `acf:field_name`: Pulls from an ACF field on the current page.
  - `acf:option:field_name`: Pulls from an ACF Options page field.
  - `option:option_name`: Pulls from a standard WordPress option.
  - `site:name` / `site:url`: Core site information.
  - `callback`: A PHP function that returns a string.
- **Conditional Logic**: Support for `{{if:tag_slug}} ... {{endif}}` and negative checks `{{if:!tag_slug}}`. Also supports safe `##` syntax (e.g., `##if:tag## ... ##endif##`).
- **Formatting**: Use the `format` argument (e.g., `tel`, `email`, `date`, `uppercase`).
- **Scanner**: The built-in "Merge Tag Scanner" helps identify where tags are used across the site.

#### 6.2 Assignable Content Fields
Allows developers to define fields at the package level that are then contextually mapped to specific pages.
- **Unified Mapping**: Configuration happens inside the package's tab in **Developer Config**.
- **Contextual Resolution**: A tag like `{{hero_title}}` will automatically resolve to the value saved specifically for the page being viewed.
- **Native Implementation**: Bypasses ACF for editing; uses `get_post_meta` and `update_post_meta` directly for maximum reliability and speed.
- **Pages Dashboard**: Pages with assigned fields automatically appear in a dedicated "Pages" group on the Owner's dashboard.

#### 6.3 Dynamic Engine (Menus)
The Dynamic Engine automates menu management by expanding a single placeholder into a list of posts or terms.
- **Usage**: Add a "Dynamic Engine" placeholder to any WordPress menu via **Appearance > Menus**.
- **Configuration**:
  - **Source Type**: Select any public Post Type.
  - **Filters**: Filter by Taxonomy (Category, Tag, or Custom).
  - **Logic**: Set limits, sorting (Date, Title, Random), and inclusion/exclusion rules.
- **Expansion**: Occurs at runtime, ensuring menus are always up-to-date with new content.

#### 6.4 Divi Modules
Slipstream provides a streamlined way to package and load custom Divi modules.
- **Directory**: Place modules in `{package}/divi/modules/{module_slug}/`.
- **Structure**:
  - `module.php`: The main module class.
  - `module_item.php`: Optional child module.
  - `assets/`: Automatically enqueues `.css` and `.js` files found here using `slipstream_package_url()`.
- **Enqueuing Assets**: When enqueuing assets manually in your package (e.g., in `functions.php`), always use the `slipstream_package_url( $path )` helper. This ensures the correct URL is generated regardless of whether the package is in the core plugin, a development folder (`.server/`), or the standard packages directory.
  ```php
  wp_enqueue_style( 'my-style', slipstream_package_url( __DIR__ . '/assets/css/style.css' ) );
  ```
- **Stability**: Always set `$vb_support = 'off';` in your module class. Slipstream focuses on frontend stability and native framework styling over the Visual Builder.

#### 6.5 ACF Integration
Slipstream deeply integrates with Advanced Custom Fields to provide a native-feeling experience.
- **Automatic Styling**: Any ACF field group viewed by an `owner` is automatically re-skinned using Tailwind CSS (rounded-3xl, slate backgrounds, indigo accents).
- **Tab Support**: Standard ACF tabs are transformed into modern, app-like navigation.
- **Dynamic Groups**: The framework automatically registers ACF field groups for assigned content fields if ACF is present, while still allowing for native fallback.
- **Restricted View**: Owners only see the fields they need; technical metaboxes and standard WordPress fields are hidden by default.

#### 6.6 Callback System
The Callback System allows packages to register functions that can be used by other parts of the framework, such as formatting data or processing external IDs.

**Best Practice**: Define callback functions at the top level of your package file (not inside a registration closure) to ensure they are available to the Registry when called.

- **Registration**:
```php
$registry->register_callback( 'my_callback', [
    'name' => '...',
    'callback' => 'my_callback_function_name',
    'description' => '...',
    'context' => [...]
] );
```
- **Description**: A short line explaining what the callback does, shown in admin UIs.
- **Context**: Optional array of strings to limit where the callback is available (e.g., `['social', 'team-members']`). If empty, it's considered universal.
- **Retrieval**: `$registry->get_callbacks( 'context' )` returns filtered results.