<?php
/**
 * ACF Handler Class - Integration with ACF
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_ACF_Handler {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'init', array( $this, 'check_acf' ) );
		add_action( 'acf/init', array( $this, 'register_dynamic_content_field_groups' ) );
	}

	/**
	 * Dynamically register ACF field groups for assigned content fields
	 */
	public function register_dynamic_content_field_groups() {
		if ( ! function_exists( 'acf_add_local_field_group' ) ) {
			return;
		}

		$registry = Slipstream_Registry::get_instance();
		$all_content_fields = $registry->get_content_fields();
		$assigned = get_option( 'slipstream_assigned_content_fields', array() );

		if ( empty( $all_content_fields ) || empty( $assigned ) ) {
			return;
		}

		// Group fields by page assignment
		$page_groups = array();
		foreach ( $assigned as $package_slug => $fields ) {
			foreach ( $fields as $field_slug => $page_ids ) {
				if ( ! isset( $all_content_fields[$package_slug][$field_slug] ) ) continue;
				
				foreach ( $page_ids as $page_id ) {
					if ( ! isset( $page_groups[$page_id] ) ) {
						$page_groups[$page_id] = array();
					}
					if ( ! isset( $page_groups[$page_id][$package_slug] ) ) {
						$page_groups[$page_id][$package_slug] = array();
					}
					$page_groups[$page_id][$package_slug][$field_slug] = $all_content_fields[$package_slug][$field_slug];
				}
			}
		}

		foreach ( $page_groups as $page_id => $packages ) {
			foreach ( $packages as $package_slug => $fields ) {
				$acf_fields = array();
				foreach ( $fields as $field_slug => $args ) {
					$acf_fields[] = array(
						'key' => 'field_ss_cf_' . $page_id . '_' . str_replace('-', '_', $package_slug) . '_' . $field_slug,
						'label' => $args['name'],
						'name' => 'ss_cf_' . str_replace('-', '_', $package_slug) . '_' . $field_slug,
						'type' => 'text', // Default to text, could be extended later
						'instructions' => $args['description'],
						'required' => 0,
					);
				}

				acf_add_local_field_group( array(
					'key' => 'group_ss_cf_' . $page_id . '_' . str_replace('-', '_', $package_slug),
					'title' => ucfirst( str_replace( '-', ' ', $package_slug ) ) . ' Content',
					'fields' => $acf_fields,
					'location' => array(
						array(
							array(
								'param' => 'post',
								'operator' => '==',
								'value' => $page_id,
							),
						),
					),
					'menu_order' => 0,
					'position' => 'normal',
					'style' => 'default',
					'label_placement' => 'top',
					'instruction_placement' => 'label',
					'hide_on_screen' => '',
					'active' => true,
					'description' => '',
				) );
			}
		}
	}

	/**
	 * Check if ACF is active and provide notices
	 */
	public function check_acf() {
		if ( ! class_exists( 'ACF' ) ) {
			add_action( 'admin_notices', array( $this, 'acf_missing_notice' ) );
		}
	}

	/**
	 * Notice if ACF is missing
	 */
	public function acf_missing_notice() {
		?>
		<div class="notice notice-warning">
			<p><?php _e( 'Slipstream: Advanced Custom Fields (ACF) is required for V1 functionality.', 'slipstream' ); ?></p>
		</div>
		<?php
	}
}
