<?php
/**
 * Admin UI Class - Admin-level interface for package management
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Admin_UI {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_init', array( $this, 'process_admin_actions' ) );

		// Post Status Toggle logic
		add_filter( 'manage_posts_columns', array( $this, 'add_status_toggle_column' ), 10, 2 );
		add_action( 'manage_posts_custom_column', array( $this, 'render_status_toggle_column' ), 10, 2 );
		add_action( 'wp_ajax_slipstream_toggle_post_status', array( $this, 'handle_status_toggle_ajax' ) );

		// Package Update Checks AJAX
		add_action( 'wp_ajax_slipstream_check_updates', array( $this, 'handle_update_checks_ajax' ) );
	}

	/**
	 * Handle AJAX Package Update Checks
	 */
	public function handle_update_checks_ajax() {
		check_ajax_referer( 'slipstream_package_manager', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}

		$remote_url = get_option( 'slipstream_remote_url' );
		$remote_key = get_option( 'slipstream_remote_key' );
		$all_remote = array();

		// 1. Get Library updates (Always force refresh for AJAX check?)
		if ( $remote_url ) {
			$response = wp_remote_get( add_query_arg( 'action', 'list', $remote_url ), array(
				'headers' => array( 'X-Slipstream-Key' => $remote_key ),
				'timeout' => 10
			) );
			if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
				$remote_packages = json_decode( wp_remote_retrieve_body( $response ), true );
				if ( is_array( $remote_packages ) ) {
					set_transient( 'slipstream_remote_packages', $remote_packages, 3600 );
					foreach ( $remote_packages as $rp ) {
						$all_remote[$rp['slug']] = $rp['version'];
					}
				}
			}
		}

		// 2. Get Example updates
		$examples_json = SLIPSTREAM_PATH . 'examples/packages.json';
		if ( file_exists( $examples_json ) ) {
			$examples = json_decode( file_get_contents( $examples_json ), true );
			if ( is_array( $examples ) ) {
				foreach ( $examples as $ex ) {
					if ( ! isset( $all_remote[$ex['slug']] ) || version_compare( $ex['version'], $all_remote[$ex['slug']], '>' ) ) {
						$all_remote[$ex['slug']] = $ex['version'];
					}
				}
			}
		}

		wp_send_json_success( array( 'updates' => $all_remote ) );
	}

	/**
	 * Add Status column to post types that opted in
	 */
	public function add_status_toggle_column( $columns, $post_type ) {
		$registry = Slipstream_Registry::get_instance();
		$enabled_post_types = $registry->get_data( 'status_toggle_post_types' );

		if ( isset( $enabled_post_types[$post_type] ) ) {
			// Add after title if possible
			$new_columns = array();
			foreach ( $columns as $key => $value ) {
				$new_columns[$key] = $value;
				if ( $key === 'title' ) {
					$new_columns['ss_status'] = __( 'Status', 'slipstream' );
				}
			}
			return $new_columns;
		}

		return $columns;
	}

	/**
	 * Render the status toggle in the list table
	 */
	public function render_status_toggle_column( $column, $post_id ) {
		if ( $column !== 'ss_status' ) {
			return;
		}

		$post = get_post( $post_id );
		$is_published = $post->post_status === 'publish';
		$nonce = wp_create_nonce( 'ss_status_toggle_' . $post_id );
		?>
		<div class="ss-status-toggle-wrapper" data-post-id="<?php echo $post_id; ?>">
			<label class="ss-status-toggle">
				<input type="checkbox" class="ss-status-toggle-input" <?php checked( $is_published ); ?> data-nonce="<?php echo $nonce; ?>">
				<span class="ss-status-slider"></span>
			</label>
			<span class="ss-status-label"><?php echo $is_published ? 'Published' : 'Draft'; ?></span>
		</div>
		<?php
	}

	/**
	 * Handle AJAX status toggle
	 */
	public function handle_status_toggle_ajax() {
		check_ajax_referer( 'ss_status_toggle_' . $_POST['post_id'], 'nonce' );

		if ( ! current_user_can( 'edit_post', $_POST['post_id'] ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		$post_id = intval( $_POST['post_id'] );
		$new_status = $_POST['status'] === 'publish' ? 'publish' : 'draft';

		$result = wp_update_post( array(
			'ID'          => $post_id,
			'post_status' => $new_status,
		) );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( array(
			'status' => $new_status,
			'label'  => $new_status === 'publish' ? 'Published' : 'Draft'
		) );
	}

	/**
	 * Process general admin actions
	 */
	public function process_admin_actions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_flush' ) {
			check_admin_referer( 'slipstream_flush_registry' );
			Slipstream_Registry::get_instance()->flush();
			set_transient( 'slipstream_success', 'Registry flushed and reloaded.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}

		if ( isset( $_POST['slipstream_save_remote'] ) ) {
			check_admin_referer( 'slipstream_save_remote' );
			$url = esc_url_raw( $_POST['slipstream_remote_url'] );
			$key = sanitize_text_field( $_POST['slipstream_remote_key'] );

			update_option( 'slipstream_remote_url', $url );
			update_option( 'slipstream_remote_key', $key );

			set_transient( 'slipstream_success', 'Remote library settings saved.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages#library' ) );
			exit;
		}

		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_disconnect_remote' ) {
			check_admin_referer( 'slipstream_disconnect_remote' );
			delete_option( 'slipstream_remote_url' );
			delete_option( 'slipstream_remote_key' );
			delete_transient( 'slipstream_remote_packages' );

			set_transient( 'slipstream_success', 'Remote library disconnected.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages#library' ) );
			exit;
		}

		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_refresh_library' ) {
			check_admin_referer( 'slipstream_refresh_library' );
			delete_transient( 'slipstream_remote_packages' );
			set_transient( 'slipstream_success', 'Library refreshed.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages#library' ) );
			exit;
		}


		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_clear_logs' ) {
			check_admin_referer( 'slipstream_clear_logs' );
			Slipstream_Activity_Log::get_instance()->clear_logs();
			set_transient( 'slipstream_success', 'Activity log cleared.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages#logs' ) );
			exit;
		}

		if ( isset( $_POST['slipstream_save_developer_config'] ) ) {
			check_admin_referer( 'slipstream_save_developer_config' );
			
			// Save Welcome Card Settings
			if ( isset( $_POST['slipstream_welcome_title'] ) ) {
				update_option( 'slipstream_welcome_title', sanitize_text_field( $_POST['slipstream_welcome_title'] ) );
			}
			if ( isset( $_POST['slipstream_welcome_desc'] ) ) {
				update_option( 'slipstream_welcome_desc', sanitize_textarea_field( $_POST['slipstream_welcome_desc'] ) );
			}

			// Save Metabox Settings
			$hidden_metaboxes_text = isset( $_POST['hidden_metaboxes'] ) ? sanitize_textarea_field( $_POST['hidden_metaboxes'] ) : '';
			$hidden_metaboxes_checks = isset( $_POST['hidden_metaboxes_list'] ) ? (array) $_POST['hidden_metaboxes_list'] : array();
			
			// Combine both sources
			$metaboxes_array = array_filter( array_map( 'trim', explode( "\n", $hidden_metaboxes_text ) ) );
			$combined = array_unique( array_merge( $metaboxes_array, $hidden_metaboxes_checks ) );
			update_option( 'slipstream_hidden_metaboxes', implode( "\n", $combined ) );

			// Save Assigned Content Fields
			$assigned_fields = isset( $_POST['assigned_fields'] ) ? $_POST['assigned_fields'] : array();
			update_option( 'slipstream_assigned_content_fields', $assigned_fields );

			// Fire an action so packages can save their own admin-only settings
			// We pass $_POST for convenience, but they should be careful
			do_action( 'slipstream_save_developer_config', $_POST );

			set_transient( 'slipstream_success', 'Configuration saved.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-developer-config' ) );
			exit;
		}
	}

	/**
	 * Register Admin Menu
	 */
	public function register_menu() {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			return;
		}

		add_menu_page(
			__( 'Slipstream Packages', 'slipstream' ),
			__( 'Slipstream', 'slipstream' ),
			'manage_options',
			'slipstream-packages',
			array( $this, 'render_page' ),
			'dashicons-layout',
			2
		);

		add_submenu_page(
			'slipstream-packages',
			__( 'Package Manager', 'slipstream' ),
			__( 'Package Manager', 'slipstream' ),
			'manage_options',
			'slipstream-packages',
			array( $this, 'render_page' )
		);

		add_submenu_page(
			'slipstream-packages',
			__( 'Developer Config', 'slipstream' ),
			__( 'Developer Config', 'slipstream' ),
			'manage_options',
			'slipstream-developer-config',
			array( $this, 'render_developer_config_page' )
		);

		add_submenu_page(
			'slipstream-packages',
			__( 'Merge Tag Scanner', 'slipstream' ),
			__( 'Merge Tag Scanner', 'slipstream' ),
			'manage_options',
			'slipstream-merge-tags',
			array( $this, 'render_merge_tags_page' )
		);

		add_submenu_page(
			null, // Hidden
			__( 'Content Editor', 'slipstream' ),
			__( 'Content Editor', 'slipstream' ),
			'read',
			'slipstream-content-editor',
			array( $this, 'render_content_editor_page' )
		);

		add_submenu_page(
			'slipstream-packages',
			__( 'Slipstream View', 'slipstream' ),
			__( 'Slipstream View', 'slipstream' ),
			'manage_options',
			admin_url( 'index.php?slipstream_owner_view=1' )
		);

		add_action( 'admin_head', array( $this, 'inject_menu_css' ) );
	}

	/**
	 * Inject CSS to style the Slipstream Admin Menu
	 */
	public function inject_menu_css() {
		?>
		<style id="slipstream-admin-menu-css">
			#toplevel_page_slipstream-packages .wp-submenu li a[href*="page=slipstream-developer-config"] {
				border-bottom: 1px solid rgba(0,0,0,0.05);
				margin-bottom: 4px;
				padding-bottom: 8px;
			}
			#toplevel_page_slipstream-packages .wp-submenu li a[href*="page=slipstream-merge-tags"],
			#toplevel_page_slipstream-packages .wp-submenu li a[href*="slipstream_owner_view=1"] {
				opacity: 0.5;
				transition: opacity 0.2s;
			}
			#toplevel_page_slipstream-packages .wp-submenu li a[href*="page=slipstream-merge-tags"]:hover,
			#toplevel_page_slipstream-packages .wp-submenu li a[href*="slipstream_owner_view=1"]:hover {
				opacity: 1;
			}
		</style>
		<?php
	}

	/**
	 * Render Package Management Page
	 */
	public function render_page() {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			wp_die( __( 'System is locked down.', 'slipstream' ) );
		}

		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		$error = get_transient( 'slipstream_error' );
		$success = get_transient( 'slipstream_success' );
		delete_transient( 'slipstream_error' );
		delete_transient( 'slipstream_success' );
		
		$manager = Slipstream_Package_Manager::get_instance();
		?>
		<div class="wrap ss-modern-admin-wrap" id="slipstream-package-manager" data-nonce="<?php echo wp_create_nonce( 'slipstream_package_manager' ); ?>">
			
			<div class="bg-white rounded-[2.5rem] shadow-sm border border-slate-100 overflow-hidden mb-12">
				<div class="p-4 px-8 border-b border-slate-50 bg-slate-50/50 flex items-center justify-between">
					<div class="flex items-center gap-4">
						<h2 class="text-lg font-bold text-slate-900"><?php _e( 'Installed Packages', 'slipstream' ); ?></h2>
						<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_flush' ), 'slipstream_flush_registry' ); ?>" class="text-slate-400 hover:text-indigo-600 transition-all flex items-center" title="<?php esc_attr_e( 'Flush Registry', 'slipstream' ); ?>">
							<span class="dashicons dashicons-update !text-[18px] !w-[18px] !h-[18px] !leading-none"></span>
						</a>
					</div>
					<span class="text-xs font-bold text-slate-400 uppercase tracking-widest"><?php echo count($packages); ?> Packages</span>
				</div>

				<div class="overflow-x-auto">
					<table class="w-full text-left border-collapse">
						<thead>
							<tr class="text-[10px] font-black uppercase text-slate-400 tracking-widest border-b border-slate-50">
								<th class="px-8 py-4"><?php _e( 'Package', 'slipstream' ); ?></th>
								<th class="px-4 py-4 w-24 text-center"><?php _e( 'Version', 'slipstream' ); ?></th>
								<th class="px-4 py-4 w-32 text-center"><?php _e( 'Core Req', 'slipstream' ); ?></th>
								<th class="px-4 py-4 w-32 text-center"><?php _e( 'Status', 'slipstream' ); ?></th>
								<th class="px-8 py-4 w-64 text-right"><?php _e( 'Actions', 'slipstream' ); ?></th>
							</tr>
						</thead>
						<tbody class="divide-y divide-slate-50">
							<?php if ( empty( $packages ) ) : ?>
								<tr>
									<td colspan="5" class="px-8 py-12 text-center text-slate-400 italic font-medium">
										<?php _e( 'No packages installed.', 'slipstream' ); ?>
									</td>
								</tr>
							<?php else : ?>
								<?php 
								// Check for updates across both remote and examples
								$remote_url = get_option( 'slipstream_remote_url' );
								$remote_key = get_option( 'slipstream_remote_key' );
								$all_remote = array();

								// 1. Get Library updates
								if ( $remote_url ) {
									$remote_packages = get_transient( 'slipstream_remote_packages' );
									if ( is_array( $remote_packages ) ) {
										foreach ( $remote_packages as $rp ) {
											$all_remote[$rp['slug']] = $rp['version'];
										}
									}
								}

								// 2. Get Example updates
								$examples_json = SLIPSTREAM_PATH . 'examples/packages.json';
								if ( file_exists( $examples_json ) ) {
									$examples = json_decode( file_get_contents( $examples_json ), true );
									if ( is_array( $examples ) ) {
										foreach ( $examples as $ex ) {
											if ( ! isset( $all_remote[$ex['slug']] ) || version_compare( $ex['version'], $all_remote[$ex['slug']], '>' ) ) {
												$all_remote[$ex['slug']] = $ex['version'];
											}
										}
									}
								}
								?>

								<?php foreach ( $packages as $slug => $data ) : ?>
									<?php 
									$is_modified = $manager->is_modified( $slug ); 
									$requires_core = isset( $data['requires_core'] ) ? $data['requires_core'] : '1.0.0';
									$is_compatible = isset( $data['compatible'] ) ? $data['compatible'] : true;
									$has_update = isset( $all_remote[$slug] ) && version_compare( $all_remote[$slug], $data['version'], '>' );
									?>
									<tr class="package-row hover:bg-slate-50/50 transition-colors" data-slug="<?php echo esc_attr( $slug ); ?>" data-version="<?php echo esc_attr( $data['version'] ); ?>" data-compatible="<?php echo $is_compatible ? '1' : '0'; ?>">
										<td class="px-8 py-2">
											<div class="flex flex-col">
												<div class="flex items-center gap-2">
													<span class="text-sm font-bold text-slate-900"><?php echo esc_html( $data['name'] ); ?></span>
													<?php if ( $is_modified ) : ?>
														<span class="px-1.5 py-0.5 rounded bg-rose-50 text-rose-600 text-[9px] font-black uppercase tracking-wider"><?php _e( 'Modified', 'slipstream' ); ?></span>
													<?php endif; ?>
												</div>
												<?php if ( ! empty( $data['note'] ) ) : ?>
													<div class="mt-2 p-2 rounded-xl bg-indigo-50/50 text-[11px] text-indigo-600 font-medium italic border border-indigo-100/50">
														<?php echo esc_html( $data['note'] ); ?>
													</div>
												<?php endif; ?>
											</div>
										</td>
										<td class="px-4 py-2 text-center">
											<div class="flex items-center justify-center gap-1.5">
												<span class="text-xs font-bold text-slate-600"><?php echo esc_html( $data['version'] ); ?></span>
												<?php if ( ! empty( $data['changelog'] ) ) : ?>
													<button type="button" class="slipstream-view-changelog text-indigo-400 hover:text-indigo-600 transition-colors" data-name="<?php echo esc_attr( $data['name'] ); ?>" data-changelog="<?php echo esc_attr( $data['changelog'] ); ?>" title="<?php esc_attr_e( 'View Changelog', 'slipstream' ); ?>">
														<span class="dashicons dashicons-megaphone !text-[14px] !w-[14px] !h-[14px] !leading-none"></span>
													</button>
												<?php endif; ?>
											</div>
										</td>
										<td class="px-4 py-2 text-center">
											<span class="text-xs font-bold <?php echo ! $is_compatible ? 'text-rose-600' : 'text-slate-500'; ?>">
												<?php echo esc_html( $requires_core ); ?>
												<?php if ( ! $is_compatible ) : ?>
													<span class="dashicons dashicons-warning !text-[14px] !w-[14px] !h-[14px] !leading-none align-middle ml-0.5" title="<?php esc_attr_e( 'Incompatible with current Core version', 'slipstream' ); ?>"></span>
												<?php endif; ?>
											</span>
										</td>
										<td class="px-4 py-2 text-center">
											<?php if ( $data['active'] ) : ?>
												<span class="inline-flex items-center px-2.5 py-1 rounded-full text-[10px] font-black uppercase tracking-wider bg-emerald-50 text-emerald-600 border border-emerald-100"><?php _e( 'Active', 'slipstream' ); ?></span>
											<?php else : ?>
												<span class="inline-flex items-center px-2.5 py-1 rounded-full text-[10px] font-black uppercase tracking-wider bg-slate-50 text-slate-400 border border-slate-100"><?php _e( 'Inactive', 'slipstream' ); ?></span>
											<?php endif; ?>
										</td>
										<td class="px-8 py-2 text-right">
											<div class="flex items-center justify-end gap-2">
												<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_toggle&package=' . $slug ), 'slipstream_toggle_package' ); ?>" class="text-[11px] font-bold py-1.5 px-3 rounded-xl transition-all <?php echo $data['active'] ? 'bg-slate-100 text-slate-600 hover:bg-slate-200' : 'bg-indigo-50 text-indigo-600 hover:bg-indigo-100'; ?> <?php echo ! $is_compatible ? 'opacity-50 cursor-not-allowed grayscale pointer-events-none' : ''; ?>">
													<?php echo $data['active'] ? __( 'Deactivate', 'slipstream' ) : __( 'Activate', 'slipstream' ); ?>
												</a>
												
												<div class="update-container inline-flex">
												<?php if ( $has_update ) : ?>
													<?php 
													$is_example = false; $example_zip = '';
													if ( file_exists( $examples_json ) ) {
														$examples_data = json_decode( file_get_contents( $examples_json ), true );
														if ( is_array( $examples_data ) ) {
															foreach ( $examples_data as $ex ) {
																if ( $ex['slug'] === $slug ) { $is_example = true; $example_zip = $ex['zip']; break; }
															}
														}
													}
													$update_url = $is_example ? admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_example&zip=' . $example_zip ) : admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_remote&slug=' . $slug );
													$update_nonce = $is_example ? 'slipstream_install_example' : 'slipstream_install_remote';
													?>
													<a href="<?php echo wp_nonce_url( $update_url, $update_nonce ); ?>" class="bg-emerald-500 text-white text-[11px] font-bold py-1.5 px-3 rounded-xl hover:bg-emerald-600 shadow-sm flex items-center gap-1.5" title="<?php printf( esc_attr__( 'Update available: %s', 'slipstream' ), $all_remote[$slug] ); ?>">
														<span class="dashicons dashicons-update !text-[12px] !w-[12px] !h-[12px] !leading-none"></span>
														<?php _e( 'Update', 'slipstream' ); ?>
													</a>
												<?php endif; ?>
												</div>

												<button type="button" class="text-slate-400 hover:text-indigo-600 p-1.5 transition-colors" onclick="document.getElementById('note-form-<?php echo $slug; ?>').classList.toggle('hidden');">
													<span class="dashicons dashicons-edit !text-[18px] !w-[18px] !h-[18px] !leading-none <?php echo !empty($data['note']) ? 'text-indigo-500' : ''; ?>"></span>
												</button>

												<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_delete&package=' . $slug ), 'slipstream_delete_package' ); ?>" class="text-rose-300 hover:text-rose-600 p-1.5 transition-colors" onclick="return confirm('Are you sure you want to delete this package?');">
													<span class="dashicons dashicons-trash !text-[18px] !w-[18px] !h!-[18px] !leading-none"></span>
												</a>
											</div>

											<div id="note-form-<?php echo $slug; ?>" class="hidden mt-4 text-left">
												<form method="post" class="space-y-3">
													<?php wp_nonce_field( 'slipstream_save_note' ); ?>
													<input type="hidden" name="package_slug" value="<?php echo esc_attr( $slug ); ?>">
													<textarea name="package_note" class="w-full h-24 p-3 rounded-2xl border border-slate-100 bg-slate-50 text-xs font-medium focus:ring-indigo-500 focus:border-indigo-500" placeholder="Add a custom note for this package..."><?php echo esc_textarea( isset($data['note']) ? $data['note'] : '' ); ?></textarea>
													<div class="flex justify-end gap-2">
														<button type="button" class="text-[10px] font-black uppercase text-slate-400 hover:text-slate-600 px-3 py-1.5" onclick="document.getElementById('note-form-<?php echo $slug; ?>').classList.add('hidden');"><?php _e( 'Cancel', 'slipstream' ); ?></button>
														<button type="submit" name="save_package_note" class="bg-indigo-600 text-white text-[10px] font-black uppercase tracking-wider px-4 py-1.5 rounded-lg shadow-md shadow-indigo-100 hover:bg-indigo-700"><?php _e( 'Save Note', 'slipstream' ); ?></button>
													</div>
												</form>
											</div>
										</td>
									</tr>
								<?php endforeach; ?>
							<?php endif; ?>
						</tbody>
						<tfoot>
							<tr class="bg-slate-50/30">
								<td colspan="5" class="px-8 py-4 rounded-b-[2.5rem]"></td>
							</tr>
						</tfoot>
					</table>
				</div>
			</div>

			<!-- Add Package Section -->
			<div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
				<div class="lg:col-span-2">
					<div class="bg-white rounded-[2.5rem] shadow-sm border border-slate-100 overflow-hidden">
						<div class="flex bg-slate-50/50 p-2 px-8 border-b border-slate-50">
							<button type="button" class="ss-mgr-tab-btn flex-1 py-3 px-4 rounded-2xl text-sm font-bold transition-all bg-white text-indigo-600 shadow-sm" data-target="mgr-upload"><?php _e( 'Upload', 'slipstream' ); ?></button>
							<button type="button" class="ss-mgr-tab-btn flex-1 py-3 px-4 rounded-2xl text-sm font-bold transition-all text-slate-500 hover:bg-white/50" data-target="mgr-library"><?php _e( 'Library', 'slipstream' ); ?></button>
							<button type="button" class="ss-mgr-tab-btn flex-1 py-3 px-4 rounded-2xl text-sm font-bold transition-all text-slate-500 hover:bg-white/50" data-target="mgr-examples"><?php _e( 'Examples', 'slipstream' ); ?></button>
						</div>

						<div class="p-8">
							<div id="mgr-upload" class="ss-mgr-tab-content">
								<h3 class="text-lg font-bold text-slate-900 mb-2"><?php _e( 'Manual Upload', 'slipstream' ); ?></h3>
								<p class="text-sm text-slate-500 mb-6"><?php _e( 'Install a new feature package by uploading a .zip archive.', 'slipstream' ); ?></p>
								
								<form method="post" enctype="multipart/form-data" class="flex flex-col sm:flex-row gap-4 items-start sm:items-center p-6 rounded-3xl bg-slate-50 border border-slate-100">
									<?php wp_nonce_field( 'slipstream_upload_package' ); ?>
									<div class="flex-1 w-full">
										<input type="file" name="slipstream_package_zip" accept=".zip" class="w-full text-sm text-slate-500 file:mr-4 file:py-2 file:px-4 file:rounded-xl file:border-0 file:text-xs file:font-black file:uppercase file:bg-indigo-50 file:text-indigo-600 hover:file:bg-indigo-100 cursor-pointer">
									</div>
									<button type="submit" name="upload_package" class="w-full sm:w-auto bg-indigo-600 text-white px-6 py-2.5 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 hover:bg-indigo-700 active:scale-95 transition-all">
										<?php _e( 'Upload ZIP', 'slipstream' ); ?>
									</button>
								</form>
							</div>

							<div id="mgr-library" class="ss-mgr-tab-content hidden">
								<div class="flex flex-col md:flex-row md:items-center justify-between mb-6 gap-4">
									<div>
										<h3 class="text-lg font-bold text-slate-900 mb-1"><?php _e( 'Package Library', 'slipstream' ); ?></h3>
										<p class="text-sm text-slate-500"><?php _e( 'Browse and install official packages from the remote repository.', 'slipstream' ); ?></p>
									</div>
									<div class="flex items-center gap-3">
										<div class="relative">
											<input type="text" id="ss-library-search" class="w-full sm:w-64 bg-slate-50 border-slate-200 rounded-xl px-4 py-2 text-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="<?php esc_attr_e( 'Search library...', 'slipstream' ); ?>">
											<span class="dashicons dashicons-search absolute right-3 top-2.5 text-slate-400 !text-sm"></span>
										</div>
										<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_refresh_library' ), 'slipstream_refresh_library' ); ?>" class="text-xs font-bold text-indigo-600 hover:text-indigo-800 transition-colors flex items-center gap-1.5 whitespace-nowrap">
											<span class="dashicons dashicons-update !text-[14px] !w-[14px] !h-[14px] !leading-none align-middle"></span>
											<?php _e( 'Refresh', 'slipstream' ); ?>
										</a>
									</div>
								</div>

								<?php if ( ! $remote_url ) : ?>
									<div class="p-8 rounded-3xl bg-slate-50 border border-dashed border-slate-200 text-center">
										<div class="w-12 h-12 rounded-2xl bg-white shadow-sm flex items-center justify-center mx-auto mb-4 text-slate-300">
											<span class="dashicons dashicons-admin-links !text-[24px] !w-[24px] !h-[24px]"></span>
										</div>
										<h4 class="text-sm font-bold text-slate-900 mb-2"><?php _e( 'Connect to Repository', 'slipstream' ); ?></h4>
										<p class="text-xs text-slate-500 mb-6 max-w-sm mx-auto"><?php _e( 'Connect to a remote Slipstream server to access additional feature packages and automatic updates.', 'slipstream' ); ?></p>
										
										<form method="post" class="space-y-4 max-w-md mx-auto">
											<?php wp_nonce_field( 'slipstream_save_remote' ); ?>
											<input name="slipstream_remote_url" type="url" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium" placeholder="https://repository-url.com/" required>
											<input name="slipstream_remote_key" type="password" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium" placeholder="API Key" required>
											<button type="submit" name="slipstream_save_remote" class="w-full bg-indigo-600 text-white py-3 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 hover:bg-indigo-700 transition-all"><?php _e( 'Connect Now', 'slipstream' ); ?></button>
										</form>
									</div>
								<?php else : ?>
									<div class="mb-6 p-4 rounded-2xl bg-emerald-50 border border-emerald-100 flex items-center justify-between">
										<div class="flex items-center gap-3">
											<div class="w-8 h-8 rounded-lg bg-white flex items-center justify-center text-emerald-500 shadow-sm">
												<span class="dashicons dashicons-yes-alt !text-[18px] !w-[18px] !h-[18px]"></span>
											</div>
											<div>
												<p class="text-xs font-bold text-emerald-800"><?php _e( 'Connected to Repository', 'slipstream' ); ?></p>
												<code class="text-[10px] text-emerald-600"><?php echo esc_html( $remote_url ); ?></code>
											</div>
										</div>
										<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_disconnect_remote' ), 'slipstream_disconnect_remote' ); ?>" class="text-[10px] font-black uppercase text-rose-600 hover:text-rose-800 transition-colors"><?php _e( 'Disconnect', 'slipstream' ); ?></a>
									</div>

									<?php
									// Refresh logic
									if ( false === get_transient( 'slipstream_remote_packages' ) ) {
										$response = wp_remote_get( add_query_arg( 'action', 'list', $remote_url ), array( 'headers' => array('X-Slipstream-Key' => $remote_key), 'timeout' => 10 ) );
										if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
											$remote_packages = json_decode( wp_remote_retrieve_body( $response ), true );
											set_transient( 'slipstream_remote_packages', $remote_packages, 3600 );
										}
									}
									$remote_packages = get_transient( 'slipstream_remote_packages' );

									if ( ! empty( $remote_packages ) ) : ?>
										<div class="space-y-3">
											<?php foreach ( $remote_packages as $remote_pkg ) : 
												$is_installed = isset( $packages[$remote_pkg['slug']] );
												$has_update = $is_installed && version_compare( $remote_pkg['version'], $packages[$remote_pkg['slug']]['version'], '>' );
												$requires_core = isset( $remote_pkg['requires_core'] ) ? $remote_pkg['requires_core'] : '1.0.0';
												$is_compatible = version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' );
												?>
												<div class="ss-library-item p-4 rounded-2xl border border-slate-100 bg-white hover:border-indigo-100 transition-all flex items-center justify-between" data-name="<?php echo esc_attr( strtolower( $remote_pkg['name'] ) ); ?>" data-desc="<?php echo esc_attr( strtolower( $remote_pkg['description'] ) ); ?>">
													<div class="flex flex-col min-w-0 pr-4">
														<span class="text-sm font-bold text-slate-900 truncate"><?php echo esc_html( $remote_pkg['name'] ); ?></span>
														<p class="text-[11px] text-slate-500 truncate"><?php echo esc_html( $remote_pkg['description'] ); ?></p>
														<div class="flex items-center gap-3 mt-1.5">
															<span class="text-[10px] font-black text-slate-400 uppercase tracking-widest">v<?php echo esc_html( $remote_pkg['version'] ); ?></span>
															<span class="text-[10px] font-bold <?php echo ! $is_compatible ? 'text-rose-500' : 'text-slate-300'; ?>"><?php echo sprintf( __('Requires Core %s', 'slipstream'), $requires_core ); ?></span>
														</div>
													</div>
													<div class="flex-shrink-0">
														<?php if ( ! $is_installed ) : ?>
															<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_remote&slug=' . $remote_pkg['slug'] ), 'slipstream_install_remote' ); ?>" class="bg-indigo-600 text-white text-[11px] font-bold py-1.5 px-4 rounded-xl hover:bg-indigo-700 transition-all <?php echo ! $is_compatible ? 'opacity-50 pointer-events-none' : ''; ?>"><?php _e( 'Install', 'slipstream' ); ?></a>
														<?php elseif ( $has_update ) : ?>
															<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_remote&slug=' . $remote_pkg['slug'] ), 'slipstream_install_remote' ); ?>" class="bg-emerald-500 text-white text-[11px] font-bold py-1.5 px-4 rounded-xl hover:bg-emerald-600 transition-all <?php echo ! $is_compatible ? 'opacity-50 pointer-events-none' : ''; ?>"><?php _e( 'Update', 'slipstream' ); ?></a>
														<?php else : ?>
															<span class="flex items-center gap-1 text-[10px] font-black uppercase text-emerald-500"><span class="dashicons dashicons-yes !text-[16px] !w-[16px] !h-[16px]"></span> <?php _e( 'Installed', 'slipstream' ); ?></span>
														<?php endif; ?>
													</div>
												</div>
											<?php endforeach; ?>
										</div>
									<?php else : ?>
										<p class="text-center py-8 text-slate-400 italic text-sm"><?php _e( 'No packages found in library.', 'slipstream' ); ?></p>
									<?php endif; ?>
								<?php endif; ?>
							</div>

							<div id="mgr-examples" class="ss-mgr-tab-content hidden">
								<h3 class="text-lg font-bold text-slate-900 mb-1"><?php _e( 'Example Packages', 'slipstream' ); ?></h3>
								<p class="text-sm text-slate-500 mb-6"><?php _e( 'Quickly install bundled example packages to explore Slipstream features.', 'slipstream' ); ?></p>
								
								<?php
								$examples_json = SLIPSTREAM_PATH . 'examples/packages.json';
								if ( file_exists( $examples_json ) ) :
									$examples = json_decode( file_get_contents( $examples_json ), true );
									if ( ! empty( $examples ) ) : ?>
										<div class="space-y-3">
											<?php foreach ( $examples as $example ) : 
												$is_installed = isset( $packages[$example['slug']] );
												$has_update = $is_installed && version_compare( $example['version'], $packages[$example['slug']]['version'], '>' );
												$requires_core = isset( $example['requires_core'] ) ? $example['requires_core'] : '1.0.0';
												$is_compatible = version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' );
												?>
												<div class="p-4 rounded-2xl border border-slate-100 bg-white hover:border-indigo-100 transition-all flex items-center justify-between">
													<div class="flex flex-col min-w-0 pr-4">
														<span class="text-sm font-bold text-slate-900 truncate"><?php echo esc_html( $example['name'] ); ?></span>
														<p class="text-[11px] text-slate-500 truncate"><?php echo esc_html( $example['description'] ); ?></p>
														<div class="flex items-center gap-3 mt-1.5">
															<span class="text-[10px] font-black text-slate-400 uppercase tracking-widest">v<?php echo esc_html( $example['version'] ); ?></span>
															<span class="text-[10px] font-bold <?php echo ! $is_compatible ? 'text-rose-500' : 'text-slate-300'; ?>"><?php echo sprintf( __('Requires Core %s', 'slipstream'), $requires_core ); ?></span>
														</div>
													</div>
													<div class="flex-shrink-0">
														<?php if ( ! $is_installed ) : ?>
															<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_example&zip=' . $example['zip'] ), 'slipstream_install_example' ); ?>" class="bg-indigo-600 text-white text-[11px] font-bold py-1.5 px-4 rounded-xl hover:bg-indigo-700 transition-all <?php echo ! $is_compatible ? 'opacity-50 pointer-events-none' : ''; ?>"><?php _e( 'Install', 'slipstream' ); ?></a>
														<?php elseif ( $has_update ) : ?>
															<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_install_example&zip=' . $example['zip'] ), 'slipstream_install_example' ); ?>" class="bg-emerald-500 text-white text-[11px] font-bold py-1.5 px-4 rounded-xl hover:bg-emerald-600 transition-all <?php echo ! $is_compatible ? 'opacity-50 pointer-events-none' : ''; ?>"><?php _e( 'Update', 'slipstream' ); ?></a>
														<?php else : ?>
															<span class="flex items-center gap-1 text-[10px] font-black uppercase text-emerald-500"><span class="dashicons dashicons-yes !text-[16px] !w!-[16px] !h-[16px]"></span> <?php _e( 'Installed', 'slipstream' ); ?></span>
														<?php endif; ?>
													</div>
												</div>
											<?php endforeach; ?>
										</div>
									<?php else : ?>
										<p class="text-center py-8 text-slate-400 italic text-sm"><?php _e( 'No examples found.', 'slipstream' ); ?></p>
									<?php endif;
								else : ?>
									<p class="text-center py-8 text-slate-400 italic text-sm"><?php _e( 'Example manifest not found.', 'slipstream' ); ?></p>
								<?php endif; ?>
							</div>
						</div>
					</div>
				</div>

				<div class="lg:col-span-1">
					<div class="bg-white rounded-[2.5rem] shadow-sm border border-slate-100 overflow-hidden h-full flex flex-col">
						<div class="p-2 px-6 border-b border-slate-50 bg-slate-50/50 flex items-center justify-between">
							<h3 class="text-base font-bold text-slate-900"><?php _e( 'Activity Log', 'slipstream' ); ?></h3>
							<a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=slipstream-packages&action=slipstream_clear_logs' ), 'slipstream_clear_logs' ); ?>" class="text-[9px] font-black uppercase text-rose-400 hover:text-rose-600 transition-colors" onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to clear the logs?', 'slipstream' ); ?>');">
								<?php _e( 'Clear', 'slipstream' ); ?>
							</a>
						</div>
						
						<div class="flex-1 overflow-y-auto custom-scrollbar p-6 max-h-[600px]">
							<?php 
							$logs = Slipstream_Activity_Log::get_instance()->get_logs();
							if ( ! empty( $logs ) ) : ?>
								<div class="space-y-6">
									<?php foreach ( $logs as $log ) : 
										$is_error = $log['status'] === 'error';
										?>
										<div class="relative pl-6 pb-6 border-l border-slate-100 last:pb-0">
											<div class="absolute left-[-5px] top-0 w-2.5 h-2.5 rounded-full <?php echo $is_error ? 'bg-rose-500' : 'bg-emerald-500'; ?> border-2 border-white shadow-sm"></div>
											<div class="flex flex-col">
												<span class="text-[9px] font-black uppercase tracking-widest text-slate-400 mb-1"><?php echo esc_html( $log['timestamp'] ); ?></span>
												<span class="text-xs font-bold text-slate-800 leading-tight"><?php echo esc_html( $log['action'] ); ?></span>
												<p class="text-[11px] text-slate-500 mt-1"><?php echo esc_html( $log['details'] ); ?></p>
												<span class="text-[9px] font-medium text-slate-300 mt-2 uppercase tracking-tighter"><?php echo sprintf( __('BY %s', 'slipstream'), $log['user'] ); ?></span>
											</div>
										</div>
									<?php endforeach; ?>
								</div>
							<?php else : ?>
								<div class="h-full flex flex-col items-center justify-center text-slate-300 opacity-50 py-12">
									<span class="dashicons dashicons-list-view !text-[48px] !w-[48px] !h-[48px] mb-4"></span>
									<p class="text-sm font-medium italic"><?php _e( 'No activity logged yet.', 'slipstream' ); ?></p>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</div>
			</div>

			<div id="ss-success-toast" class="fixed bottom-10 left-1/2 transform -translate-x-1/2 translate-y-20 opacity-0 transition-all duration-500 bg-slate-900 text-white px-6 py-3 rounded-full shadow-2xl flex items-center gap-3 z-[100000]">
				<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-emerald-400"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
				<span class="text-sm font-semibold"><?php echo esc_html( $success ?: $error ); ?></span>
			</div>

			<script>
				jQuery(document).ready(function($) {
					// Handle Toast
					<?php if ( $success || $error ) : ?>
						const $toast = $('#ss-success-toast');
						<?php if ($error) : ?>
							$toast.removeClass('bg-slate-900').addClass('bg-rose-600');
							$toast.find('svg').removeClass('text-emerald-400').addClass('text-white');
						<?php endif; ?>
						$toast.removeClass('translate-y-20 opacity-0').addClass('translate-y-0 opacity-100');
						setTimeout(() => {
							$toast.addClass('translate-y-20 opacity-0').removeClass('translate-y-0 opacity-100');
						}, 4000);
					<?php endif; ?>

					// Library Search
					$('#ss-library-search').on('input', function() {
						const term = $(this).val().toLowerCase();
						$('.ss-library-item').each(function() {
							const name = $(this).data('name') || '';
							const desc = $(this).data('desc') || '';
							if (name.includes(term) || desc.includes(term)) {
								$(this).css('display', 'flex');
							} else {
								$(this).hide();
							}
						});
					});

					// Tab switching
					$('.ss-mgr-tab-btn').on('click', function(e) {
						e.preventDefault();
						const target = $(this).data('target');
						$('.ss-mgr-tab-btn').removeClass('bg-white text-indigo-600 shadow-sm').addClass('text-slate-500 hover:bg-white/50');
						$(this).addClass('bg-white text-indigo-600 shadow-sm').removeClass('text-slate-500 hover:bg-white/50');
						$('.ss-mgr-tab-content').addClass('hidden');
						$('#' + target).removeClass('hidden');
					});

					// Changelog Modal
					$('.slipstream-view-changelog').on('click', function(e) {
						e.preventDefault();
						var name = $(this).data('name');
						var changelog = $(this).data('changelog');
						
						var modalHtml = '<div id="ss-changelog-modal" class="fixed inset-0 bg-slate-900/60 backdrop-blur-sm z-[99999] flex items-center justify-center p-4">' +
							'<div class="bg-white w-full max-w-2xl rounded-[2.5rem] shadow-2xl overflow-hidden flex flex-col max-h-[80vh]">' +
								'<div class="p-8 pb-4 flex items-center justify-between border-b border-slate-50">' +
									'<h3 class="text-lg font-bold text-slate-900">' + name + ' Changelog</h3>' +
									'<button id="ss-close-modal" class="p-2 text-slate-300 hover:text-slate-500 transition-colors"><svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg></button>' +
								'</div>' +
								'<div class="flex-1 overflow-y-auto custom-scrollbar p-8 pt-6">' +
									'<div class="prose prose-slate prose-sm max-w-none text-slate-600">' + changelog + '</div>' +
								'</div>' +
							'</div></div>';
						
						$('body').append(modalHtml);
						
						$('#ss-close-modal, #ss-changelog-modal').on('click', function(e) {
							if (e.target === this || $(this).attr('id') === 'ss-close-modal' || $(this).closest('#ss-close-modal').length) {
								$('#ss-changelog-modal').remove();
							}
						});
					});

					// AJAX Update Check
					const $manager = $('#slipstream-package-manager');
					if ($manager.length) {
						const nonce = $manager.data('nonce');
						$.ajax({
							url: ajaxurl,
							type: 'GET',
							data: { action: 'slipstream_check_updates', nonce: nonce },
							success: function(response) {
								if (response.success && response.data.updates) {
									const updates = response.data.updates;
									let foundNew = false;
									$('.package-row').each(function() {
										const slug = $(this).data('slug');
										const version = $(this).data('version').toString();
										if (updates[slug] && isNewerVersion(updates[slug], version)) {
											if (!$(this).find('.bg-emerald-500').length) { foundNew = true; }
										}
									});
									if (foundNew) { location.reload(); }
								}
							}
						});
					}

					function isNewerVersion(v1, v2) {
						const parts1 = v1.split('.').map(Number);
						const parts2 = v2.split('.').map(Number);
						for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
							const p1 = parts1[i] || 0;
							const p2 = parts2[i] || 0;
							if (p1 > p2) return true;
							if (p1 < p2) return false;
						}
						return false;
					}
				});
			</script>

			<style>
				.ss-modern-admin-wrap { margin-top: 2rem; max-width: 1200px; margin-left: auto; margin-right: auto; padding-right: 2rem; }
				.custom-scrollbar::-webkit-scrollbar { width: 4px; }
				.custom-scrollbar::-webkit-scrollbar-thumb { background: #e2e8f0; border-radius: 10px; }
				.prose h1, .prose h2, .prose h3 { font-weight: 800; color: #0f172a; margin-top: 1.5em; margin-bottom: 0.5em; }
				.prose h1:first-child, .prose h2:first-child, .prose h3:first-child { margin-top: 0; }
				.prose p { margin-bottom: 1em; }
				.prose ul { list-style-type: disc; padding-left: 1.5em; margin-bottom: 1em; }
				.prose li { margin-bottom: 0.25em; }
			</style>
		</div>
		<?php
	}

	/**
	 * Render Merge Tag Scanner Page
	 */
	public function render_merge_tags_page() {
		?>
		<div class="wrap">
			<h1 class="wp-heading-inline"><?php _e( 'Merge Tag Scanner', 'slipstream' ); ?></h1>
			<hr class="wp-header-end">

			<div class="mt-8">
				<div class="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden">
					<div class="p-6 border-b border-slate-100 flex items-center justify-between bg-slate-50/50">
						<div>
							<h2 class="text-lg font-bold text-slate-800 m-0"><?php _e( 'Database Scanner', 'slipstream' ); ?></h2>
							<p class="text-slate-500 text-sm m-0 mt-1"><?php _e( 'Locate all merge tags used across posts, meta, and options.', 'slipstream' ); ?></p>
						</div>
						<button type="button" id="slipstream-scan-mt-btn" class="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg font-bold text-sm transition-colors flex items-center shadow-sm">
							<span class="dashicons dashicons-search mr-2 !text-[18px] !w-[18px] !h-[18px] !flex !items-center !justify-center"></span>
							<?php _e( 'Start Scan', 'slipstream' ); ?>
						</button>
					</div>

					<div id="slipstream-scan-results" class="p-6">
						<div id="slipstream-scan-loading" class="hidden py-12 text-center">
							<span class="spinner is-active !float-none !m-0"></span>
							<p class="mt-4 text-slate-500 font-medium"><?php _e( 'Analyzing database records...', 'slipstream' ); ?></p>
						</div>

						<div id="slipstream-scan-empty" class="py-12 text-center bg-slate-50 rounded-lg border-2 border-dashed border-slate-200">
							<div class="mx-auto w-12 h-12 text-slate-300 mb-4">
								<svg fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" /></svg>
							</div>
							<p class="text-slate-500 font-medium m-0"><?php _e( 'No scan results to display. Click "Start Scan" to begin.', 'slipstream' ); ?></p>
						</div>

						<div id="slipstream-scan-container" class="hidden space-y-4">
							<!-- Grouped results will be injected here -->
						</div>
					</div>
				</div>
			</div>
		</div>

		<script>
		jQuery(document).ready(function($) {
			$('#slipstream-scan-mt-btn').on('click', function() {
				const $btn = $(this);
				const $container = $('#slipstream-scan-container');
				const $loading = $('#slipstream-scan-loading');
				const $empty = $('#slipstream-scan-empty');

				$btn.prop('disabled', true).addClass('opacity-50 cursor-not-allowed');
				$container.hide().empty();
				$empty.hide();
				$loading.removeClass('hidden').show();

				$.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'slipstream_scan_merge_tags',
						_ajax_nonce: '<?php echo wp_create_nonce("slipstream_scan_merge_tags"); ?>'
					},
					success: function(response) {
						$loading.hide();
						$btn.prop('disabled', false).removeClass('opacity-50 cursor-not-allowed');

						if (response.success && response.data.length > 0) {
							// Group by slug
							const grouped = {};
							response.data.forEach(item => {
								if (!grouped[item.slug]) {
									grouped[item.slug] = {
										slug: item.slug,
										tag: item.tag,
										exists: item.exists,
										items: []
									};
								}
								grouped[item.slug].items.push(item);
							});

							Object.values(grouped).forEach(group => {
								const statusBadge = group.exists 
									? '<span class="px-2 py-0.5 rounded text-[10px] font-bold uppercase bg-emerald-100 text-emerald-700 ml-3">Registered</span>'
									: '<span class="px-2 py-0.5 rounded text-[10px] font-bold uppercase bg-rose-100 text-rose-700 ml-3">Missing</span>';
								
								let rows = '';
								group.items.forEach(item => {
									const statusLabel = item.status ? `<span class="px-2 py-0.5 rounded text-[10px] font-medium bg-slate-100 text-slate-600 mr-2 capitalize">${item.status}</span>` : '';
									const typeLabel = item.post_type ? `<span class="text-xs text-slate-400 font-mono mr-2">[${item.post_type}]</span>` : '';

									rows += `
										<div class="flex items-center justify-between p-3 border-b border-slate-50 last:border-0 hover:bg-slate-50/50 transition-colors">
											<div class="flex flex-col">
												<div class="flex items-center">
													${statusLabel}
													<span class="text-sm font-medium text-slate-700">${item.source_label}</span>
												</div>
												<div class="text-[11px] text-slate-400 mt-1 flex items-center">
													<span class="font-bold uppercase mr-1">${item.source_type}:</span> 
													${typeLabel}
													ID: ${item.source_id}
												</div>
											</div>
											<a href="${item.edit_link}" class="text-indigo-600 hover:text-indigo-800 text-xs font-bold transition-colors flex items-center" target="_blank">
												Edit Source <span class="dashicons dashicons-external !text-[14px] !w-[14px] !h-[14px] ml-1"></span>
											</a>
										</div>
									`;
								});

								const groupHtml = `
									<div class="border border-slate-200 rounded-lg overflow-hidden bg-white shadow-sm hover:border-slate-300 transition-all">
										<button type="button" class="ss-toggle-group w-full flex items-center justify-between p-4 text-left hover:bg-slate-50 transition-colors">
											<div class="flex items-center">
												<code class="bg-indigo-50 text-indigo-700 px-2 py-1 rounded text-sm font-bold">${group.tag}</code>
												${statusBadge}
											</div>
											<div class="flex items-center text-slate-500">
												<span class="text-xs font-bold mr-3 bg-slate-100 px-2 py-1 rounded-full">${group.items.length} occurrences</span>
												<span class="dashicons dashicons-arrow-down-alt2 transition-transform duration-300"></span>
											</div>
										</button>
										<div class="ss-group-details hidden border-t border-slate-100 bg-white">
											${rows}
										</div>
									</div>
								`;
								$container.append(groupHtml);
							});
							$container.show();
						} else {
							$empty.show();
						}
					},
					error: function() {
						$loading.hide();
						$btn.prop('disabled', false).removeClass('opacity-50 cursor-not-allowed');
						alert('An error occurred during scanning.');
					}
				});
			});

			$(document).on('click', '.ss-toggle-group', function() {
				const $btn = $(this);
				const $details = $btn.next('.ss-group-details');
				const $icon = $btn.find('.dashicons-arrow-down-alt2');
				
				$details.slideToggle(200);
				$icon.toggleClass('rotate-180');
			});
		});
		</script>
		<?php
	}

	/**
	 * Render Custom Content Editor Page
	 */
	public function render_content_editor_page() {
		$post_id = isset( $_GET['post_id'] ) ? intval( $_GET['post_id'] ) : 0;
		if ( ! $post_id ) {
			wp_die( __( 'Invalid request.', 'slipstream' ) );
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			wp_die( __( 'Post not found.', 'slipstream' ) );
		}

		// Check if the user can edit this post
		if ( ! current_user_can( 'edit_post', $post_id ) && ! current_user_can( 'is_slipstream_owner' ) ) {
			wp_die( __( 'You do not have permission to edit this content.', 'slipstream' ) );
		}

		$success = get_transient( 'slipstream_success' );
		delete_transient( 'slipstream_success' );

		// Process save if submitted
		if ( isset( $_POST['slipstream_save_content'] ) ) {
			check_admin_referer( 'slipstream_save_content_' . $post_id );
			
			$assigned = get_option( 'slipstream_assigned_content_fields', array() );
			
			foreach ( $assigned as $package_slug => $fields ) {
				if ( ! is_array( $fields ) ) continue;
				foreach ( $fields as $field_slug => $page_ids ) {
					if ( is_array( $page_ids ) && in_array( $post_id, $page_ids ) ) {
						$field_name = 'ss_cf_' . str_replace( '-', '_', $package_slug ) . '_' . $field_slug;
						if ( isset( $_POST[$field_name] ) ) {
							update_post_meta( $post_id, $field_name, wp_unslash( $_POST[$field_name] ) );
						}
					}
				}
			}

			set_transient( 'slipstream_success', 'Content saved successfully.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-content-editor&post_id=' . $post_id ) );
			exit;
		}

		?>
		<div class="wrap ss-modern-admin-wrap" id="slipstream-content-editor">
			
			<div class="flex items-center justify-between mb-8">
				<div class="flex items-center gap-4">
					<h1 class="text-2xl font-bold text-slate-900"><?php echo esc_html( $post->post_title ); ?></h1>
					<a href="<?php echo get_permalink($post_id); ?>" target="_blank" class="text-slate-400 hover:text-indigo-600 transition-all flex items-center" title="<?php esc_attr_e( 'View Page', 'slipstream' ); ?>">
						<span class="dashicons dashicons-external !text-[18px] !w-[18px] !h-[18px] !leading-none"></span>
					</a>
				</div>
				
				<button type="button" onclick="document.getElementById('ss-content-editor-form').submit();" class="bg-indigo-600 text-white px-8 py-2.5 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 transition-all hover:bg-indigo-700 active:scale-95 flex items-center gap-2">
					<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg>
					<span><?php _e( 'Save Changes', 'slipstream' ); ?></span>
				</button>
			</div>

			<div class="grid grid-cols-1 gap-8">
				<div class="lg:col-span-3">
					<div class="bg-white rounded-[2.5rem] shadow-sm border border-slate-100 overflow-hidden mb-12">
						<div class="p-4 px-8 border-b border-slate-50 bg-slate-50/50 flex items-center justify-between">
							<h2 class="text-lg font-bold text-slate-900"><?php _e( 'Page Content', 'slipstream' ); ?></h2>
							<span class="text-xs font-bold text-slate-400 uppercase tracking-widest"><?php _e( 'Assigned Fields', 'slipstream' ); ?></span>
						</div>

						<div class="p-8">
							<form id="ss-content-editor-form" method="post" class="space-y-8">
								<?php wp_nonce_field( 'slipstream_save_content_' . $post_id ); ?>
								<input type="hidden" name="slipstream_save_content" value="1">

								<?php 
								$registry = Slipstream_Registry::get_instance();
								$all_fields = $registry->get_content_fields();
								$assigned = get_option( 'slipstream_assigned_content_fields', array() );
								$found_fields = false;

								foreach ( $assigned as $package_slug => $fields ) {
									if ( ! is_array( $fields ) ) continue;
									
									$package_has_fields = false;
									$package_fields_html = '';

									foreach ( $fields as $field_slug => $page_ids ) {
										if ( is_array( $page_ids ) && in_array( $post_id, $page_ids ) ) {
											if ( isset( $all_fields[$package_slug][$field_slug] ) ) {
												$args = $all_fields[$package_slug][$field_slug];
												$field_name = 'ss_cf_' . str_replace( '-', '_', $package_slug ) . '_' . $field_slug;
												$value = get_post_meta( $post_id, $field_name, true );
												
												$package_has_fields = true;
												$found_fields = true;

												ob_start();
												?>
												<div class="ss-field-item py-4 first:pt-0 last:pb-0 border-b last:border-0 border-slate-100/50">
													<label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest mb-2 ml-1"><?php echo esc_html( $args['name'] ); ?></label>
													<textarea name="<?php echo esc_attr( $field_name ); ?>" rows="2" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium focus:ring-2 focus:ring-indigo-500 transition-all" placeholder="<?php echo esc_attr( $args['description'] ); ?>"><?php echo esc_textarea( $value ); ?></textarea>
													<div class="flex items-center justify-between mt-2 px-1">
														<p class="text-[9px] text-slate-400 italic"><?php echo esc_html( $args['description'] ); ?></p>
														<code class="text-[9px] font-bold text-indigo-300 bg-indigo-50/50 px-1.5 py-0.5 rounded admin-only">{{<?php echo esc_html( $args['merge_tag'] ); ?>}}</code>
													</div>
												</div>
												<?php
												$package_fields_html .= ob_get_clean();
											}
										}
									}

									if ( $package_has_fields ) {
										?>
										<div class="mb-12 last:mb-0">
											<h3 class="text-[10px] font-black uppercase text-slate-400 tracking-[0.2em] mb-4 ml-2"><?php echo esc_html( str_replace('-', ' ', $package_slug) ); ?></h3>
											<div class="bg-slate-50 border border-slate-100 rounded-[2rem] p-6 space-y-2">
												<?php echo $package_fields_html; ?>
											</div>
										</div>
										<?php
									}
								}

								if ( ! $found_fields ) {
									?>
									<div class="p-12 text-center text-slate-400 font-medium italic">
										<?php _e( 'No fields assigned to this page.', 'slipstream' ); ?>
									</div>
									<?php
								}
								?>
							</form>
						</div>
					</div>
				</div>
			</div>

			<div id="ss-success-toast" class="fixed bottom-10 left-1/2 transform -translate-x-1/2 translate-y-20 opacity-0 transition-all duration-500 bg-slate-900 text-white px-6 py-3 rounded-full shadow-2xl flex items-center gap-3 z-[100000]">
				<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-emerald-400"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
				<span class="text-sm font-semibold"><?php echo esc_html( $success ); ?></span>
			</div>

			<script>
				jQuery(document).ready(function($) {
					<?php if ( $success ) : ?>
						const $toast = $('#ss-success-toast');
						$toast.removeClass('translate-y-20 opacity-0').addClass('translate-y-0 opacity-100');
						setTimeout(() => {
							$toast.addClass('translate-y-20 opacity-0').removeClass('translate-y-0 opacity-100');
						}, 3000);
					<?php endif; ?>

					// Dynamic Save Button Pulse
					$(document).on('input change', '#ss-content-editor-form input, #ss-content-editor-form textarea, #ss-content-editor-form select', function() {
						$('button[onclick*="ss-content-editor-form"]').addClass('animate-pulse bg-emerald-500 shadow-emerald-100').removeClass('bg-indigo-600 shadow-indigo-100');
					});
				});
			</script>
		</div>
		<?php
	}

	/**
	 * Render Developer Config Page
	 */
	public function render_developer_config_page() {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			wp_die( __( 'System is locked down.', 'slipstream' ) );
		}

		$success = get_transient( 'slipstream_success' );
		delete_transient( 'slipstream_success' );
		
		$registry = Slipstream_Registry::get_instance();
		$package_configs = $registry->get_admin_configs();

		// Prepare Core Configs
		$welcome_title = get_option( 'slipstream_welcome_title', 'Welcome back, {{user_name}}!' );
		$welcome_desc = get_option( 'slipstream_welcome_desc', 'Manage your site settings and content from the {{brand_name}} Dashboard.' );
		
		$hidden_metaboxes_raw = get_option( 'slipstream_hidden_metaboxes', '' );
		$hidden_metaboxes_array = array_filter( array_map( 'trim', explode( "\n", $hidden_metaboxes_raw ) ) );

		// Get all registered metaboxes
		$post_types = get_post_types( array( 'public' => true ), 'names' );
		foreach ( $post_types as $post_type ) {
			do_action( 'add_meta_boxes', $post_type, (object) array( 'ID' => 0, 'post_type' => $post_type ) );
			do_action( "add_meta_boxes_{$post_type}", (object) array( 'ID' => 0, 'post_type' => $post_type ) );
		}
		global $wp_meta_boxes;
		$all_metaboxes = array();
		if ( ! empty( $wp_meta_boxes ) ) {
			foreach ( $wp_meta_boxes as $screen => $contexts ) {
				foreach ( $contexts as $context => $priorities ) {
					foreach ( $priorities as $priority => $boxes ) {
						foreach ( $boxes as $id => $box ) {
							if ( ! isset( $all_metaboxes[ $id ] ) ) {
								$all_metaboxes[ $id ] = array(
									'title' => $box['title'] ?? $id,
									'screens' => array( $screen ),
								);
							} else {
								if ( ! in_array( $screen, $all_metaboxes[ $id ]['screens'] ) ) {
									$all_metaboxes[ $id ]['screens'][] = $screen;
								}
							}
						}
					}
				}
			}
		}
		ksort( $all_metaboxes );

		// Merge Core and Package Configs
		$configs = array();
		
		// 1. Dashboard Tab
		$configs['dashboard'] = array(
			'title' => __( 'Dashboard', 'slipstream' ),
			'icon'  => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="7" height="9" x="3" y="3" rx="1"/><rect width="7" height="5" x="14" y="3" rx="1"/><rect width="7" height="9" x="14" y="12" rx="1"/><rect width="7" height="5" x="3" y="16" rx="1"/></svg>',
			'callback' => function() use ( $welcome_title, $welcome_desc ) {
				?>
				<div class="space-y-6">
					<div>
						<h2 class="text-lg font-bold text-slate-900 mb-1"><?php _e( 'Welcome Card Content', 'slipstream' ); ?></h2>
						<p class="text-sm text-slate-500 mb-4"><?php _e( 'Customize the text displayed on the dashboard welcome card.', 'slipstream' ); ?></p>
					</div>
					<div class="space-y-4">
						<div class="ss-field">
							<label class="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-2 ml-1"><?php _e( 'Welcome Title', 'slipstream' ); ?></label>
							<input name="slipstream_welcome_title" type="text" value="<?php echo esc_attr( $welcome_title ); ?>" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium">
							<p class="text-[10px] text-slate-400 mt-2 ml-1"><?php _e( 'Available tags: <code>{{user_name}}</code>, <code>{{brand_name}}</code>', 'slipstream' ); ?></p>
						</div>
						<div class="ss-field">
							<label class="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-2 ml-1"><?php _e( 'Welcome Description', 'slipstream' ); ?></label>
							<textarea name="slipstream_welcome_desc" rows="3" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium"><?php echo esc_textarea( $welcome_desc ); ?></textarea>
							<p class="text-[10px] text-slate-400 mt-2 ml-1"><?php _e( 'Available tags: <code>{{user_name}}</code>, <code>{{brand_name}}</code>', 'slipstream' ); ?></p>
						</div>
					</div>
				</div>
				<?php
			}
		);

		// 2. Visibility Tab
		$configs['visibility'] = array(
			'title' => __( 'Visibility', 'slipstream' ),
			'icon'  => '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M2 12s3-7 10-7 10 7 10 7-3 7-10 7-10-7-10-7Z"/><circle cx="12" cy="12" r="3"/></svg>',
			'callback' => function() use ( $all_metaboxes, $hidden_metaboxes_array, $hidden_metaboxes_raw ) {
				?>
				<div class="space-y-6">
					<div>
						<h2 class="text-lg font-bold text-slate-900 mb-1"><?php _e( 'Metabox Visibility', 'slipstream' ); ?></h2>
						<p class="text-sm text-slate-500 mb-4"><?php _e( 'Select the metaboxes you want to hide from the Owner view.', 'slipstream' ); ?></p>
					</div>

					<?php if ( ! empty( $all_metaboxes ) ) : ?>
						<div class="bg-slate-50 border border-slate-100 rounded-3xl p-6 max-h-[400px] overflow-y-auto custom-scrollbar">
							<div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
								<?php foreach ( $all_metaboxes as $id => $data ) : 
									$is_checked = in_array( $id, $hidden_metaboxes_array );
									?>
									<label class="flex items-start gap-3 p-3 bg-white rounded-2xl border border-slate-100 cursor-pointer hover:border-indigo-200 transition-all">
										<input type="checkbox" name="hidden_metaboxes_list[]" value="<?php echo esc_attr( $id ); ?>" <?php checked( $is_checked ); ?> class="mt-1 rounded text-indigo-600 focus:ring-indigo-500">
										<div class="min-w-0">
											<span class="block text-sm font-bold text-slate-700 truncate"><?php echo esc_html( $data['title'] ); ?></span>
											<code class="text-[10px] text-slate-400"><?php echo esc_html( $id ); ?></code>
										</div>
									</label>
								<?php endforeach; ?>
							</div>
						</div>
					<?php else: ?>
						<div class="p-6 bg-rose-50 border border-rose-100 rounded-3xl text-rose-700 text-sm italic">
							<?php _e( 'No registered metaboxes found. They might only be registered on specific editor screens.', 'slipstream' ); ?>
						</div>
					<?php endif; ?>

					<div class="ss-field pt-4">
						<label class="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-2 ml-1"><?php _e( 'Manual Metabox IDs', 'slipstream' ); ?></label>
						<textarea name="hidden_metaboxes" rows="4" class="w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-mono" placeholder="wpseo_meta&#10;rank_math_metabox"><?php echo esc_textarea( $hidden_metaboxes_raw ); ?></textarea>
						<p class="text-[10px] text-slate-400 mt-2 ml-1"><?php _e( 'Enter one ID per line for metaboxes not listed above.', 'slipstream' ); ?></p>
					</div>
				</div>
				<?php
			}
		);

		// 3. Package Configs
		$packages_data = $registry->get_data( 'panels' );
		$all_content_fields = $registry->get_content_fields();
		$assigned = get_option( 'slipstream_assigned_content_fields', array() );
		$pages = get_pages();

		foreach ( $package_configs as $slug => $data ) {
			$icon = $data['icon'];
			
			// If no specific icon, try to get it from the package panel registration
			if ( ! $icon && isset( $packages_data[$slug] ) && ! empty( $packages_data[$slug]['icon'] ) ) {
				$icon = $packages_data[$slug]['icon'];
			}

			// Fallback icon
			if ( ! $icon ) {
				$icon = '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="m7.5 4.27 9 5.15"/><path d="M21 8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16Z"/><path d="m3.3 7 8.7 5 8.7-5"/><path d="M12 22V12"/></svg>';
			}

			$configs[$slug] = array(
				'title' => $data['title'],
				'icon'  => $icon,
				'callback' => function() use ( $slug, $data, $all_content_fields, $assigned, $pages ) {
					// Render the package's own config
					if ( is_callable( $data['callback'] ) ) {
						call_user_func( $data['callback'] );
					}

					// Render Content Fields for this package if any exist
					if ( ! empty( $all_content_fields[$slug] ) ) {
						?>
						<div class="mt-12 pt-12 border-t border-slate-100">
							<div class="mb-6">
								<h2 class="text-lg font-bold text-slate-900 mb-1"><?php _e( 'Assignable Content Fields', 'slipstream' ); ?></h2>
								<p class="text-sm text-slate-500"><?php _e( 'Map fields registered by this package to specific site pages.', 'slipstream' ); ?></p>
							</div>

							<div class="space-y-4">
								<?php foreach ( $all_content_fields[$slug] as $field_slug => $args ) : 
									$current_assignment = isset( $assigned[$slug][$field_slug] ) ? $assigned[$slug][$field_slug] : array();
									?>
									<div class="p-6 bg-slate-50 rounded-3xl border border-slate-100 space-y-4">
										<div class="flex items-start justify-between">
											<div>
												<h4 class="text-sm font-bold text-slate-900"><?php echo esc_html( $args['name'] ); ?></h4>
												<p class="text-xs text-slate-500"><?php echo esc_html( $args['description'] ); ?></p>
												<code class="inline-block mt-2 px-2 py-0.5 bg-indigo-50 text-indigo-600 rounded text-[10px] font-bold">{{<?php echo esc_html( $args['merge_tag'] ); ?>}}</code>
											</div>
										</div>
										<div class="ss-field">
											<label class="block text-[10px] font-bold text-slate-400 uppercase tracking-wider mb-2 ml-1"><?php _e( 'Assign to Pages', 'slipstream' ); ?></label>
											<div class="ss-multiselect-container relative">
												<select name="assigned_fields[<?php echo esc_attr( $slug ); ?>][<?php echo esc_attr( $field_slug ); ?>][]" class="ss-modern-multiselect w-full bg-white border-slate-200 rounded-2xl p-3 text-sm font-medium" multiple style="display: none;">
													<?php foreach ( $pages as $page ) : ?>
														<option value="<?php echo $page->ID; ?>" <?php echo in_array( $page->ID, $current_assignment ) ? 'selected' : ''; ?>>
															<?php echo esc_html( $page->post_title ); ?>
														</option>
													<?php endforeach; ?>
												</select>
												<div class="ss-multiselect-trigger w-full bg-white border border-slate-200 rounded-2xl p-3 min-h-[50px] cursor-pointer flex flex-wrap gap-2 items-center">
													<?php if ( empty( $current_assignment ) ) : ?>
														<span class="text-slate-400 text-sm"><?php _e( 'Select pages...', 'slipstream' ); ?></span>
													<?php else : ?>
														<?php foreach ( $pages as $page ) : 
															if ( in_array( $page->ID, $current_assignment ) ) : ?>
																<span class="inline-flex items-center gap-1.5 px-3 py-1 bg-indigo-50 text-indigo-600 rounded-xl text-xs font-bold border border-indigo-100">
																	<?php echo esc_html( $page->post_title ); ?>
																	<button type="button" class="ss-remove-selection hover:text-indigo-800" data-id="<?php echo $page->ID; ?>">
																		<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><path d="M18 6 6 18M6 6l12 12"/></svg>
																	</button>
																</span>
															<?php endif; ?>
														<?php endforeach; ?>
													<?php endif; ?>
												</div>
												<div class="ss-multiselect-dropdown absolute top-full left-0 right-0 mt-2 bg-white border border-slate-100 rounded-2xl shadow-xl z-50 max-h-60 overflow-y-auto hidden">
													<div class="p-2 space-y-1">
														<?php foreach ( $pages as $page ) : ?>
															<div class="ss-multiselect-option px-4 py-2.5 rounded-xl text-sm font-medium cursor-pointer hover:bg-slate-50 transition-colors flex items-center justify-between <?php echo in_array( $page->ID, $current_assignment ) ? 'bg-indigo-50/50 text-indigo-600' : 'text-slate-700'; ?>" data-id="<?php echo $page->ID; ?>">
																<?php echo esc_html( $page->post_title ); ?>
																<?php if ( in_array( $page->ID, $current_assignment ) ) : ?>
																	<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="text-indigo-600"><path d="M20 6 9 17l-5-5"/></svg>
																<?php endif; ?>
															</div>
														<?php endforeach; ?>
													</div>
												</div>
											</div>
										</div>
									</div>
								<?php endforeach; ?>
							</div>
						</div>
						<?php
					}
				}
			);
		}

		// Apply filters for configuration tabs
		$configs = apply_filters( 'slipstream_developer_configs', $configs );
		?>
		<div class="wrap ss-developer-config-wrap">
			<form method="post" id="ss-developer-config-form">
				<?php wp_nonce_field( 'slipstream_save_developer_config' ); ?>
				<input type="hidden" name="slipstream_save_developer_config" value="1">
				
				<div class="flex flex-col md:flex-row gap-8 items-start">
					<!-- Sidebar Navigation -->
					<div class="w-full md:w-64 flex-shrink-0 bg-white rounded-3xl shadow-sm border border-slate-100 overflow-hidden sticky top-8">
						<div class="p-4 border-b border-slate-50 bg-slate-50/50">
							<button type="submit" class="w-full bg-indigo-600 text-white py-3 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 transition-all hover:bg-indigo-700 active:scale-95 flex items-center justify-center gap-2">
								<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg>
								<span><?php _e( 'Save All', 'slipstream' ); ?></span>
							</button>
						</div>
						<nav class="flex flex-col p-2 ss-developer-nav">
							<?php 
							$first = true;
							foreach ( $configs as $slug => $data ) : 
								$active_class = $first ? 'bg-indigo-50 text-indigo-600' : 'text-slate-500 hover:bg-slate-50 hover:text-slate-700';
								?>
								<a href="#<?php echo esc_attr( $slug ); ?>" class="flex items-center gap-3 px-4 py-3 rounded-2xl text-sm font-bold transition-all ss-dev-tab-trigger <?php echo $active_class; ?>" data-target="tab-<?php echo esc_attr( $slug ); ?>">
									<span class="flex-shrink-0 opacity-70 ss-icon-container"><?php echo $data['icon']; ?></span>
									<span class="truncate"><?php echo esc_html( $data['title'] ); ?></span>
								</a>
								<?php 
								$first = false;
							endforeach; 
							?>
						</nav>
					</div>

					<!-- Content Area -->
				<div class="flex-1 w-full min-w-0">
					<?php 
					$first = true;
					foreach ( $configs as $slug => $data ) : 
						$display = $first ? 'block' : 'none';
						?>
						<div class="ss-developer-tab-content bg-white rounded-[2.5rem] shadow-sm border border-slate-100 p-8 md:p-10 mb-8" id="tab-<?php echo esc_attr( $slug ); ?>" style="display: <?php echo $display; ?>;">
							<div class="mb-8 pb-6 border-b border-slate-50 flex items-center gap-4">
								<div class="w-12 h-12 rounded-2xl bg-indigo-50 text-indigo-600 flex items-center justify-center ss-icon-container">
									<?php echo $data['icon']; ?>
								</div>
								<div>
									<h2 class="text-xl font-bold text-slate-900"><?php echo esc_html( $data['title'] ); ?></h2>
									<p class="text-xs font-bold text-slate-400 uppercase tracking-widest"><?php echo sprintf( __( '%s Configuration', 'slipstream' ), esc_html( $data['title'] ) ); ?></p>
								</div>
							</div>
							<?php 
								if ( is_callable( $data['callback'] ) ) {
									call_user_func( $data['callback'] );
								}
							?>
						</div>
						<?php 
						$first = false;
					endforeach; 
					?>
				</div>
			</div>

			<div id="ss-success-toast" class="fixed bottom-10 left-1/2 transform -translate-x-1/2 translate-y-20 opacity-0 transition-all duration-500 bg-slate-900 text-white px-6 py-3 rounded-full shadow-2xl flex items-center gap-3 z-[100000]">
				<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-emerald-400"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
				<span class="text-sm font-semibold"><?php echo esc_html( $success ); ?></span>
			</div>

			<script>
				jQuery(document).ready(function($) {
					// Handle Success Toast
					<?php if ( $success ) : ?>
						const $toast = $('#ss-success-toast');
						$toast.removeClass('translate-y-20 opacity-0').addClass('translate-y-0 opacity-100');
						setTimeout(() => {
							$toast.addClass('translate-y-20 opacity-0').removeClass('translate-y-0 opacity-100');
						}, 3000);
					<?php endif; ?>

					// Tab switching
					$('.ss-dev-tab-trigger').on('click', function(e) {
						e.preventDefault();
						const $this = $(this);
						const target = $this.data('target');
						
						// Sidebar styling
						$('.ss-dev-tab-trigger').removeClass('bg-indigo-50 text-indigo-600').addClass('text-slate-500 hover:bg-slate-50 hover:text-slate-700');
						$this.addClass('bg-indigo-50 text-indigo-600').removeClass('text-slate-500 hover:bg-slate-50 hover:text-slate-700');
						
						// Content switching
						$('.ss-developer-tab-content').hide();
						$('#' + target).fadeIn(200);
						
						// Update hash
						window.location.hash = 'dev-' + target.replace('tab-', '');
					});

					// Handle Save Button state
					const $saveBtn = $('#ss-developer-config-form button[type="submit"]');
					const originalBtnText = $saveBtn.html();
					
					$(document).on('input change', '#ss-developer-config-form input, #ss-developer-config-form textarea, #ss-developer-config-form select', function() {
						$saveBtn.removeClass('bg-indigo-600 shadow-indigo-100').addClass('bg-emerald-500 shadow-emerald-100 animate-pulse');
						$saveBtn.find('span').text('<?php _e( "Save Changes *", "slipstream" ); ?>');
					});

					// Handle instant conditionals
					function handleConditionals() {
						$('.ss-toggle-conditional').each(function() {
							const $this = $(this);
							const target = $this.data('target');
							const $target = $('#' + target);
							
							if ($this.is(':checkbox')) {
								if ($this.is(':checked')) {
									$target.show();
								} else {
									$target.hide();
								}
							} else if ($this.is('select') || $this.is(':radio')) {
								const val = $this.val();
								const expected = $this.data('value');
								if (val == expected) {
									$target.show();
								} else {
									$target.hide();
								}
							}
						});
					}

					$(document).on('change input', '.ss-toggle-conditional', function() {
						const $this = $(this);
						const target = $this.data('target');
						const $target = $('#' + target);

						if ($this.is(':checkbox')) {
							if ($this.is(':checked')) {
								$target.stop(true, true).slideDown(200);
							} else {
								$target.stop(true, true).slideUp(200);
							}
						} else if ($this.is('select') || $this.is(':radio')) {
							const val = $this.val();
							const expected = $this.data('value');
							if (val == expected) {
								$target.stop(true, true).slideDown(200);
							} else {
								$target.stop(true, true).slideUp(200);
							}
						}
					});

						// Initial run
						handleConditionals();

						// Handle hash on load
						if (window.location.hash && window.location.hash.indexOf('#dev-') === 0) {
							const slug = window.location.hash.replace('#dev-', '');
							const $tab = $(`.ss-dev-tab-trigger[data-target="tab-${slug}"]`);
							if ($tab.length) {
								$tab.trigger('click');
							}
						}

						// Modern Multiselect Logic
						$(document).on('click', '.ss-multiselect-trigger', function(e) {
							e.stopPropagation();
							const $dropdown = $(this).siblings('.ss-multiselect-dropdown');
							$('.ss-multiselect-dropdown').not($dropdown).addClass('hidden');
							$dropdown.toggleClass('hidden');
						});

						$(document).on('click', '.ss-multiselect-option', function(e) {
							e.stopPropagation();
							const $option = $(this);
							const id = $option.data('id');
							const $container = $option.closest('.ss-multiselect-container');
							const $select = $container.find('select');
							const $trigger = $container.find('.ss-multiselect-trigger');
							
							const $realOption = $select.find(`option[value="${id}"]`);
							const isSelected = $realOption.prop('selected');
							
							if (isSelected) {
								$realOption.prop('selected', false);
								$option.removeClass('bg-indigo-50/50 text-indigo-600').addClass('text-slate-700');
								$option.find('svg').remove();
							} else {
								$realOption.prop('selected', true);
								$option.addClass('bg-indigo-50/50 text-indigo-600').removeClass('text-slate-700');
								if (!$option.find('svg').length) {
									$option.append('<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" class="text-indigo-600"><path d="M20 6 9 17l-5-5"/></svg>');
								}
							}
							
							$select.trigger('change');
							updateMultiselectTrigger($container);
						});

						$(document).on('click', '.ss-remove-selection', function(e) {
							e.stopPropagation();
							const id = $(this).data('id');
							const $container = $(this).closest('.ss-multiselect-container');
							const $select = $container.find('select');
							
							$select.find(`option[value="${id}"]`).prop('selected', false);
							$select.trigger('change');
							
							const $option = $container.find(`.ss-multiselect-option[data-id="${id}"]`);
							$option.removeClass('bg-indigo-50/50 text-indigo-600').addClass('text-slate-700');
							$option.find('svg').remove();
							
							updateMultiselectTrigger($container);
						});

						$(document).on('click', function() {
							$('.ss-multiselect-dropdown').addClass('hidden');
						});

						function updateMultiselectTrigger($container) {
							const $select = $container.find('select');
							const $trigger = $container.find('.ss-multiselect-trigger');
							const selectedOptions = $select.find('option:selected');
							
							if (selectedOptions.length === 0) {
								$trigger.html('<span class="text-slate-400 text-sm"><?php _e( "Select pages...", "slipstream" ); ?></span>');
							} else {
								let html = '';
								selectedOptions.each(function() {
									html += `
										<span class="inline-flex items-center gap-1.5 px-3 py-1 bg-indigo-50 text-indigo-600 rounded-xl text-xs font-bold border border-indigo-100">
											${$(this).text()}
											<button type="button" class="ss-remove-selection hover:text-indigo-800" data-id="${$(this).val()}">
												<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round"><path d="M18 6 6 18M6 6l12 12"/></svg>
											</button>
										</span>
									`;
								});
								$trigger.html(html);
							}
						}
					});
				</script>
			</form>
		</div>
		<style>
			.ss-developer-config-wrap { margin-top: 2rem; max-width: 1200px; margin-left: auto; margin-right: auto; padding-right: 2rem; }
			.ss-developer-tab-content .form-table th { width: 200px; padding: 20px 10px 20px 0; font-weight: 700; color: #1e293b; }
			.ss-developer-tab-content .form-table td { padding: 15px 0; }
			.ss-developer-tab-content .large-text { border-radius: 1rem; border: 1px solid #e2e8f0; padding: 0.75rem 1rem; }
			.ss-developer-tab-content hr { border: 0; border-top: 1px solid #f1f5f9; margin: 2rem 0; }
			.ss-developer-nav { max-height: calc(100vh - 200px); overflow-y: auto; }
			.ss-developer-nav::-webkit-scrollbar { width: 4px; }
			.ss-developer-nav::-webkit-scrollbar-thumb { background: #e2e8f0; border-radius: 10px; }
			.ss-icon-container svg { width: 1.25rem; height: 1.25rem; display: block; }
			.ss-developer-tab-content .ss-icon-container svg { width: 1.5rem; height: 1.5rem; }
		</style>
		<?php
	}
}
