<?php
/**
 * API Class - Handles remote authenticated requests
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_API {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		register_rest_route( 'slipstream/v1', '/lockdown', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'handle_lockdown' ),
			'permission_callback' => array( $this, 'check_permission' ),
		) );
	}

	/**
	 * Permission callback for the lockdown endpoint
	 */
	public function check_permission( $request ) {
		$api_key = $request->get_header( 'X-Slipstream-Key' );
		$stored_key = get_option( 'slipstream_remote_key' );

		if ( empty( $stored_key ) ) {
			return false;
		}

		return $api_key === $stored_key;
	}

	/**
	 * Handle the lockdown request
	 */
	public function handle_lockdown( $request ) {
		update_option( 'slipstream_locked_down', 'yes' );

		// Physically remove the update code block from slipstream.php
		$slipstream_file = SLIPSTREAM_PATH . 'slipstream.php';
		if ( file_exists( $slipstream_file ) ) {
			$content = file_get_contents( $slipstream_file );
			
			// Pattern to match the UPDATES section
			$pattern = '/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\s+UPDATES\s+\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\n.*$/s';
			
			if ( preg_match( $pattern, $content ) ) {
				$new_content = preg_replace( $pattern, '', $content );
				file_put_contents( $slipstream_file, trim( $new_content ) . "\n" );
			} else {
				// Fallback pattern if the exact number of slashes or spacing differs
				$pattern_fallback = '/\/\/+\s+UPDATES\s+\/\/+.*$/s';
				if ( preg_match( $pattern_fallback, $content ) ) {
					$new_content = preg_replace( $pattern_fallback, '', $content );
					file_put_contents( $slipstream_file, trim( $new_content ) . "\n" );
				}
			}
		}

		Slipstream_Activity_Log::get_instance()->log( 'System Lockdown', 'Site has been locked down via API request and update code removed.' );

		return new WP_REST_Response( array(
			'success' => true,
			'message' => 'Site has been successfully locked down and update code removed.',
		), 200 );
	}
}
