<?php
/**
 * Assets Manager for Slipstream
 * Handles the custom assets directory in the uploads folder.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Assets {

	private static $instance;
	private $assets_dir_name = 'slipstream_assets';

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		// Ensure directory exists on init
		add_action( 'admin_init', array( $this, 'ensure_assets_dir' ) );
	}

	/**
	 * Get the full path to the assets directory
	 */
	public function get_base_path() {
		$upload_dir = wp_get_upload_dir();
		return $upload_dir['basedir'] . DIRECTORY_SEPARATOR . $this->assets_dir_name;
	}

	/**
	 * Get the URL to the assets directory
	 */
	public function get_base_url() {
		$upload_dir = wp_get_upload_dir();
		return $upload_dir['baseurl'] . '/' . $this->assets_dir_name;
	}

	/**
	 * Ensure the assets directory exists and is protected from indexing
	 */
	public function ensure_assets_dir() {
		$path = $this->get_base_path();

		if ( ! file_exists( $path ) ) {
			wp_mkdir_p( $path );
		}

		// Add an index.php to prevent directory listing
		$index_file = $path . DIRECTORY_SEPARATOR . 'index.php';
		if ( ! file_exists( $index_file ) ) {
			file_put_contents( $index_file, '<?php // Silence is golden' );
		}
	}

	/**
	 * Save a file to the assets directory
	 *
	 * @param string $filename The filename.
	 * @param string $content  The file content (binary or text).
	 * @return string|false    The file URL on success, false on failure.
	 */
	public function save_file( $filename, $content ) {
		$this->ensure_assets_dir();
		$path = $this->get_base_path() . DIRECTORY_SEPARATOR . $filename;

		if ( file_put_contents( $path, $content ) !== false ) {
			return $this->get_base_url() . '/' . $filename;
		}

		return false;
	}

	/**
	 * Delete a file from the assets directory
	 */
	public function delete_file( $filename ) {
		$path = $this->get_base_path() . DIRECTORY_SEPARATOR . $filename;
		if ( file_exists( $path ) ) {
			return unlink( $path );
		}
		return false;
	}

	/**
	 * Get file URL
	 */
	public function get_file_url( $filename ) {
		$path = $this->get_base_path() . DIRECTORY_SEPARATOR . $filename;
		if ( file_exists( $path ) ) {
			return $this->get_base_url() . '/' . $filename;
		}
		return false;
	}
}
