<?php
/**
 * Menu Handler Class - Modern UI & Injection Fixed
 */

if ( ! defined( 'ABSPATH' ) ) exit;

final class Slipstream_Menu_Handler {

    private static $instance;

    public static function get_instance() {
        if ( ! isset( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_filter( 'wp_nav_menu_objects', array( $this, 'inject_late' ), 10, 2 );
        add_action( 'admin_init', array( $this, 'register_menu_metabox' ) );
        add_action( 'wp_nav_menu_item_custom_fields', array( $this, 'add_custom_fields' ), 10, 4 );
        add_action( 'wp_update_nav_menu_item', array( $this, 'save_custom_fields' ), 10, 2 );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_filter( 'wp_setup_nav_menu_item', array( $this, 'setup_nav_menu_item' ) );
    }

    public function enqueue_admin_assets( $hook ) {
        if ( 'nav-menus.php' !== $hook ) return;

        wp_add_inline_style( 'nav-menu', "
			.dml-modern-wrapper { 
				background: #ffffff; 
				padding: 18px; 
				border: 1px solid #dcdcde; 
				border-left: 4px solid #2271b1; 
				margin: 12px 0; 
				border-radius: 4px;
				box-shadow: 0 2px 5px rgba(0,0,0,0.04);
			}
			.dml-modern-wrapper h4 { 
				margin: 0 0 16px 0; 
				color: #1d2327; 
				font-size: 11px; 
				letter-spacing: 0.05em;
				display: flex; 
				align-items: center; 
				gap: 10px; 
				text-transform: uppercase;
				border-bottom: 1px solid #f0f0f1;
				padding-bottom: 10px;
			}
			.dml-modern-wrapper h4 .dashicons { font-size: 16px; color: #2271b1; }
			.dml-modern-wrapper label { font-weight: 700; display: block; margin-bottom: 6px; font-size: 10px; color: #646970; }
			.dml-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
			.dml-full { grid-column: span 2; }
			.dml-mt-15 { margin-top: 15px; }
			.dml-field select, .dml-field input { 
				width: 100%; height: 32px; border-radius: 4px; border: 1px solid #8c8f94; background-color: #fcfcfc; padding: 0 8px;
			}
			.dml-badge { background: #f0f6fb; color: #2271b1; padding: 2px 8px; border-radius: 12px; font-size: 9px; font-weight: bold; border: 1px solid #c3d4e4; }
		");
    }

    public function setup_nav_menu_item( $item ) {
        if ( isset($item->url) && $item->url === '#dynamic' ) {
            $item->type_label = __( 'Dynamic Engine', 'slipstream' );
            if ( ! in_array( 'is-dynamic-placeholder', (array) $item->classes ) ) {
                $item->classes[] = 'is-dynamic-placeholder';
            }
        }
        return $item;
    }

    public function save_custom_fields($menu_id, $menu_item_db_id) {
        $fields = ['dml_post_type', 'dml_taxonomy', 'dml_limit', 'dml_orderby', 'dml_order', 'dml_operator', 'dml_title_meta'];
        foreach ($fields as $field) {
            if (isset($_POST[$field][$menu_item_db_id])) {
                update_post_meta($menu_item_db_id, '_' . $field, sanitize_text_field($_POST[$field][$menu_item_db_id]));
            }
        }
    }

    public function add_custom_fields($item_id, $item, $depth, $args) {
        if (!in_array('is-dynamic-placeholder', (array)$item->classes) && $item->url !== '#dynamic') return;

        $pt_selected      = get_post_meta($item_id, '_dml_post_type', true) ?: 'post';
        $tax_selected     = get_post_meta($item_id, '_dml_taxonomy', true) ?: '';
        $limit            = get_post_meta($item_id, '_dml_limit', true) ?: '5';
        $orderby_selected = get_post_meta($item_id, '_dml_orderby', true) ?: 'date';
        $order_selected   = get_post_meta($item_id, '_dml_order', true) ?: 'DESC';
        $operator         = get_post_meta($item_id, '_dml_operator', true) ?: 'IN';
        $title_meta       = get_post_meta($item_id, '_dml_title_meta', true) ?: '';

        $post_types = get_post_types(['public' => true], 'objects');
        $taxonomies = get_taxonomies(['public' => true], 'objects');
        ?>
        <div class="dml-modern-wrapper">
            <h4><span class="dashicons dashicons-database"></span> DYNAMIC ENGINE <span class="dml-badge">ACTIVE</span></h4>

            <div class="dml-grid">
                <div class="dml-field">
                    <label>Source Type</label>
                    <select name="dml_post_type[<?php echo $item_id; ?>]">
                        <?php foreach ($post_types as $pt) : ?>
                            <option value="<?php echo $pt->name; ?>" <?php selected($pt_selected, $pt->name); ?>><?php echo $pt->label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="dml-field">
                    <label>Limit</label>
                    <input type="number" name="dml_limit[<?php echo $item_id; ?>]" value="<?php echo esc_attr($limit); ?>">
                </div>
                <div class="dml-field dml-full">
                    <label>Taxonomy Filter</label>
                    <select name="dml_taxonomy[<?php echo $item_id; ?>]">
                        <option value="">-- All Posts --</option>
                        <?php foreach ($taxonomies as $tax) : ?>
                            <optgroup label="<?php echo $tax->label; ?>">
                                <?php
                                $terms = get_terms(['taxonomy' => $tax->name, 'hide_empty' => false]);
                                if(!is_wp_error($terms)) {
                                    foreach($terms as $term) {
                                        $val = $tax->name . ':' . $term->slug;
                                        echo '<option value="'.$val.'" '.selected($tax_selected, $val, false).'>'.$term->name.'</option>';
                                    }
                                }
                                ?>
                            </optgroup>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="dml-grid dml-mt-15">
                <div class="dml-field">
                    <label>Condition</label>
                    <select name="dml_operator[<?php echo $item_id; ?>]">
                        <option value="IN" <?php selected($operator, 'IN'); ?>>Include Selected</option>
                        <option value="NOT IN" <?php selected($operator, 'NOT IN'); ?>>Exclude Selected</option>
                    </select>
                </div>
                <div class="dml-field">
                    <label>Sort By</label>
                    <select name="dml_orderby[<?php echo $item_id; ?>]">
                        <option value="date" <?php selected($orderby_selected, 'date'); ?>>Date</option>
                        <option value="title" <?php selected($orderby_selected, 'title'); ?>>Title</option>
                        <option value="menu_order" <?php selected($orderby_selected, 'menu_order'); ?>>Menu Order</option>
                        <option value="rand" <?php selected($orderby_selected, 'rand'); ?>>Random</option>
                    </select>
                </div>
            </div>

            <div class="dml-grid dml-mt-15">
                <div class="dml-field dml-full">
                    <label>Title Meta Key Override (Optional)</label>
                    <input type="text" name="dml_title_meta[<?php echo $item_id; ?>]" value="<?php echo esc_attr($title_meta); ?>" placeholder="e.g. short_title">
                </div>
            </div>
        </div>
        <?php
    }

    public function inject_late($sorted_menu_items, $args) {
        if (is_admin()) return $sorted_menu_items;

        $new_items = [];
        $fake_id = 999000;

        foreach ($sorted_menu_items as $item) {
            if ($item->url === '#dynamic' || in_array('is-dynamic-placeholder', (array)$item->classes)) {

                $pt      = get_post_meta($item->ID, '_dml_post_type', true) ?: 'post';
                $limit   = (int)get_post_meta($item->ID, '_dml_limit', true) ?: 5;
                $tax     = get_post_meta($item->ID, '_dml_taxonomy', true);
                $orderby = get_post_meta($item->ID, '_dml_orderby', true) ?: 'date';
                $order   = get_post_meta($item->ID, '_dml_order', true) ?: 'DESC';
                $operator = get_post_meta($item->ID, '_dml_operator', true) ?: 'IN';
                $title_meta_key = get_post_meta($item->ID, '_dml_title_meta', true);

                $query_args = [
                        'post_type'      => $pt,
                        'posts_per_page' => $limit,
                        'post_status'    => 'publish',
                        'orderby'        => $orderby,
                        'order'          => $order
                ];

                if ($orderby === 'menu_order' && $order === 'DESC') {
                    $query_args['order'] = 'ASC';
                }

                if (!empty($tax) && strpos($tax, ':') !== false) {
                    list($taxonomy, $slug) = explode(':', $tax);
                    $query_args['tax_query'] = [['taxonomy' => $taxonomy, 'field' => 'slug', 'terms' => $slug, 'operator' => $operator]];
                }

                $posts = get_posts($query_args);
                foreach ($posts as $post) {
                    $unique_id = $fake_id++;
                    $p_item = clone $item;
                    $p_item->ID               = $unique_id;
                    $p_item->db_id            = $unique_id;

                    $display_title = get_the_title($post->ID);
                    if (!empty($title_meta_key)) {
                        $meta_title = get_post_meta($post->ID, $title_meta_key, true);
                        if (!empty($meta_title)) {
                            $display_title = $meta_title;
                        }
                    }

                    $p_item->title            = $display_title;
                    $p_item->url              = get_permalink($post->ID);
                    $p_item->object_id        = $post->ID;
                    $p_item->type_label       = '';
                    $p_item->classes          = array_diff((array)$item->classes, ['is-dynamic-placeholder']);
                    $p_item->classes[]        = 'dynamic-item';
                    $new_items[] = $p_item;
                }
            } else {
                $new_items[] = $item;
            }
        }
        return $new_items;
    }

    public function register_menu_metabox() {
        add_meta_box('slipstream-dynamic-menus', 'Dynamic Engine', function() {
            global $_nav_menu_placeholder, $nav_menu_selected_id;
            ?>
            <div id="dml-metabox" class="posttypediv">
                <div class="tabs-panel tabs-panel-active">
                    <ul class="categorychecklist form-no-clear">
                        <li>
                            <label class="menu-item-title">
                                <input type="checkbox" class="menu-item-checkbox" name="menu-item[-1][menu-item-object-id]" value="-1">
                                Add Dynamic Engine Placeholder
                            </label>
                            <input type="hidden" class="menu-item-type" name="menu-item[-1][menu-item-type]" value="custom">
                            <input type="hidden" class="menu-item-title" name="menu-item[-1][menu-item-title]" value="Dynamic Engine">
                            <input type="hidden" class="menu-item-url" name="menu-item[-1][menu-item-url]" value="#dynamic">
                            <input type="hidden" class="menu-item-classes" name="menu-item[-1][menu-item-classes]" value="is-dynamic-placeholder">
                        </li>
                    </ul>
                </div>
                <p class="button-controls">
					<span class="add-to-menu">
						<input type="submit" class="button-secondary submit-add-to-menu right" value="Add to Menu" id="submit-dml-metabox" name="add-dml-menu-item">
						<span class="spinner"></span>
					</span>
                </p>
            </div>
            <?php
        }, 'nav-menus', 'side', 'default');
    }
}