<?php
/**
 * Merge Tags Handler Class
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Merge_Tags {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_filter( 'the_content', array( $this, 'process_merge_tags' ) );
		add_filter( 'widget_text', array( $this, 'process_merge_tags' ) );
		
		// Divi specific filters
		add_filter( 'et_builder_render_layout', array( $this, 'process_merge_tags' ) );
		
		// Admin Bar
		add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 100 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_admin_bar_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_bar_scripts' ) );

		// AJAX for scanner
		add_action( 'wp_ajax_slipstream_scan_merge_tags', array( $this, 'handle_scan_ajax' ) );
	}

	/**
	 * AJAX handler for scanning merge tags
	 */
	public function handle_scan_ajax() {
		check_ajax_referer( 'slipstream_scan_merge_tags' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}

		$results = $this->scan_for_tags();
		wp_send_json_success( $results );
	}

	/**
	 * Scan database for merge tags
	 */
	public function scan_for_tags() {
		global $wpdb;
		$found_tags = array();
		$registry = Slipstream_Registry::get_instance();
		$registered_tags = $registry->get_data( 'merge_tags' );
		$registered_slugs = array_keys( $registered_tags );

		// Get Theme Builder Template mapping if Divi is active
		$theme_builder_map = array();
		if ( $wpdb->get_var( "SELECT count(*) FROM {$wpdb->posts} WHERE post_type IN ('et_header_layout', 'et_body_layout', 'et_footer_layout') LIMIT 1" ) ) {
			// Query for templates to build a map of layout_id => template_title
			$templates = $wpdb->get_results( "SELECT ID, post_title, post_content FROM {$wpdb->posts} WHERE post_type = 'et_theme_builder' AND post_status = 'publish'" );
			foreach ( $templates as $template ) {
				// Divi templates store layout IDs in post_content as JSON or serialized? 
				// Actually they use meta or something else usually, but the user provided a bundle JSON.
				// In the DB, et_theme_builder posts have meta _et_header_layout_id, _et_body_layout_id, _et_footer_layout_id
			}
			
			$template_meta = $wpdb->get_results( "SELECT post_id, meta_key, meta_value FROM {$wpdb->postmeta} WHERE meta_key IN ('_et_header_layout_id', '_et_body_layout_id', '_et_footer_layout_id')" );
			$template_names = $wpdb->get_results( "SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'et_theme_builder'", OBJECT_K );
			
			foreach ( $template_meta as $meta ) {
				$layout_id = (int) $meta->meta_value;
				if ( $layout_id && isset( $template_names[$meta->post_id] ) ) {
					$template_name = $template_names[$meta->post_id]->post_title;
					if ( empty( $template_name ) ) {
						$template_name = 'Default Website Template'; // Often the default has no title
					}
					$theme_builder_map[$layout_id] = $template_name;
				}
			}
		}

		// 1. Scan Posts (content and title)
		$posts = $wpdb->get_results( "SELECT ID, post_title, post_content, post_type, post_status FROM {$wpdb->posts} WHERE (post_content LIKE '%{{%}}%' OR post_title LIKE '%{{%}}%') AND post_type != 'revision'" );
		foreach ( $posts as $post ) {
			$label = $post->post_title;
			if ( $post->post_type === 'et_body_layout' && isset( $theme_builder_map[$post->ID] ) ) {
				$label .= ' [Template: ' . $theme_builder_map[$post->ID] . ']';
			} elseif ( $post->post_type === 'et_header_layout' && isset( $theme_builder_map[$post->ID] ) ) {
				$label .= ' [Template: ' . $theme_builder_map[$post->ID] . ']';
			} elseif ( $post->post_type === 'et_footer_layout' && isset( $theme_builder_map[$post->ID] ) ) {
				$label .= ' [Template: ' . $theme_builder_map[$post->ID] . ']';
			}

			$this->extract_tags_from_string( $post->post_content, 'post', $post->ID, $label, $found_tags, $registered_slugs, $post );
			$this->extract_tags_from_string( $post->post_title, 'post_title', $post->ID, $label, $found_tags, $registered_slugs, $post );
		}

		// 2. Scan Post Meta
		$meta = $wpdb->get_results( "SELECT pm.post_id, pm.meta_key, pm.meta_value, p.post_title, p.post_type, p.post_status FROM {$wpdb->postmeta} pm JOIN {$wpdb->posts} p ON pm.post_id = p.ID WHERE pm.meta_value LIKE '%{{%}}%' AND p.post_type != 'revision'" );
		foreach ( $meta as $m ) {
			if ( is_serialized( $m->meta_value ) ) {
				continue; // Skip serialized for now or handle specifically if needed
			}
			
			$label = $m->post_title;
			if ( ( $m->post_type === 'et_body_layout' || $m->post_type === 'et_header_layout' || $m->post_type === 'et_footer_layout' ) && isset( $theme_builder_map[$m->post_id] ) ) {
				$label .= ' [Template: ' . $theme_builder_map[$m->post_id] . ']';
			}
			$label .= ' (' . $m->meta_key . ')';

			$this->extract_tags_from_string( $m->meta_value, 'postmeta', $m->post_id, $label, $found_tags, $registered_slugs, $m );
		}

		// 3. Scan Options
		$options = $wpdb->get_results( "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_value LIKE '%{{%}}%'" );
		foreach ( $options as $opt ) {
			if ( is_serialized( $opt->option_value ) ) {
				continue;
			}
			// Skip some internal WP options that might contain noise
			if ( strpos( $opt->option_name, '_' ) === 0 || $opt->option_name === 'slipstream_registry' ) {
				continue;
			}
			$this->extract_tags_from_string( $opt->option_value, 'option', $opt->option_name, $opt->option_name, $found_tags, $registered_slugs );
		}

		return array_values( $found_tags );
	}

	/**
	 * Helper to extract tags from a string and add to found_tags array
	 */
	private function extract_tags_from_string( $string, $source_type, $source_id, $source_label, &$found_tags, $registered_slugs, $context = null ) {
		if ( empty( $string ) || ! is_string( $string ) ) {
			return;
		}

		// Pattern for {{tag_slug}} or ##tag_slug## or {{if:tag_slug}} or {{if:!tag_slug}}
		// Also handle {{tag_slug|format}} or ##tag_slug|format##
		preg_match_all( '/(?:\{\{|##)([a-zA-Z0-9_\-|:!]+)(?:\}\}|##)/', $string, $matches );

		if ( ! empty( $matches[1] ) ) {
			foreach ( $matches[1] as $index => $match ) {
				$full_tag = $matches[0][$index];
				
				// Clean the tag to get the base slug
				$tag_slug = $match;
				if ( strpos( $tag_slug, 'if:' ) === 0 ) {
					$tag_slug = substr( $tag_slug, 3 );
					if ( strpos( $tag_slug, '!' ) === 0 ) {
						$tag_slug = substr( $tag_slug, 1 );
					}
				}
				if ( strpos( $tag_slug, '|' ) !== false ) {
					$tag_slug = explode( '|', $tag_slug )[0];
				}

				$exists = in_array( $tag_slug, $registered_slugs );
				
				$key = $tag_slug . '_' . $source_type . '_' . $source_id;
				
				if ( ! isset( $found_tags[$key] ) ) {
					$status = '';
					$post_type = '';
					if ( $context ) {
						$status = isset( $context->post_status ) ? $context->post_status : '';
						$post_type = isset( $context->post_type ) ? $context->post_type : '';
					}

					$found_tags[$key] = array(
						'tag' => $full_tag,
						'slug' => $tag_slug,
						'source_type' => $source_type,
						'source_id' => $source_id,
						'source_label' => $source_label,
						'exists' => $exists,
						'status' => $status,
						'post_type' => $post_type,
						'edit_link' => $this->get_edit_link( $source_type, $source_id, $context )
					);
				}
			}
		}
	}

	/**
	 * Get edit link for a source
	 */
	private function get_edit_link( $source_type, $source_id, $context = null ) {
		switch ( $source_type ) {
			case 'post':
			case 'post_title':
			case 'postmeta':
				if ( $context && isset( $context->post_status ) && $context->post_status === 'trash' ) {
					return admin_url( 'edit.php?post_status=trash&post_type=' . ( isset( $context->post_type ) ? $context->post_type : 'post' ) );
				}
				
				// Handle Divi layouts specifically if possible
				if ( $context && isset( $context->post_type ) ) {
					if ( $context->post_type === 'et_pb_layout' ) {
						return admin_url( 'post.php?post=' . $source_id . '&action=edit' );
					}
					if ( $context->post_type === 'et_body_layout' ) {
						return admin_url( 'admin.php?page=et_theme_builder' );
					}
				}

				return get_edit_post_link( $source_id );
			case 'option':
				return admin_url( 'options.php' ); // Generic, could be improved if we know the page
			default:
				return '#';
		}
	}

	/**
	 * Process content and replace merge tags
	 */
	public function process_merge_tags( $content ) {
		if ( empty( $content ) || ! is_string( $content ) ) {
			return $content;
		}

		$registry = Slipstream_Registry::get_instance();
		$tags = $registry->get_data( 'merge_tags' );

		if ( empty( $tags ) ) {
			return $content;
		}

		// 1. Process Conditional Tags first: {{if:tag}}...{{endif}} or ##if:tag##...##endif##
		$content = $this->process_conditional_tags( $content, $tags );

		// 2. Process Standard Tags: {{tag}} or ##tag##
		foreach ( $tags as $tag_slug => $args ) {
			$value = null;
			
			// Try {{tag}}
			$tag_pattern_curly = '{{' . $tag_slug . '}}';
			if ( strpos( $content, $tag_pattern_curly ) !== false ) {
				$value = $this->resolve_tag_value( $args, $tag_slug );
				$content = str_replace( $tag_pattern_curly, $value, $content );
			}

			// Try ##tag##
			$tag_pattern_hash = '##' . $tag_slug . '##';
			if ( strpos( $content, $tag_pattern_hash ) !== false ) {
				if ( $value === null ) {
					$value = $this->resolve_tag_value( $args, $tag_slug );
				}
				$content = str_replace( $tag_pattern_hash, $value, $content );
			}
		}

		return $content;
	}

	/**
	 * Process conditional tags: {{if:tag}}...{{endif}}
	 */
	private function process_conditional_tags( $content, $tags ) {
		// Pattern for {{if:tag_slug}}content{{endif}} or ##if:tag_slug##content##endif##
		// Support both positive and negative checks: {{if:tag}} and {{if:!tag}}
		$pattern = '/(?:\{\{|##)if:(!?)([a-zA-Z0-9_\-]+)(?:\}\}|##)(.*?)(?:\{\{|##)endif(?:\}\}|##)/is';

		return preg_replace_callback( $pattern, function( $matches ) use ( $tags ) {
			$is_negative = $matches[1] === '!';
			$tag_slug = $matches[2];
			$inner_content = $matches[3];

			if ( ! isset( $tags[$tag_slug] ) ) {
				return $is_negative ? $inner_content : '';
			}

			$value = $this->resolve_tag_value( $tags[$tag_slug], $tag_slug );
			$has_value = ! empty( $value );

			if ( ( $has_value && ! $is_negative ) || ( ! $has_value && $is_negative ) ) {
				return $inner_content;
			}

			return '';
		}, $content );
	}

	/**
	 * Resolve the value of a merge tag
	 */
    private function resolve_tag_value( $args, $tag_slug = '' ) {
        $value = '';

        // 1. Get Data Source
        $source = isset( $args['data_source'] ) ? $args['data_source'] : '';

        // --- START FIX ---
        // Handle Closures/Callables immediately to prevent strpos errors
        if ( is_callable( $source ) && ! is_string( $source ) ) {
            $value = call_user_func( $source );
            $source = $value; // Update source so it doesn't fail subsequent string checks
        }
        // --- END FIX ---

        // If tag_slug is provided, check if it's a dynamic content field
        if ( $tag_slug ) {
            $registry = Slipstream_Registry::get_instance();
            $all_content_fields = $registry->get_content_fields();
            $is_content_field = false;
            $package_slug = '';
            $field_slug = '';

            foreach ( $all_content_fields as $p_slug => $fields ) {
                foreach ( $fields as $f_slug => $f_args ) {
                    if ( isset($f_args['merge_tag']) && $f_args['merge_tag'] === $tag_slug ) {
                        $is_content_field = true;
                        $package_slug = $p_slug;
                        $field_slug = $f_slug;
                        break 2;
                    }
                }
            }

            if ( $is_content_field ) {
                $post_id = get_queried_object_id() ?: get_the_ID();

                if ( $post_id ) {
                    $assigned = get_option( 'slipstream_assigned_content_fields', array() );
                    if ( isset( $assigned[$package_slug][$field_slug] ) && in_array( $post_id, $assigned[$package_slug][$field_slug] ) ) {
                        $acf_field_name = 'ss_cf_' . str_replace( '-', '_', $package_slug ) . '_' . $field_slug;
                        $value = get_post_meta( $post_id, $acf_field_name, true );
                    }
                }

                if ( $value !== '' ) {
                    return isset( $args['format'] ) ? $this->apply_format( $value, $args['format'] ) : $value;
                }
            }
        }

        // Ensure source is a string before doing string operations
        if ( is_string( $source ) ) {
            if ( strpos( $source, 'acf:' ) === 0 ) {
                $field_name = str_replace( 'acf:', '', $source );
                if ( function_exists( 'get_field' ) ) {
                    if ( strpos( $field_name, 'option:' ) === 0 ) {
                        $field_name = str_replace( 'option:', '', $field_name );
                        $value = get_field( $field_name, 'option' );
                    } else {
                        $post_id = get_queried_object_id() ?: get_the_ID();
                        $value = get_field( $field_name, $post_id );
                    }
                }
            } elseif ( strpos( $source, 'option:' ) === 0 ) {
                $option_name = str_replace( 'option:', '', $source );
                $value = get_option( $option_name );
            } elseif ( $source === 'site:name' ) {
                $value = get_bloginfo( 'name' );
            } elseif ( $source === 'site:url' ) {
                $value = get_bloginfo( 'url' );
            } else {
                $value = $source; // Literal value
            }
        } elseif ( ! is_callable( $source ) ) {
            $value = $source; // Handle non-string, non-callable types (null, etc)
        }

        // 2. Format / Process
        if ( isset( $args['format'] ) ) {
            $value = $this->apply_format( $value, $args['format'] );
        }

        return $value;
    }

	/**
	 * Apply formatting to a value
	 */
	private function apply_format( $value, $format ) {
		if ( is_callable( $format ) ) {
			return call_user_func( $format, $value );
		}

		switch ( $format ) {
			case 'tel':
				$clean = preg_replace( '/[^0-9+]/', '', $value );
				return sprintf( '<a href="tel:%s">%s</a>', esc_attr( $clean ), esc_html( $value ) );
			case 'email':
				return sprintf( '<a href="mailto:%s">%s</a>', esc_attr( $value ), esc_html( $value ) );
			case 'date':
				return date_i18n( get_option( 'date_format' ), strtotime( $value ) );
			case 'uppercase':
				return strtoupper( $value );
			case 'lowercase':
				return strtolower( $value );
		}

		return $value;
	}

	/**
	 * Add Merge Tags list to Admin Bar
	 */
	public function add_admin_bar_menu( $wp_admin_bar ) {
		if ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'is_slipstream_owner' ) ) {
			return;
		}

		$registry = Slipstream_Registry::get_instance();
		$tags = $registry->get_data( 'merge_tags' );

		if ( empty( $tags ) ) {
			return;
		}

		$wp_admin_bar->add_node( array(
			'id'    => 'slipstream-merge-tags',
			'title' => '<span class="ab-icon dashicons-tag" style="color: #4f46e5 !important;"></span>',
			'meta'  => array(
				'class' => 'slipstream-merge-tags-node',
				'onclick' => 'openSlipstreamMergeTagsModal(); return false;',
				'tabindex' => '0',
				'style' => 'cursor: pointer;',
			),
		) );
	}

	/**
	 * Render the HTML content for the merge tags modal
	 */
	public function render_modal_content() {
		if ( ( ! is_admin_bar_showing() && ! isset( $_GET['et_fb'] ) ) || ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'is_slipstream_owner' ) ) ) {
			return;
		}

		$registry = Slipstream_Registry::get_instance();
		$tags = $registry->get_data( 'merge_tags' );
		$groups = $registry->get_merge_tag_groups();

		// Group the tags
		$grouped_tags = array();
		foreach ( $tags as $slug => $args ) {
			$group = isset( $args['group'] ) ? $args['group'] : 'custom';
			if ( ! isset( $grouped_tags[$group] ) ) {
				$grouped_tags[$group] = array();
			}
			$args['slug'] = $slug;
			$grouped_tags[$group][] = $args;
		}

		?>
		<div id="slipstream-mt-modal" class="slipstream-modal-overlay" style="display: none;">
			<div class="slipstream-modal-container">
				<div class="slipstream-modal-header">
					<div class="header-left">
						<span class="dashicons dashicons-tag"></span>
						<h2><?php _e( 'Merge Tags', 'slipstream' ); ?></h2>
					</div>
					<div class="header-right">
						<div class="modal-search">
							<span class="dashicons dashicons-search"></span>
							<input type="text" id="slipstream-mt-search" placeholder="<?php esc_attr_e( 'Search tags...', 'slipstream' ); ?>" autocomplete="off">
						</div>
						<button class="modal-close" onclick="closeSlipstreamMergeTagsModal()">&times;</button>
					</div>
				</div>
				<div class="slipstream-modal-body">
					<div class="modal-sidebar">
						<ul class="modal-categories">
							<li class="category-item active" data-group="all">
								<?php _e( 'All Tags', 'slipstream' ); ?>
								<span class="count"><?php echo count( $tags ); ?></span>
							</li>
							<?php foreach ( $groups as $group_slug => $group_args ) : 
								if ( empty( $grouped_tags[$group_slug] ) ) continue;
								?>
								<li class="category-item" data-group="<?php echo esc_attr( $group_slug ); ?>">
									<?php echo esc_html( $group_args['name'] ); ?>
									<span class="count"><?php echo count( $grouped_tags[$group_slug] ); ?></span>
								</li>
							<?php endforeach; ?>
							<li class="category-item border-t mt-4 pt-4 text-indigo-600" data-group="help">
								<span class="dashicons dashicons-editor-help !text-[14px] !w-[14px] !h-[14px] !leading-none mr-1"></span>
								<?php _e( 'Syntax Help', 'slipstream' ); ?>
							</li>
						</ul>
					</div>
					<div class="modal-content-area">
						<?php foreach ( $groups as $group_slug => $group_args ) : 
							if ( empty( $grouped_tags[$group_slug] ) ) continue;
							
							// Sort tags by priority within group
							usort( $grouped_tags[$group_slug], function( $a, $b ) {
								return ( $a['priority'] ?? 10 ) - ( $b['priority'] ?? 10 );
							} );
							?>
							<div class="mt-group-section" data-group="<?php echo esc_attr( $group_slug ); ?>">
								<h3 class="mt-group-title"><?php echo esc_html( $group_args['name'] ); ?></h3>
								<div class="mt-items-grid">
									<?php foreach ( $grouped_tags[$group_slug] as $tag ) : ?>
										<div class="mt-card" 
											 data-tag="<?php echo esc_attr( $tag['slug'] ); ?>"
											 onclick="copyToClipboardMT('{{<?php echo esc_js( $tag['slug'] ); ?>}}', this)">
											<div class="mt-card-content">
												<div class="mt-tag-code">{{<?php echo esc_html( $tag['slug'] ); ?>}}</div>
												<?php if ( ! empty( $tag['description'] ) ) : ?>
													<div class="mt-tag-description"><?php echo esc_html( $tag['description'] ); ?></div>
												<?php endif; ?>
											</div>
											<div class="mt-card-action">
												<span class="dashicons dashicons-admin-page"></span>
											</div>
											<div class="mt-copied-overlay">
												<span class="dashicons dashicons-yes"></span>
												<span><?php _e( 'Copied!', 'slipstream' ); ?></span>
											</div>
										</div>
									<?php endforeach; ?>
								</div>
							</div>
						<?php endforeach; ?>

						<div id="mt-help-section" class="mt-group-section" data-group="help" style="display: none;">
							<h3 class="mt-group-title"><?php _e( 'Merge Tag Syntax', 'slipstream' ); ?></h3>
							<div class="bg-slate-50 rounded-2xl p-6 space-y-6 text-sm text-slate-600">
								<div>
									<h4 class="font-bold text-slate-900 mb-2"><?php _e( 'Standard Syntax', 'slipstream' ); ?></h4>
									<p class="mb-2"><?php _e( 'Use curly braces for standard text fields:', 'slipstream' ); ?></p>
									<code class="block bg-white p-3 border border-slate-100 rounded-xl text-indigo-600 mb-2">{{tag_name}}</code>
								</div>
								
								<div>
									<h4 class="font-bold text-slate-900 mb-2"><?php _e( 'Safe Syntax (Divi & URLs)', 'slipstream' ); ?></h4>
									<p class="mb-2"><?php _e( 'Use double hashes for URL fields or when Divi sanitization blocks curly braces:', 'slipstream' ); ?></p>
									<code class="block bg-white p-3 border border-slate-100 rounded-xl text-indigo-600 mb-2">##tag_name##</code>
								</div>

								<div>
									<h4 class="font-bold text-slate-900 mb-2"><?php _e( 'Formatting', 'slipstream' ); ?></h4>
									<p class="mb-2"><?php _e( 'Some tags support automatic formatting (tel, email, etc) which is defined during registration. You can also use conditional logic:', 'slipstream' ); ?></p>
									<code class="block bg-white p-3 border border-slate-100 rounded-xl text-indigo-600 mb-2">{{if:tag_name}} ... {{endif}}</code>
									<p class="mt-2 text-xs italic"><?php _e( 'Note: Conditional logic also supports the safe ## syntax.', 'slipstream' ); ?></p>
								</div>
							</div>
						</div>
						
						<div id="mt-no-results" style="display: none;">
							<div class="no-results-content">
								<span class="dashicons dashicons-search"></span>
								<p><?php _e( 'No tags found matching your search.', 'slipstream' ); ?></p>
							</div>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Enqueue scripts for Admin Bar functionality
	 */
	public function enqueue_admin_bar_scripts() {
		if ( ( ! is_admin_bar_showing() && ! isset( $_GET['et_fb'] ) ) || ( ! current_user_can( 'manage_options' ) && ! current_user_can( 'is_slipstream_owner' ) ) ) {
			return;
		}

		add_action( 'wp_footer', array( $this, 'render_modal_content' ) );
		add_action( 'admin_footer', array( $this, 'render_modal_content' ) );
		
		add_action( 'wp_footer', array( $this, 'render_admin_bar_js' ) );
		add_action( 'admin_footer', array( $this, 'render_admin_bar_js' ) );

		// Divi Visual Builder support
		// Divi uses et_fb=1 in the URL when the visual builder is active.
		// We need to ensure our modal and JS are also injected into the Divi Builder context if possible,
		// but since it's an iframe, we might need to hook into Divi-specific actions.
		add_action( 'et_before_main_content', array( $this, 'render_modal_content' ) );
		add_action( 'et_before_main_content', array( $this, 'render_admin_bar_js' ) );

		// Also hook into after_footer to be super sure
		add_action( 'wp_print_footer_scripts', array( $this, 'render_modal_content' ), 999 );
		add_action( 'wp_print_footer_scripts', array( $this, 'render_admin_bar_js' ), 999 );
	}

	/**
	 * Render JS for Admin Bar search and clipboard
	 */
	public function render_admin_bar_js() {
		static $rendered = false;
		if ( $rendered ) return;
		$rendered = true;
		?>
		<script>
		(function() {
			function openSlipstreamMergeTagsModal() {
				var modal = document.getElementById('slipstream-mt-modal');
				if (modal) {
					modal.style.display = 'flex';
					document.body.style.overflow = 'hidden';
					var searchInput = document.getElementById('slipstream-mt-search');
					if (searchInput) {
						searchInput.focus();
					}
				}
			}

			// Make it global on BOTH window and top window (for iframes)
			window.openSlipstreamMergeTagsModal = openSlipstreamMergeTagsModal;
			if (window.parent && window.parent !== window) {
				window.parent.openSlipstreamMergeTagsModal = openSlipstreamMergeTagsModal;
			}
			if (window.top && window.top !== window) {
				window.top.openSlipstreamMergeTagsModal = openSlipstreamMergeTagsModal;
			}

			function closeSlipstreamMergeTagsModal() {
				var modal = document.getElementById('slipstream-mt-modal');
				if (modal) {
					modal.style.display = 'none';
					document.body.style.overflow = '';
				}
			}
			window.closeSlipstreamMergeTagsModal = closeSlipstreamMergeTagsModal;
			if (window.parent && window.parent !== window) {
				window.parent.closeSlipstreamMergeTagsModal = closeSlipstreamMergeTagsModal;
			}
			if (window.top && window.top !== window) {
				window.top.closeSlipstreamMergeTagsModal = closeSlipstreamMergeTagsModal;
			}

			function copyToClipboardMT(text, element) {
				var dummy = document.createElement("textarea");
				document.body.appendChild(dummy);
				dummy.value = text;
				dummy.select();
				document.execCommand("copy");
				document.body.removeChild(dummy);
				
				// Visual feedback on card
				element.classList.add('is-copied');
				setTimeout(function() {
					element.classList.remove('is-copied');
				}, 1500);

				// Feedback on admin bar if it's visible
				var node = document.getElementById('wp-admin-bar-slipstream-merge-tags');
				if (node) {
					var icon = node.querySelector('.ab-icon');
					if (icon) {
						var originalColor = icon.style.color;
						icon.style.color = '#10b981'; // Success green
						setTimeout(function() {
							icon.style.color = originalColor;
						}, 1000);
					}
				}
			}
			window.copyToClipboardMT = copyToClipboardMT;
			if (window.parent && window.parent !== window) {
				window.parent.copyToClipboardMT = copyToClipboardMT;
			}
			if (window.top && window.top !== window) {
				window.top.copyToClipboardMT = copyToClipboardMT;
			}

			// Divi Visual Builder Integration
			function initDiviIntegration() {
				if (typeof window.ET_Builder === 'undefined' && !document.querySelector('.et-fb-app')) {
					setTimeout(initDiviIntegration, 500);
					return;
				}

				// Try to find Divi's toolbar and add our button if it's not there
				var interval = setInterval(function() {
					var diviToolbar = document.querySelector('.et-fb-settings-bar');
					if (diviToolbar && !document.getElementById('ss-divi-mt-trigger')) {
						var trigger = document.createElement('div');
						trigger.id = 'ss-divi-mt-trigger';
						trigger.className = 'et-fb-button et-fb-button--primary';
						trigger.innerHTML = '<span class="dashicons dashicons-tag" style="margin-right: 5px; vertical-align: middle;"></span> Merge Tags';
						trigger.style.marginLeft = '10px';
						trigger.style.cursor = 'pointer';
						trigger.style.display = 'flex';
						trigger.style.alignItems = 'center';
						trigger.onclick = openSlipstreamMergeTagsModal;
						
						// Insert before the save button or at the end
						var saveButton = diviToolbar.querySelector('.et-fb-button--save');
						if (saveButton) {
							saveButton.parentNode.insertBefore(trigger, saveButton);
						} else {
							diviToolbar.appendChild(trigger);
						}
						// Don't clear interval yet, Divi might re-render
					}
				}, 2000);
			}

			if (window.location.search.indexOf('et_fb=1') > -1) {
				initDiviIntegration();
			}

			var initModalLogic = function() {
				var search = document.getElementById('slipstream-mt-search');
				var categories = document.querySelectorAll('.category-item');
				var groups = document.querySelectorAll('.mt-group-section');
				var items = document.querySelectorAll('.mt-card');
				var noResults = document.getElementById('mt-no-results');

				// Search functionality
				if (search) {
					search.addEventListener('keyup', function() {
						var filter = search.value.toLowerCase();
						var hasResults = false;

						items.forEach(function(item) {
							var tag = item.getAttribute('data-tag').toLowerCase();
							if (tag.indexOf(filter) > -1) {
								item.style.display = "";
								hasResults = true;
							} else {
								item.style.display = "none";
							}
						});

						groups.forEach(function(group) {
							var visibleItems = group.querySelectorAll('.mt-card:not([style*="display: none"])');
							if (visibleItems.length === 0) {
								group.style.display = "none";
							} else {
								group.style.display = "";
							}
						});

						noResults.style.display = hasResults ? "none" : "block";
						
						// If searching, deactivate category selection visual but show all relevant
						if (filter !== '') {
							categories.forEach(c => c.classList.remove('active'));
							// Hide help during search unless search matches "help"
							var helpSection = document.getElementById('mt-help-section');
							if (helpSection) helpSection.style.display = 'none';
						} else {
							var allCat = document.querySelector('.category-item[data-group="all"]');
							if (allCat) allCat.classList.add('active');
						}
					});
				}

				// Category filtering
				categories.forEach(function(cat) {
					cat.addEventListener('click', function() {
						var group = cat.getAttribute('data-group');
						
						categories.forEach(c => c.classList.remove('active'));
						cat.classList.add('active');
						
						if (search) search.value = ''; // Clear search when clicking category
						if (noResults) noResults.style.display = "none";

						var helpSection = document.getElementById('mt-help-section');
						if (helpSection) {
							helpSection.style.display = (group === 'help') ? '' : 'none';
						}

						groups.forEach(function(g) {
							if (g.id === 'mt-help-section') return;
							if (group === 'all' || g.getAttribute('data-group') === group) {
								g.style.display = "";
								g.querySelectorAll('.mt-card').forEach(i => i.style.display = "");
							} else {
								g.style.display = "none";
							}
						});
					});
				});

				// Close modal on escape key
				document.addEventListener('keydown', function(e) {
					if (e.key === "Escape") {
						closeSlipstreamMergeTagsModal();
					}
				});

				// Close modal when clicking outside container
				var modalOverlay = document.getElementById('slipstream-mt-modal');
				if (modalOverlay) {
					modalOverlay.addEventListener('click', function(e) {
						if (e.target === modalOverlay) {
							closeSlipstreamMergeTagsModal();
						}
					});
				}
			};

			if (document.readyState === 'loading') {
				document.addEventListener('DOMContentLoaded', initModalLogic);
			} else {
				initModalLogic();
			}
		})();
		</script>
		<style>
		.slipstream-modal-overlay {
			position: fixed;
			top: 0;
			left: 0;
			width: 100%;
			height: 100%;
			background: rgba(0, 0, 0, 0.7);
			display: flex;
			align-items: center;
			justify-content: center;
			z-index: 999999;
			backdrop-filter: blur(4px);
		}

		.slipstream-modal-container {
			background: #fff;
			width: 90%;
			max-width: 900px;
			height: 80%;
			max-height: 700px;
			border-radius: 20px;
			overflow: hidden;
			display: flex;
			flex-direction: column;
			box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
			animation: slipstreamModalFadeIn 0.3s ease-out;
		}

		@keyframes slipstreamModalFadeIn {
			from { opacity: 0; transform: scale(0.95) translateY(10px); }
			to { opacity: 1; transform: scale(1) translateY(0); }
		}

		.slipstream-modal-header {
			padding: 20px 30px;
			background: #fff;
			border-bottom: 1px solid #f1f1f1;
			display: flex;
			align-items: center;
			justify-content: space-between;
		}

		.header-left {
			display: flex;
			align-items: center;
			gap: 12px;
		}

		.header-left .dashicons {
			color: #4f46e5;
			font-size: 24px;
			width: 24px;
			height: 24px;
		}

		.header-left h2 {
			margin: 0;
			font-size: 20px;
			font-weight: 700;
			color: #111827;
		}

		.header-right {
			display: flex;
			align-items: center;
			gap: 20px;
		}

		.modal-search {
			position: relative;
			width: 300px;
		}

		.modal-search .dashicons {
			position: absolute;
			left: 12px;
			top: 50%;
			transform: translateY(-50%);
			color: #9ca3af;
			pointer-events: none;
		}

		.modal-search input {
			width: 100%;
			padding: 10px 15px 10px 40px !important;
			border: 1px solid #e5e7eb !important;
			border-radius: 12px !important;
			font-size: 14px !important;
			background: #f9fafb !important;
			transition: all 0.2s;
		}

		.modal-search input:focus {
			background: #fff !important;
			border-color: #4f46e5 !important;
			box-shadow: 0 0 0 4px rgba(79, 70, 229, 0.1) !important;
			outline: none;
		}

		.modal-close {
			background: none;
			border: none;
			font-size: 30px;
			color: #9ca3af;
			cursor: pointer;
			padding: 0;
			line-height: 1;
			transition: color 0.2s;
		}

		.modal-close:hover {
			color: #111827;
		}

		.slipstream-modal-body {
			flex: 1;
			display: flex;
			overflow: hidden;
		}

		.modal-sidebar {
			width: 220px;
			background: #f9fafb;
			border-right: 1px solid #f1f1f1;
			padding: 20px 10px;
			overflow-y: auto;
		}

		.modal-categories {
			list-style: none;
			padding: 0;
			margin: 0;
		}

		.category-item {
			padding: 10px 15px;
			border-radius: 10px;
			font-size: 14px;
			font-weight: 500;
			color: #4b5563;
			cursor: pointer;
			display: flex;
			align-items: center;
			justify-content: space-between;
			transition: all 0.2s;
			margin-bottom: 4px;
		}

		.category-item:hover {
			background: #f3f4f6;
			color: #111827;
		}

		.category-item.active {
			background: #4f46e5;
			color: #fff;
		}

		.category-item .count {
			font-size: 11px;
			background: #e5e7eb;
			color: #6b7280;
			padding: 2px 8px;
			border-radius: 20px;
		}

		.category-item.active .count {
			background: rgba(255, 255, 255, 0.2);
			color: #fff;
		}

		.modal-content-area {
			flex: 1;
			padding: 30px;
			overflow-y: auto;
			background: #fff;
		}

		.mt-group-section {
			margin-bottom: 40px;
		}

		.mt-group-section:last-child {
			margin-bottom: 0;
		}

		.mt-group-title {
			font-size: 12px;
			text-transform: uppercase;
			letter-spacing: 0.1em;
			color: #9ca3af;
			font-weight: 700;
			margin: 0 0 20px 0;
			display: flex;
			align-items: center;
			gap: 10px;
		}

		.mt-group-title::after {
			content: '';
			flex: 1;
			height: 1px;
			background: #f1f1f1;
		}

		.mt-items-grid {
			display: grid;
			grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
			gap: 16px;
		}

		.mt-card {
			position: relative;
			background: #fff;
			border: 1px solid #e5e7eb;
			border-radius: 16px;
			padding: 16px;
			display: flex;
			align-items: flex-start;
			justify-content: space-between;
			cursor: pointer;
			transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
		}

		.mt-card:hover {
			border-color: #4f46e5;
			box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.05);
			transform: translateY(-2px);
		}

		.mt-card-content {
			flex: 1;
		}

		.mt-tag-code {
			font-family: 'SFMono-Regular', Consolas, 'Liberation Mono', Menlo, monospace;
			font-size: 14px;
			font-weight: 600;
			color: #111827;
			margin-bottom: 4px;
		}

		.mt-tag-description {
			font-size: 12px;
			color: #6b7280;
			line-height: 1.4;
		}

		.mt-card-action {
			color: #9ca3af;
			padding-left: 12px;
			transition: color 0.2s;
		}

		.mt-card:hover .mt-card-action {
			color: #4f46e5;
		}

		.mt-copied-overlay {
			position: absolute;
			top: 0;
			left: 0;
			width: 100%;
			height: 100%;
			background: #4f46e5;
			border-radius: 16px;
			display: flex;
			flex-direction: column;
			align-items: center;
			justify-content: center;
			color: #fff;
			opacity: 0;
			pointer-events: none;
			transition: opacity 0.2s;
			z-index: 10;
		}

		.mt-card.is-copied .mt-copied-overlay {
			opacity: 1;
		}

		.mt-copied-overlay .dashicons {
			font-size: 32px;
			width: 32px;
			height: 32px;
			margin-bottom: 4px;
		}

		.mt-copied-overlay span {
			font-size: 14px;
			font-weight: 600;
		}

		.no-results-content {
			text-align: center;
			padding: 60px 0;
			color: #9ca3af;
		}

		.no-results-content .dashicons {
			font-size: 48px;
			width: 48px;
			height: 48px;
			margin-bottom: 16px;
		}

		.no-results-content p {
			font-size: 16px;
			margin: 0;
		}
		</style>
		<?php
	}
}
