<?php
/**
 * Package Loader Class - Scans and Loads Packages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Package_Loader {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'plugins_loaded', array( $this, 'load_packages' ), 20 );
		add_action( 'slipstream_registry_flushed', array( $this, 'load_packages' ) );
		add_action( 'init', array( $this, 'save_registry' ), 999 );
	}

	/**
	 * Save the registry after all init hooks have run
	 */
	public function save_registry() {
		$registry = Slipstream_Registry::get_instance();
		$registry->save();
	}

	/**
	 * Scan the packages directory and include active packages
	 */
	public function load_packages() {
		if ( ! file_exists( SLIPSTREAM_PACKAGES_DIR ) ) {
			return;
		}

		$registry = Slipstream_Registry::get_instance();
		
		// If init hasn't fired yet, clear memory to allow fresh registration
		if ( ! did_action( 'init' ) ) {
			$registry->clear_type_in_memory( 'widgets' );
			$registry->clear_type_in_memory( 'panels' );
			$registry->clear_type_in_memory( 'merge_tags' );
			$registry->clear_type_in_memory( 'admin_configs' );
			$registry->clear_type_in_memory( 'welcome_tabs' );
		}
		
		$packages_on_disk = scandir( SLIPSTREAM_PACKAGES_DIR );
		$stored_packages = $registry->get_data( 'packages' );
		$needs_save = false;

		foreach ( $packages_on_disk as $package_slug ) {
			if ( $package_slug === '.' || $package_slug === '..' || $package_slug === '.htaccess' ) {
				continue;
			}

			$package_path = SLIPSTREAM_PACKAGES_DIR . '/' . $package_slug;

			if ( is_dir( $package_path ) ) {
				// Auto-discovery: If package is on disk but not in registry, add it
				if ( ! isset( $stored_packages[$package_slug] ) ) {
					$manifest_file = $package_path . '/manifest.json';
					$manifest = array( 'name' => $package_slug, 'version' => '1.0.0' );
					if ( file_exists( $manifest_file ) ) {
						$manifest = json_decode( file_get_contents( $manifest_file ), true ) ?: $manifest;
					}

					// Read changelog
					$changelog = '';
					$changelog_file = $package_path . '/changelog.md';
					if ( file_exists( $changelog_file ) ) {
						$changelog = file_get_contents( $changelog_file );
					} elseif ( isset( $manifest['changelog'] ) ) {
						$changelog = $manifest['changelog'];
					}

					// Check Core Compatibility
					$requires_core = $manifest['requires_core'] ?? '1.0.0';
					$is_compatible = version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' );

					$package_data = array(
						'name'          => $manifest['name'],
						'version'       => $manifest['version'],
						'active'        => true, // Default to active on discovery
						'path'          => $package_path,
						'changelog'     => $changelog,
						'requires_core' => $requires_core,
						'compatible'    => $is_compatible,
					);
					$registry->register( 'packages', $package_slug, $package_data );
					$stored_packages[$package_slug] = $package_data;
					$needs_save = true;
				} else {
					// Check if version in registry matches version on disk for metadata updates
					$manifest_file = $package_path . '/manifest.json';
					if ( file_exists( $manifest_file ) ) {
						$manifest = json_decode( file_get_contents( $manifest_file ), true );
						if ( $manifest && isset( $manifest['version'] ) ) {
							$disk_version = $manifest['version'];
							$reg_version  = $stored_packages[$package_slug]['version'] ?? '0';

							if ( version_compare( $disk_version, $reg_version, '>' ) ) {
								// Metadata update (e.g., someone FTP'd a new version)
								$requires_core = $manifest['requires_core'] ?? '1.0.0';
								$is_compatible = version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' );

								$stored_packages[$package_slug]['version']       = $disk_version;
								$stored_packages[$package_slug]['name']          = $manifest['name'] ?? $stored_packages[$package_slug]['name'];
								$stored_packages[$package_slug]['requires_core'] = $requires_core;
								$stored_packages[$package_slug]['compatible']    = $is_compatible;

								// Update changelog too
								$changelog = '';
								$changelog_file = $package_path . '/changelog.md';
								if ( file_exists( $changelog_file ) ) {
									$changelog = file_get_contents( $changelog_file );
								} elseif ( isset( $manifest['changelog'] ) ) {
									$changelog = $manifest['changelog'];
								}
								$stored_packages[$package_slug]['changelog'] = $changelog;

								$registry->register( 'packages', $package_slug, $stored_packages[$package_slug] );
								$needs_save = true;
							}
						}
					}
				}

				// Boot if active and compatible
				if ( isset( $stored_packages[$package_slug] ) && $stored_packages[$package_slug]['active'] ) {
					$is_compatible = $stored_packages[$package_slug]['compatible'] ?? true;
					if ( ! $is_compatible ) {
						// Double check compatibility in case core was updated since last scan
						$requires_core = $stored_packages[$package_slug]['requires_core'] ?? '1.0.0';
						if ( version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' ) ) {
							$is_compatible = true;
							$stored_packages[$package_slug]['compatible'] = true;
							$registry->register( 'packages', $package_slug, $stored_packages[$package_slug] );
							$needs_save = true;
						}
					}

					if ( $is_compatible ) {
						$this->boot_package( $package_path, $package_slug );
					}
					
					// If init has already fired, we might need to manually trigger registration
					// for packages that only register on 'init'
					if ( did_action( 'init' ) ) {
						// We can't easily force an 'init' priority 20 hook to fire again 
						// if it was added AFTER init already fired.
						// So we might want to encourage packages to register immediately 
						// or we can trigger a custom hook.
					}
				}
			}
		}

		if ( $needs_save ) {
			$registry->save();
		}

		// Save the registry after all packages have potentially registered their panels/widgets
		// but only if we actually booted something or flushed
		if ( ! empty( $stored_packages ) ) {
			// $registry->save(); // Removed redundant save, will be handled by init hook
		}
	}

	/**
	 * Boot an individual package
	 */
	private function boot_package( $path, $slug ) {
		$main_file = $path . '/index.php';
		
		// List of standard registration files to auto-load
		$auto_load_files = array(
			'functions.php',
			'post-types.php',
			'acf-fields.php',
			'settings.php',
			'panels.php',
			'merge-tags.php',
			'widgets.php',
		);

		foreach ( $auto_load_files as $file ) {
			$file_path = $path . '/' . $file;
			if ( file_exists( $file_path ) ) {
				try {
					include_once $file_path;
				} catch ( Throwable $t ) {
					$this->handle_boot_error( $slug, $file, $t );
				}
			}
		}

		// Auto-load Divi modules if they exist
		$divi_modules_dir = $path . '/divi/modules';
		if ( is_dir( $divi_modules_dir ) ) {
			add_action( 'et_builder_ready', function() use ( $divi_modules_dir, $slug ) {
				if ( ! class_exists( 'ET_Builder_Module' ) ) {
					return;
				}

				$modules = scandir( $divi_modules_dir );
				foreach ( $modules as $module_folder ) {
					if ( $module_folder === '.' || $module_folder === '..' ) {
						continue;
					}

					$module_path = $divi_modules_dir . '/' . $module_folder;
					if ( is_dir( $module_path ) ) {
						$module_file      = $module_path . '/module.php';
						$module_item_file = $module_path . '/module_item.php';

						// Load assets if they exist
						$assets_dir = $module_path . '/assets';
						if ( is_dir( $assets_dir ) ) {
							$this->enqueue_module_assets( $assets_dir, $module_folder );
						}

						if ( file_exists( $module_item_file ) ) {
							include_once $module_item_file;
						}

						if ( file_exists( $module_file ) ) {
							include_once $module_file;
						}
					}
				}
			} );
		}

		if ( file_exists( $main_file ) ) {
			try {
				include_once $main_file;
			} catch ( Throwable $t ) {
				$this->handle_boot_error( $slug, 'index.php', $t );
			}
		}
	}

	/**
	 * Handle errors during package boot (Safe Mode)
	 */
	private function handle_boot_error( $slug, $file, $error ) {
		$error_message = $error->getMessage();
		$trace = $error->getTraceAsString();
		
		Slipstream_Activity_Log::get_instance()->log( 
			'Safe Mode', 
			"Disabled package ($slug) due to error in $file: $error_message. \nTrace: $trace", 
			'error' 
		);

		// Deactivate the package in the registry to prevent future crashes
		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		if ( isset( $packages[$slug] ) ) {
			$packages[$slug]['active'] = false;
			$registry->register( 'packages', $slug, $packages[$slug] );
			$registry->save();
		}

		if ( is_admin() ) {
			set_transient( 'slipstream_error', "Safe Mode: Package '$slug' was disabled because it caused a fatal error: $error_message", 45 );
		}
	}

	/**
	 * Enqueue assets for a Divi module
	 */
	private function enqueue_module_assets( $assets_dir, $module_slug ) {
		$files = scandir( $assets_dir );
		foreach ( $files as $file ) {
			if ( $file === '.' || $file === '..' ) {
				continue;
			}

			$file_path = $assets_dir . '/' . $file;
			$file_url = slipstream_package_url( $file_path );
			
			$ext = pathinfo( $file, PATHINFO_EXTENSION );
			if ( $ext === 'css' ) {
				wp_enqueue_style( 'ss-module-' . $module_slug . '-' . $file, $file_url, array(), SLIPSTREAM_VERSION );
			} elseif ( $ext === 'js' ) {
				wp_enqueue_script( 'ss-module-' . $module_slug . '-' . $file, $file_url, array( 'jquery' ), SLIPSTREAM_VERSION, true );
			}
		}
	}
}
