<?php
/**
 * Package Manager Class - Handles ZIP uploads and security
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Package_Manager {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_init', array( $this, 'process_actions' ) );
	}

	/**
	 * Process package actions (upload, delete, etc)
	 */
	public function process_actions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			return;
		}

		// Handle Upload
		if ( isset( $_POST['upload_package'] ) && isset( $_FILES['slipstream_package_zip'] ) ) {
			check_admin_referer( 'slipstream_upload_package' );
			$result = $this->handle_upload( $_FILES['slipstream_package_zip'] );
			if ( is_wp_error( $result ) ) {
				set_transient( 'slipstream_error', $result->get_error_message(), 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Upload Package', 'Failed to upload package from ZIP: ' . $result->get_error_message(), 'error' );
			} else {
				set_transient( 'slipstream_success', 'Package uploaded and activated successfully.', 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Upload Package', 'Successfully uploaded and activated package from ZIP.' );
			}
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}

		// Handle Install/Update Remote
		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_install_remote' ) {
			check_admin_referer( 'slipstream_install_remote' );
			$slug = isset( $_GET['slug'] ) ? sanitize_text_field( $_GET['slug'] ) : '';
			
			$result = $this->handle_remote_install( $slug );
			
			if ( is_wp_error( $result ) ) {
				set_transient( 'slipstream_error', $result->get_error_message(), 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Remote Install', "Failed to install/update remote package ($slug): " . $result->get_error_message(), 'error' );
			} else {
				set_transient( 'slipstream_success', 'Remote package installed/updated.', 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Remote Install', "Successfully installed/updated remote package ($slug)." );
			}
			
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages#library' ) );
			exit;
		}

		// Handle Install/Update Example
		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_install_example' ) {
			check_admin_referer( 'slipstream_install_example' );
			$zip_name = isset( $_GET['zip'] ) ? sanitize_text_field( $_GET['zip'] ) : '';
			$zip_path = SLIPSTREAM_PATH . 'examples/' . $zip_name;

			if ( file_exists( $zip_path ) ) {
				$result = $this->handle_upload( array(
					'tmp_name' => $zip_path,
					'name'     => $zip_name
				) );
			if ( is_wp_error( $result ) ) {
				set_transient( 'slipstream_error', $result->get_error_message(), 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Example Install', "Failed to install/update example package ($zip_name): " . $result->get_error_message(), 'error' );
			} else {
				set_transient( 'slipstream_success', 'Example package installed/updated.', 45 );
				Slipstream_Activity_Log::get_instance()->log( 'Example Install', "Successfully installed/updated example package ($zip_name)." );
			}
			} else {
				set_transient( 'slipstream_error', 'Example ZIP file not found.', 45 );
			}
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}

		// Handle Toggle Status
		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_toggle' && isset( $_GET['package'] ) ) {
			check_admin_referer( 'slipstream_toggle_package' );
			$slug = sanitize_text_field( $_GET['package'] );
			$this->toggle_package( $slug );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}

		// Handle Delete
		if ( isset( $_GET['action'] ) && $_GET['action'] === 'slipstream_delete' && isset( $_GET['package'] ) ) {
			check_admin_referer( 'slipstream_delete_package' );
			$slug = sanitize_text_field( $_GET['package'] );
			$this->delete_package( $slug );
			Slipstream_Activity_Log::get_instance()->log( 'Delete Package', "Successfully deleted package ($slug)." );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}
		
		// Handle Note Save
		if ( isset( $_POST['save_package_note'] ) && isset( $_POST['package_slug'] ) ) {
			check_admin_referer( 'slipstream_save_note' );
			$slug = sanitize_text_field( $_POST['package_slug'] );
			$note = sanitize_textarea_field( $_POST['package_note'] );
			$this->save_note( $slug, $note );
			set_transient( 'slipstream_success', 'Note saved successfully.', 45 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-packages' ) );
			exit;
		}
	}

	/**
	 * Toggle package active status
	 */
	public function toggle_package( $slug ) {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			return;
		}
		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		
		if ( isset( $packages[$slug] ) ) {
			$new_status = ! $packages[$slug]['active'];
			$packages[$slug]['active'] = $new_status;
			$registry->register( 'packages', $slug, $packages[$slug] );
			$registry->save();

			$status_text = $new_status ? 'activated' : 'deactivated';
			Slipstream_Activity_Log::get_instance()->log( 
				'Package Status', 
				"Package ($slug) was $status_text manually." 
			);
		}
	}

	/**
	 * Delete package and its files
	 */
	public function delete_package( $slug ) {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			return;
		}
		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		
		if ( isset( $packages[$slug] ) ) {
			$path = $packages[$slug]['path'];
			
			// Remove files
			$this->recursive_rmdir( $path );
			
			// Remove from registry
			$packages = $registry->get_data( 'packages' );
			unset( $packages[$slug] );
			
			// We need a way to clear the whole type or re-save
			$registry->clear_type( 'packages' );
			foreach ( $packages as $s => $data ) {
				$registry->register( 'packages', $s, $data );
			}
			$registry->save();
		}
	}

	/**
	 * Save a note for a package
	 */
	public function save_note( $slug, $note ) {
		if ( get_option( 'slipstream_locked_down' ) === 'yes' ) {
			return;
		}
		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		
		if ( isset( $packages[$slug] ) ) {
			$packages[$slug]['note'] = $note;
			$registry->register( 'packages', $slug, $packages[$slug] );
			$registry->save();
		}
	}

	/**
	 * Recursive rmdir
	 */
	private function recursive_rmdir( $dir ) {
		if ( is_dir( $dir ) ) {
			$objects = scandir( $dir );
			foreach ( $objects as $object ) {
				if ( $object != "." && $object != ".." ) {
					if ( is_dir( $dir . DIRECTORY_SEPARATOR . $object ) && ! is_link( $dir . "/" . $object ) )
						$this->recursive_rmdir( $dir . DIRECTORY_SEPARATOR . $object );
					else
						unlink( $dir . DIRECTORY_SEPARATOR . $object );
				}
			}
			rmdir( $dir );
		}
	}

	/**
	 * Handle remote package installation
	 */
	public function handle_remote_install( $slug ) {
		$remote_url = get_option( 'slipstream_remote_url' );
		$remote_key = get_option( 'slipstream_remote_key' );

		if ( ! $remote_url || ! $remote_key ) {
			return new WP_Error( 'remote_not_configured', 'Remote server not configured.' );
		}

		$download_url = add_query_arg( array(
			'action' => 'download',
			'slug'   => $slug,
			'key'    => $remote_key
		), $remote_url );

		// Download to temp file
		$temp_file = download_url( $download_url );

		if ( is_wp_error( $temp_file ) ) {
			return $temp_file;
		}

		$result = $this->handle_upload( array(
			'tmp_name' => $temp_file,
			'name'     => $slug . '.zip'
		) );

		// Cleanup
		@unlink( $temp_file );

		return $result;
	}

	/**
	 * Handle package upload
	 */
	public function handle_upload( $file ) {
		$is_local = !isset($file['error']);
		if ( ! $is_local && empty( $file['tmp_name'] ) ) {
			return new WP_Error( 'no_file', 'No file uploaded.' );
		}

		$tmp_path = $file['tmp_name'];
		$original_name = $file['name'];

		// Basic Security Scan
		$scan_result = $this->scan_package( $tmp_path );
		if ( is_wp_error( $scan_result ) ) {
			if ( ! $is_local ) @unlink( $tmp_path );
			return $scan_result;
		}

		// Extract ZIP
		$zip = new ZipArchive();
		if ( $zip->open( $tmp_path ) === TRUE ) {
			// Derive slug from ZIP filename
			$slug = sanitize_title( pathinfo( $original_name, PATHINFO_FILENAME ) );
			$extract_path = SLIPSTREAM_PACKAGES_DIR . '/' . $slug;

			// Ensure directory exists or clean it
			if ( ! file_exists( $extract_path ) ) {
				wp_mkdir_p( $extract_path );
			}

			$zip->extractTo( $extract_path );
			$zip->close();

			// Read manifest for metadata
			$manifest_file = $extract_path . '/manifest.json';
			$manifest = array( 'name' => $slug, 'version' => '1.0.0' );
			if ( file_exists( $manifest_file ) ) {
				$manifest = json_decode( file_get_contents( $manifest_file ), true ) ?: $manifest;
			}

			// Read changelog
			$changelog = '';
			$changelog_file = $extract_path . '/changelog.md';
			if ( file_exists( $changelog_file ) ) {
				$changelog = file_get_contents( $changelog_file );
			} elseif ( isset( $manifest['changelog'] ) ) {
				$changelog = $manifest['changelog'];
			}

			// Generate checksum
			$checksum = $this->generate_checksum( $extract_path );

			// Check Core Compatibility
			$requires_core = $manifest['requires_core'] ?? '1.0.0';
			$is_compatible = version_compare( SLIPSTREAM_VERSION, $requires_core, '>=' );

			// Update registry
			$registry = Slipstream_Registry::get_instance();
			$registry->register( 'packages', $slug, array(
				'name'          => $manifest['name'],
				'version'       => $manifest['version'],
				'active'        => true,
				'path'          => $extract_path,
				'checksum'      => $checksum,
				'changelog'     => $changelog,
				'requires_core' => $requires_core,
				'compatible'    => $is_compatible,
			) );
			$registry->save();

			Slipstream_Activity_Log::get_instance()->log( 
				'Package Upload', 
				"Package ($slug) v{$manifest['version']} uploaded and activated successfully." 
			);

			return true;
		} else {
			return new WP_Error( 'zip_error', 'Could not open ZIP file.' );
		}
	}

	/**
	 * Generate MD5 checksum of a directory
	 */
	public function generate_checksum( $path ) {
		$files = array();
		$dir = new RecursiveDirectoryIterator( $path );
		$it = new RecursiveIteratorIterator( $dir );
		foreach ( $it as $file ) {
			if ( $file->isFile() && basename( $file ) !== '.DS_Store' ) {
				$files[] = $file->getPathname();
			}
		}
		sort( $files );
		
		$checksums = '';
		foreach ( $files as $file ) {
			$checksums .= md5_file( $file );
		}
		return md5( $checksums );
	}

	/**
	 * Verify if a package has been modified
	 */
	public function is_modified( $slug ) {
		$registry = Slipstream_Registry::get_instance();
		$packages = $registry->get_data( 'packages' );
		
		if ( ! isset( $packages[$slug] ) || ! isset( $packages[$slug]['checksum'] ) ) {
			return false;
		}
		
		$current_checksum = $this->generate_checksum( $packages[$slug]['path'] );
		return $current_checksum !== $packages[$slug]['checksum'];
	}

	/**
	 * Scan package for malicious code
	 */
	private function scan_package( $zip_path ) {
		// This is a simplified scan as requested
		// In a real implementation, we would open the ZIP and scan each PHP file
		// For now, let's assume we can scan the ZIP content strings if possible or after extraction
		
		// Placeholder for scan logic
		$suspicious = array( 'eval(', 'exec(', 'base64_decode(', 'system(' );
		
		// Open ZIP and iterate files
		$zip = new ZipArchive();
		if ( $zip->open( $zip_path ) === TRUE ) {
			for ( $i = 0; $i < $zip->numFiles; $i++ ) {
				$filename = $zip->getNameIndex( $i );
				
				// Only scan PHP files
				if ( pathinfo( $filename, PATHINFO_EXTENSION ) === 'php' ) {
					$content = $zip->getFromIndex( $i );
					foreach ( $suspicious as $string ) {
						if ( stripos( $content, $string ) !== false ) {
							$zip->close();
							return new WP_Error( 'malicious_code', "Malicious code detected in $filename: $string" );
						}
					}
				}
			}
			$zip->close();
		} else {
			return new WP_Error( 'zip_error', 'Could not open ZIP file.' );
		}

		return true;
	}
}
