<?php
/**
 * User Profile UI Class - Custom tabbed interface for User Profile & Security
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Profile_UI {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_init', array( $this, 'process_profile_actions' ) );
	}

	/**
	 * Register Admin Menu for the custom profile page
	 */
	public function register_menu() {
		add_submenu_page(
			null, // Hidden from standard menu
			__( 'My Profile', 'slipstream' ),
			__( 'My Profile', 'slipstream' ),
			'read',
			'slipstream-profile',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Process profile actions (save profile, change password)
	 */
	public function process_profile_actions() {
		if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'slipstream-profile' ) {
			return;
		}

		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return;
		}

		// Save Profile Info
		if ( isset( $_POST['slipstream_save_profile'] ) ) {
			check_admin_referer( 'slipstream_save_profile' );

			$email = sanitize_email( $_POST['email'] );
			$first_name = sanitize_text_field( $_POST['first_name'] );
			$last_name = sanitize_text_field( $_POST['last_name'] );
			$display_name = sanitize_text_field( $_POST['display_name'] );

			$errors = new WP_Error();

			if ( ! is_email( $email ) ) {
				$errors->add( 'email_invalid', __( 'Invalid email address.', 'slipstream' ) );
			} elseif ( email_exists( $email ) && email_exists( $email ) != $user_id ) {
				$errors->add( 'email_exists', __( 'This email is already registered.', 'slipstream' ) );
			}

			if ( empty( $errors->get_error_codes() ) ) {
				wp_update_user( array(
					'ID'           => $user_id,
					'user_email'   => $email,
					'first_name'   => $first_name,
					'last_name'    => $last_name,
					'display_name' => $display_name,
				) );

				Slipstream_Activity_Log::get_instance()->log( 'Profile Updated', 'User updated their profile information.' );
				set_transient( 'slipstream_profile_success', __( 'Profile updated successfully.', 'slipstream' ), 30 );
				wp_safe_redirect( admin_url( 'admin.php?page=slipstream-profile#general' ) );
				exit;
			} else {
				set_transient( 'slipstream_profile_error', $errors->get_error_message(), 30 );
			}
		}

		// Change Password
		if ( isset( $_POST['slipstream_change_password'] ) ) {
			check_admin_referer( 'slipstream_change_password' );

			$pass1 = $_POST['pass1'];
			$pass2 = $_POST['pass2'];

			if ( ! empty( $pass1 ) ) {
				if ( $pass1 === $pass2 ) {
					wp_set_password( $pass1, $user_id );
					// wp_set_password logs the user out, we might need to re-log them in or redirect to login
					Slipstream_Activity_Log::get_instance()->log( 'Password Changed', 'User changed their password.' );
					
					// Re-auth
					$user = get_user_by( 'id', $user_id );
					wp_set_current_user( $user_id, $user->user_login );
					wp_set_auth_cookie( $user_id );
					do_action( 'wp_login', $user->user_login, $user );

					set_transient( 'slipstream_profile_success', __( 'Password changed successfully.', 'slipstream' ), 30 );
					wp_safe_redirect( admin_url( 'admin.php?page=slipstream-profile#security' ) );
					exit;
				} else {
					set_transient( 'slipstream_profile_error', __( 'Passwords do not match.', 'slipstream' ), 30 );
				}
			}
		}

		// Logout other sessions
		if ( isset( $_POST['slipstream_logout_others'] ) ) {
			check_admin_referer( 'slipstream_logout_others' );
			$verifier = wp_get_session_token();
			wp_get_session_manager()->destroy_others( $verifier );
			
			Slipstream_Activity_Log::get_instance()->log( 'Sessions Terminated', 'User logged out of other sessions.' );
			set_transient( 'slipstream_profile_success', __( 'Other sessions terminated.', 'slipstream' ), 30 );
			wp_safe_redirect( admin_url( 'admin.php?page=slipstream-profile#security' ) );
			exit;
		}
	}

	/**
	 * Render the Profile Page
	 */
	public function render_page() {
		$user = wp_get_current_user();
		$success_msg = get_transient( 'slipstream_profile_success' );
		$error_msg = get_transient( 'slipstream_profile_error' );
		delete_transient( 'slipstream_profile_success' );
		delete_transient( 'slipstream_profile_error' );

		$sessions = wp_get_all_sessions();
		$current_session = wp_get_session_token();
		?>
		<div class="wrap">
			<div class="flex items-center justify-between mb-8">
				<div class="flex gap-2 bg-slate-100 p-1.5 rounded-2xl w-fit">
					<button type="button" onclick="window.ssSwitchTab('general')" id="tab-btn-general" class="px-6 py-2 rounded-xl text-sm font-bold transition-all bg-white text-slate-900 shadow-sm">General</button>
					<button type="button" onclick="window.ssSwitchTab('security')" id="tab-btn-security" class="px-6 py-2 rounded-xl text-sm font-bold transition-all text-slate-500 hover:text-slate-700">Security</button>
				</div>
				
				<div id="save-btn-container">
					<button type="submit" form="profile-form" id="main-save-btn" class="bg-indigo-600 text-white px-6 py-2 rounded-2xl text-sm font-bold shadow-lg shadow-indigo-100 transition-all hover:bg-indigo-700 active:scale-95">Save Profile</button>
				</div>
			</div>

			<?php if ( $success_msg ) : ?>
				<div class="bg-emerald-50 border border-emerald-100 text-emerald-700 px-6 py-4 rounded-2xl mb-6 font-medium flex items-center gap-3">
					<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-emerald-500"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
					<?php echo esc_html( $success_msg ); ?>
				</div>
			<?php endif; ?>

			<?php if ( $error_msg ) : ?>
				<div class="bg-rose-50 border border-rose-100 text-rose-700 px-6 py-4 rounded-2xl mb-6 font-medium flex items-center gap-3">
					<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-rose-500"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="12"/><line x1="12" y1="16" x2="12.01" y2="16"/></svg>
					<?php echo esc_html( $error_msg ); ?>
				</div>
			<?php endif; ?>

			<div class="w-full">
				<!-- General Tab -->
				<div id="tab-content-general" class="block">
					<form method="post" id="profile-form" class="space-y-6">
						<?php wp_nonce_field( 'slipstream_save_profile' ); ?>
						<input type="hidden" name="slipstream_save_profile" value="1">
						
						<div class="bg-white rounded-3xl shadow-sm border border-slate-100 p-8 space-y-8">
							<div class="grid grid-cols-1 md:grid-cols-2 gap-8">
								<div>
									<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">First Name</label>
									<input type="text" name="first_name" value="<?php echo esc_attr( $user->first_name ); ?>" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
								</div>
								<div>
									<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">Last Name</label>
									<input type="text" name="last_name" value="<?php echo esc_attr( $user->last_name ); ?>" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
								</div>
							</div>

							<div>
								<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">Display Name</label>
								<input type="text" name="display_name" value="<?php echo esc_attr( $user->display_name ); ?>" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
								<p class="mt-2 text-xs text-slate-400">Public name shown in the interface.</p>
							</div>

							<div>
								<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">Email Address</label>
								<input type="email" name="email" value="<?php echo esc_attr( $user->user_email ); ?>" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
							</div>
						</div>
					</form>
				</div>

				<!-- Security Tab -->
				<div id="tab-content-security" class="hidden space-y-8">
					<form method="post" id="password-form" class="bg-white rounded-3xl shadow-sm border border-slate-100 p-8 space-y-8">
						<?php wp_nonce_field( 'slipstream_change_password' ); ?>
						<input type="hidden" name="slipstream_change_password" value="1">
						
						<div>
							<h3 class="text-lg font-bold text-slate-900 mb-1">Change Password</h3>
							<p class="text-sm text-slate-500 mb-6">Ensure your account is using a long, random password to stay secure.</p>
							
							<div class="grid grid-cols-1 md:grid-cols-2 gap-8">
								<div>
									<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">New Password</label>
									<input type="password" name="pass1" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
								</div>
								<div>
									<label class="block text-xs font-bold text-slate-400 uppercase mb-2 ml-1">Confirm Password</label>
									<input type="password" name="pass2" class="w-full bg-slate-50 border-none rounded-2xl text-sm font-bold text-slate-700 focus:ring-indigo-500 p-3">
								</div>
							</div>
							
							<div class="mt-6">
								<button type="submit" class="bg-slate-900 text-white px-6 py-2 rounded-2xl text-sm font-bold transition-all hover:bg-slate-800 active:scale-95">Update Password</button>
							</div>
						</div>
					</form>

					<div class="bg-white rounded-3xl shadow-sm border border-slate-100 p-8">
						<h3 class="text-lg font-bold text-slate-900 mb-1">Browser Sessions</h3>
						<p class="text-sm text-slate-500 mb-6">Manage and log out your active sessions on other browsers and devices.</p>
						
						<div class="space-y-4">
							<div class="flex items-center gap-4 p-4 bg-slate-50 rounded-2xl">
								<div class="text-slate-400">
									<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-monitor"><rect width="20" height="14" x="2" y="3" rx="2"/><line x1="8" y1="21" x2="16" y2="21"/><line x1="12" y1="17" x2="12" y2="21"/></svg>
								</div>
								<div>
									<div class="text-sm font-bold text-slate-900">This device</div>
									<div class="text-xs text-slate-500">Currently active</div>
								</div>
							</div>

							<?php if ( count( $sessions ) > 1 ) : ?>
								<form method="post">
									<?php wp_nonce_field( 'slipstream_logout_others' ); ?>
									<input type="hidden" name="slipstream_logout_others" value="1">
									<button type="submit" class="text-sm font-bold text-indigo-600 hover:text-indigo-700">Log out other browser sessions</button>
								</form>
							<?php endif; ?>
						</div>
					</div>
				</div>
			</div>
		</div>

		<script>
			window.ssSwitchTab = function(tab) {
				const genBtn = document.getElementById('tab-btn-general');
				const secBtn = document.getElementById('tab-btn-security');
				const genCont = document.getElementById('tab-content-general');
				const secCont = document.getElementById('tab-content-security');
				const saveBtnCont = document.getElementById('save-btn-container');

				[genBtn, secBtn].forEach(btn => btn.classList.remove('bg-white', 'text-slate-900', 'shadow-sm'));
				[genBtn, secBtn].forEach(btn => btn.classList.add('text-slate-500'));
				[genCont, secCont].forEach(cont => cont.classList.add('hidden'));

				if (tab === 'general') {
					genBtn.classList.add('bg-white', 'text-slate-900', 'shadow-sm');
					genBtn.classList.remove('text-slate-500');
					genCont.classList.remove('hidden');
					saveBtnCont.classList.remove('hidden');
					history.replaceState(null, null, '#general');
				} else if (tab === 'security') {
					secBtn.classList.add('bg-white', 'text-slate-900', 'shadow-sm');
					secBtn.classList.remove('text-slate-500');
					secCont.classList.remove('hidden');
					saveBtnCont.classList.add('hidden');
					history.replaceState(null, null, '#security');
				}
			};

			// Handle initial hash
			if (window.location.hash === '#security') {
				window.ssSwitchTab('security');
			}
		</script>
		<?php
	}
}
