<?php
/**
 * Registry Singleton Class
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Registry {

	private static $instance;
	private $data = array();
	private $groups = array();
	private $merge_tag_groups = array();
	private $admin_config = array();
	private $owner_capabilities = array();
	private $option_name = 'slipstream_registry';
	private $content_fields = array();
	private $callbacks = array();

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->load_from_db();
		$this->register_base_groups();
	}

	/**
	 * Register base groups
	 */
	private function register_base_groups() {
		$this->register_group( 'user', array( 'name' => 'User', 'priority' => 10 ) );
		$this->register_group( 'settings', array( 'name' => 'Settings', 'priority' => 20 ) );
		$this->register_group( 'content', array( 'name' => 'Content', 'priority' => 30 ) );
		$this->register_group( 'pages', array( 'name' => 'Pages', 'priority' => 25 ) );

		// Merge Tag Groups
		$this->register_merge_tag_group( 'site', array( 'name' => 'Site Info', 'priority' => 10 ) );
		$this->register_merge_tag_group( 'acf', array( 'name' => 'ACF Fields', 'priority' => 20 ) );
		$this->register_merge_tag_group( 'custom', array( 'name' => 'Custom', 'priority' => 30 ) );
	}

	/**
	 * Register a merge tag group
	 */
	public function register_merge_tag_group( $slug, $args ) {
		$this->merge_tag_groups[$slug] = wp_parse_args( $args, array(
			'name'     => ucfirst( $slug ),
			'priority' => 10,
		) );
	}

	/**
	 * Get registered merge tag groups
	 */
	public function get_merge_tag_groups() {
		uasort( $this->merge_tag_groups, function( $a, $b ) {
			return $a['priority'] - $b['priority'];
		} );
		return $this->merge_tag_groups;
	}

	/**
	 * Load registry data from wp_options
	 */
	private function load_from_db() {
		$stored = get_option( $this->option_name );
		if ( $stored ) {
			$this->data = json_decode( $stored, true ) ?: array();
		}
	}

	/**
	 * Get registered dynamic menu sources
	 * @deprecated Use Slipstream_Menu_Handler instead
	 */
	public function get_dynamic_menu_sources() {
		return array();
	}

	/**
	 * Save registry data to wp_options
	 */
	public function save() {
		// Only save if we have actual packages or we explicitly want to flush
		// This prevents accidental wiping of data during boot if registration hasn't happened yet
		if ( empty( $this->data['packages'] ) && ! empty( get_option( $this->option_name ) ) ) {
			// If we have packages in DB but none in memory, don't save yet 
			// unless it's an explicit flush (which we can't easily detect here)
			// But for now, let's just be careful.
		}

		update_option( $this->option_name, json_encode( $this->data ) );
	}

	/**
	 * Register a group
	 */
	public function register_group( $slug, $args ) {
		$this->groups[$slug] = wp_parse_args( $args, array(
			'name'     => ucfirst( $slug ),
			'priority' => 10,
		) );
	}

	/**
	 * Get registered groups
	 */
	public function get_groups() {
		uasort( $this->groups, function( $a, $b ) {
			return $a['priority'] - $b['priority'];
		} );
		return $this->groups;
	}

	/**
	 * Register a component
	 */
	public function register( $type, $slug, $args ) {
		if ( ! isset( $this->data[$type] ) ) {
			$this->data[$type] = array();
		}

		// Version-based collision handling
		if ( isset( $this->data[$type][$slug] ) ) {
			$existing_version = isset( $this->data[$type][$slug]['version'] ) ? $this->data[$type][$slug]['version'] : '0';
			$new_version = isset( $args['version'] ) ? $args['version'] : '0';

			if ( version_compare( $new_version, $existing_version, '<' ) ) {
				return false; // Higher version already registered
			}
		}

		$args = apply_filters( "slipstream_register_{$type}_args", $args, $slug );

		$defaults = array(
			'group'         => $type === 'merge_tags' ? 'custom' : 'settings',
			'priority'      => 10,
			'count'         => null,
			'save_mapping'  => null, // New field for auto-save button
		);

		// Specific defaults for widgets
		if ( $type === 'widgets' ) {
			$defaults = array(
				'title'    => ucfirst( $slug ),
				'content'  => '', // Callback or string
				'priority' => 10,
				'width'    => 'full', // 'full' or 'half'
			);
		}

		// Specific defaults for welcome tabs
		if ( $type === 'welcome_tabs' ) {
			$defaults = array(
				'title'    => ucfirst( $slug ),
				'content'  => '', // Callback or string
				'priority' => 10,
				'active'   => false,
			);
		}

        // Keep persistent flags if not provided in args
        if ( isset( $this->data[$type][$slug] ) ) {
            $persistent_flags = array( 'active', 'note', 'checksum', 'path' );
            foreach ( $persistent_flags as $flag ) {
                if ( isset( $this->data[$type][$slug][$flag] ) && ! isset( $args[$flag] ) ) {
                    $args[$flag] = $this->data[$type][$slug][$flag];
                }
            }
        }

		$this->data[$type][$slug] = wp_parse_args( $args, $defaults );

		return true;
	}

	/**
	 * Register a Blueprint (Bundle of registrations)
	 *
	 * @param string $slug Unique slug for the blueprint.
	 * @param array  $registrations Array of registrations grouped by type.
	 */
	public function register_blueprint( $slug, $registrations ) {
		foreach ( $registrations as $type => $items ) {
			foreach ( $items as $item_slug => $args ) {
				$this->register( $type, $item_slug, $args );
			}
		}
	}

	/**
	 * Register a dashboard widget
	 */
	private function register_widget( $slug, $args ) {
		// For now, this just ensures it's in the data. 
		// Real registration might involve hooks if needed.
	}

	/**
	 * Register a settings panel
	 */
	private function register_panel( $slug, $args ) {
		// Ensures it's in the data for the SSPanel
	}

	/**
	 * Get registered data
	 */
	public function get_data( $type = null ) {
		$data = $this->data;
		if ( $type ) {
			$data = isset( $this->data[$type] ) ? $this->data[$type] : array();
		}

		/**
		 * Filter the registry data
		 * 
		 * @param array  $data The registry data.
		 * @param string $type The component type (e.g., 'panels', 'packages').
		 */
		return apply_filters( "slipstream_get_registry_{$type}", $data, $type );
	}

	/**
	 * Flush the registry
	 */
	public function flush() {
		$this->data = array();
		$this->save();
		
		// Re-trigger package loading to re-populate widgets if they use init hook
		do_action( 'slipstream_registry_flushed' );
	}

	/**
	 * Clear a specific type from registry (persistent)
	 */
	public function clear_type( $type ) {
		if ( isset( $this->data[$type] ) ) {
			unset( $this->data[$type] );
			$this->save();
		}
	}

	/**
	 * Clear a specific type from registry in memory only
	 */
	public function clear_type_in_memory( $type ) {
		if ( isset( $this->data[$type] ) ) {
			unset( $this->data[$type] );
		}
	}

	/**
	 * Register an administrator-only configuration callback for a package
	 *
	 * @param string $slug The package slug.
	 * @param array|callable $args Array of arguments (title, icon, callback) or just the callback for BC.
	 */
	public function register_admin_config( $slug, $args ) {
		if ( is_callable( $args ) ) {
			$args = array( 'callback' => $args );
		}

		$this->admin_config[$slug] = wp_parse_args( $args, array(
			'title'    => ucfirst( str_replace( '-', ' ', $slug ) ),
			'callback' => null,
			'icon'     => null,
		) );
	}

	/**
	 * Get all registered admin configurations
	 *
	 * @return array
	 */
	public function get_admin_configs() {
		return $this->admin_config;
	}

	/**
	 * Register an assignable content field for a package
	 *
	 * @param string $slug The package slug.
	 * @param array  $args Field arguments (name, description, merge_tag).
	 */
	public function register_content_field( $slug, $args ) {
		if ( ! isset( $this->content_fields[$slug] ) ) {
			$this->content_fields[$slug] = array();
		}

		$field_slug = str_replace( '-', '_', sanitize_title( $args['name'] ) );
		$this->content_fields[$slug][$field_slug] = wp_parse_args( $args, array(
			'name'        => '',
			'description' => '',
			'merge_tag'   => '',
		) );
	}

	/**
	 * Get all registered content fields
	 *
	 * @return array
	 */
	public function get_content_fields() {
		return $this->content_fields;
	}

	/**
	 * Add a capability to the Owner role (requested by a package)
	 *
	 * @param string|array $capability The capability slug or an array of capabilities.
	 */
	public function add_owner_capability( $capability ) {
		if ( is_array( $capability ) ) {
			$this->owner_capabilities = array_unique( array_merge( $this->owner_capabilities, $capability ) );
		} else {
			if ( ! in_array( $capability, $this->owner_capabilities ) ) {
				$this->owner_capabilities[] = $capability;
			}
		}
	}

	/**
	 * Helper to grant all capabilities for a post type to the Owner role
	 *
	 * @param string $post_type The post type slug.
	 */
	public function add_owner_capability_all( $post_type ) {
		$plural = $post_type . 's'; // Basic pluralization
		if ( substr( $post_type, -1 ) === 'y' ) {
			$plural = substr( $post_type, 0, -1 ) . 'ies';
		}

		$this->add_owner_capability( array(
			"edit_{$plural}",
			"publish_{$plural}",
			"delete_{$plural}",
			"edit_others_{$plural}",
			"delete_others_{$plural}",
			"read_private_{$plural}",
			"edit_private_{$plural}",
			"delete_private_{$plural}",
			"edit_published_{$plural}",
			"delete_published_{$plural}"
		) );
	}

	/**
	 * Helper to grant view/read capabilities for a post type to the Owner role
	 *
	 * @param string $post_type The post type slug.
	 */
	public function add_owner_capability_view( $post_type ) {
		$plural = $post_type . 's';
		if ( substr( $post_type, -1 ) === 'y' ) {
			$plural = substr( $post_type, 0, -1 ) . 'ies';
		}

		$this->add_owner_capability( array(
			"edit_{$plural}", // Need edit to see the list in some cases
			"read_private_{$plural}"
		) );
	}

	/**
	 * Register a callback function that can be used by other packages/components
	 *
	 * @param string $slug Unique slug for the callback.
	 * @param array  $args Arguments (name, callback, description, context).
	 */
	public function register_callback( $slug, $args ) {
		$this->callbacks[$slug] = wp_parse_args( $args, array(
			'name'        => $slug,
			'callback'    => null,
			'description' => '',
			'context'     => array(), // array of strings, e.g., array('team-members', 'social')
		) );
	}

	/**
	 * Get registered callbacks, optionally filtered by context
	 *
	 * @param string|array $context Optional context filter.
	 * @return array
	 */
	public function get_callbacks( $context = null ) {
		if ( $context === null ) {
			return $this->callbacks;
		}

		$filtered = array();
		$contexts = (array) $context;

		foreach ( $this->callbacks as $slug => $args ) {
			if ( empty( $args['context'] ) ) {
				$filtered[$slug] = $args;
				continue;
			}

			foreach ( $contexts as $ctx ) {
				if ( in_array( $ctx, (array) $args['context'] ) ) {
					$filtered[$slug] = $args;
					break;
				}
			}
		}

		return $filtered;
	}

	/**
	 * Get all requested owner capabilities
	 *
	 * @return array
	 */
	public function get_owner_capabilities() {
		return $this->owner_capabilities;
	}

	/**
	 * Helper to remove columns from a post type list table
	 *
	 * @param string $post_type The post type slug.
	 * @param array $columns The column keys to remove.
	 */
	public function remove_post_type_columns( $post_type, $columns ) {
		add_filter( "manage_{$post_type}_posts_columns", function( $current_columns ) use ( $columns ) {
			foreach ( $columns as $column ) {
				unset( $current_columns[$column] );
			}
			return $current_columns;
		} );
	}

	/**
	 * Helper to add columns to a post type list table
	 *
	 * @param string $post_type The post type slug.
	 * @param array $columns The columns to add [key => label].
	 * @param callable $callback The callback to render column content.
	 */
	public function add_post_type_columns( $post_type, $columns, $callback ) {
		add_filter( "manage_{$post_type}_posts_columns", function( $current_columns ) use ( $columns ) {
			return array_merge( $current_columns, $columns );
		} );

		add_action( "manage_{$post_type}_posts_custom_column", function( $column_key, $post_id ) use ( $columns, $callback ) {
			if ( array_key_exists( $column_key, $columns ) ) {
				call_user_func( $callback, $column_key, $post_id );
			}
		}, 10, 2 );
	}

	/**
	 * Helper to register an existing post type for the Slipstream Owner dashboard.
	 * This handles capabilities, status toggles, and adding a dashboard panel.
	 *
	 * @param string $post_type The post type slug.
	 * @param array  $args      {
	 *     Optional. Arguments for the registration.
	 *
	 *     @type string $name        The display name for the panel.
	 *     @type string $description The description for the panel.
	 *     @type string $icon        SVG icon for the panel.
	 *     @type string $icon_bg     Background color for the icon.
	 *     @type string $icon_color  Color for the icon.
	 *     @type int    $priority    Panel priority.
	 *     @type bool   $status_toggle Whether to enable the status toggle in the list table. Default true.
	 *     @type bool   $all_caps      Whether to grant all capabilities or just view. Default true.
	 * }
	 */
	public function register_post_type_for_owner( $post_type, $args = array() ) {
		$defaults = array(
			'name'          => ucfirst( $post_type ) . 's',
			'description'   => 'Manage your ' . $post_type . 's.',
			'icon'          => '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" /></svg>',
			'icon_bg'       => '#4f46e5',
			'icon_color'    => '#ffffff',
			'priority'      => 20,
			'status_toggle' => true,
			'all_caps'      => true,
		);

		$args = wp_parse_args( $args, $defaults );

		// 1. Grant capabilities
		if ( $args['all_caps'] ) {
			$this->add_owner_capability_all( $post_type );
		} else {
			$this->add_owner_capability_view( $post_type );
		}

		// 2. Register Panel
		$this->register( 'panels', $post_type, array(
			'name'        => $args['name'],
			'description' => $args['description'],
			'icon'        => $args['icon'],
			'icon_bg'     => $args['icon_bg'],
			'icon_color'  => $args['icon_color'],
			'url'         => admin_url( "edit.php?post_type={$post_type}" ),
			'group'       => 'content',
			'priority'    => $args['priority'],
			'count'       => function() use ( $post_type, $args ) {
				$count = wp_count_posts( $post_type )->publish;
				return $count > 0 ? '<span class="dml-badge" style="background: ' . $args['icon_bg'] . ';">' . $count . ' ' . $args['name'] . '</span>' : 0;
			},
		) );

		// 3. Status Toggle
		if ( $args['status_toggle'] ) {
			$this->register( 'status_toggle_post_types', $post_type, array( 'enabled' => true ) );
		}
	}
}
