<?php
/**
 * Router Class - Handles Lockdown & Shell Routing
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Router {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		Slipstream_Profile_UI::get_instance();
		add_action( 'admin_init', array( $this, 'handle_view_toggle' ) );
		add_action( 'admin_init', array( $this, 'lockdown' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ) );
		add_action( 'admin_head', array( $this, 'inject_css' ) );
		add_action( 'admin_head', array( $this, 'start_skin_buffering' ), 999 );
		add_action( 'admin_footer', array( $this, 'inject_skin_shell' ), 1 );
		add_filter( 'show_admin_bar', array( $this, 'hide_admin_bar' ) );
		add_action( 'init', array( $this, 'register_core_panels' ), 10 );
		add_action( 'add_meta_boxes', array( $this, 'hide_metaboxes' ), 999 );
		add_action( 'admin_head', array( $this, 'filter_editor_assets' ) );
		add_filter( 'admin_body_class', array( $this, 'add_admin_body_class' ) );
	}

	/**
	 * Add a body class to the admin screen for administrators in Slipstream view
	 */
	public function add_admin_body_class( $classes ) {
		if ( ! $this->is_owner() ) {
			return $classes;
		}

		if ( current_user_can( 'manage_options' ) ) {
			$classes .= ' slipstream-is-admin';
		}

		return $classes;
	}

	/**
	 * Handle toggling the Slipstream View for administrators
	 */
	public function handle_view_toggle() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		if ( isset( $_GET['slipstream_owner_view'] ) ) {
			$enabled = ( $_GET['slipstream_owner_view'] === '1' );
			update_user_meta( get_current_user_id(), 'slipstream_owner_view_enabled', $enabled ? '1' : '0' );
			
			// Redirect back without the toggle param
			$url = remove_query_arg( 'slipstream_owner_view' );
			wp_safe_redirect( $url );
			exit;
		}
	}

	/**
	 * Check if we should hijack the content for the dashboard
	 */
	public function should_hijack_dashboard() {
		if ( ! $this->is_owner() ) {
			return false;
		}

		if ( ! function_exists( 'get_current_screen' ) ) {
			return false;
		}

		$current_screen = get_current_screen();
		$is_dashboard = ( $current_screen && ( $current_screen->base === 'dashboard' || $current_screen->base === 'dashboard-network' ) );
		
		if ( ! $is_dashboard && ( basename( $_SERVER['PHP_SELF'] ) === 'index.php' || isset( $_GET['slipstream_preview'] ) ) && ! isset( $_GET['page'] ) ) {
			$is_dashboard = true;
		}

		return $is_dashboard;
	}

	/**
	 * End Output Buffering and discard the captured WP dashboard content
	 */
	public function end_content_hijack() {
		if ( ob_get_level() > 0 ) {
			ob_end_clean();
		}
	}

	/**
	 * Register core panels that should always be present
	 */
	public function register_core_panels() {
		$registry = Slipstream_Registry::get_instance();
		
		$user = wp_get_current_user();

		// My Profile (Hero Style)
		$registry->register( 'panels', 'profile', array(
			'name'        => $user->display_name,
			'description' => $user->user_email,
			'icon'        => '<div>' . strtoupper( substr( $user->display_name, 0, 1 ) ) . '</div>',
			'url'         => admin_url( 'admin.php?page=slipstream-profile' ),
			'group'       => 'user',
			'priority'    => 1,
			'style'       => 'hero',
		) );

		// Logout (Subtle Style)
		$registry->register( 'panels', 'logout', array(
			'name'     => __( 'Logout', 'slipstream' ),
			'url'      => wp_logout_url(),
			'group'    => 'user',
			'priority' => 100, // Bottom of the group
			'style'    => 'subtle',
		) );

		// Register Dynamic Content Panels for Pages
		$this->register_dynamic_content_panels();
	}

	/**
	 * Register panels for pages that have assigned content fields
	 */
	private function register_dynamic_content_panels() {
		$assigned = get_option( 'slipstream_assigned_content_fields', array() );
		if ( empty( $assigned ) ) {
			return;
		}

		$registry = Slipstream_Registry::get_instance();
		$pages_with_fields = array();

		// Collect all pages that have at least one field assigned
		foreach ( $assigned as $package_slug => $fields ) {
			foreach ( $fields as $field_slug => $page_ids ) {
				foreach ( $page_ids as $page_id ) {
					if ( ! in_array( $page_id, $pages_with_fields ) ) {
						$pages_with_fields[] = $page_id;
					}
				}
			}
		}

		// Sort pages by menu_order
		if ( ! empty( $pages_with_fields ) ) {
			$pages_query = new WP_Query( array(
				'post_type'      => 'page',
				'post__in'       => $pages_with_fields,
				'orderby'        => 'menu_order',
				'order'          => 'ASC',
				'posts_per_page' => -1,
			) );

			if ( $pages_query->have_posts() ) {
				foreach ( $pages_query->posts as $page ) {
					$registry->register( 'panels', 'page_content_' . $page->ID, array(
						'name'        => $page->post_title,
						'description' => sprintf( __( 'Manage content for the %s page.', 'slipstream' ), $page->post_title ),
						'icon'        => '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/><polyline points="14.5 2 14.5 8 20 8"/></svg>',
						'icon_bg'     => '#6366f1',
						'icon_color'  => '#ffffff',
						'url'         => admin_url( 'admin.php?page=slipstream-content-editor&post_id=' . $page->ID ),
						'group'       => 'pages',
						'priority'    => 5, // High priority in group
					) );
				}
			}
			wp_reset_postdata();
		}
	}

	/**
	 * Hide Yoast SEO and other unwanted elements from the custom content editor
	 */
	public function filter_editor_assets() {
		if ( ! function_exists( 'get_current_screen' ) ) {
			return;
		}

		$current_screen = get_current_screen();
		if ( $current_screen && $current_screen->id === 'admin_page_slipstream-content-editor' ) {
			// Add styles to hide standard WP elements if they leak in
			echo '<style>#wpadminbar, #adminmenumain, #wpfooter { display: none !important; }</style>';
		}
	}

	/**
	 * Helper to check if current user is an owner or admin previewing
	 */
	private function is_owner() {
		if ( ! is_user_logged_in() ) {
			return false;
		}

		$user = wp_get_current_user();
		if ( ! $user ) {
			return false;
		}

		// Explicit role check is the most reliable
		if ( in_array( 'owner', (array) $user->roles ) ) {
			return true;
		}

		// Allow admins to preview if they use the toggle or have it persisted
		if ( current_user_can( 'manage_options' ) || is_super_admin() ) {
			if ( isset( $_GET['slipstream_preview'] ) || get_user_meta( $user->ID, 'slipstream_owner_view_enabled', true ) === '1' ) {
				return true;
			}
		}

		// Also treat the custom content editor as an owner view for layout purposes
		if ( function_exists( 'get_current_screen' ) ) {
			$current_screen = get_current_screen();
			if ( $current_screen && $current_screen->id === 'admin_page_slipstream-content-editor' ) {
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Inject Hiding CSS into Admin Head
	 */
	public function inject_css() {
		if ( ! $this->is_owner() ) {
			return;
		}

		$brand_color = get_option( 'slipstream_brand_color', '#4f46e5' );
		$rgb = $this->hex_to_rgb( $brand_color );
		?>
		<style id="slipstream-hide-wp-admin">
			:root {
				--ss-brand-color: <?php echo esc_attr( $brand_color ); ?>;
				--ss-brand-color-rgb: <?php echo esc_attr( "{$rgb['r']}, {$rgb['g']}, {$rgb['b']}" ); ?>;
				--ss-brand-color-hover: <?php echo esc_attr( $this->adjust_brightness( $brand_color, -20 ) ); ?>;
			}

			/* Hide standard WP admin elements */
			#adminmenumain, #wpadminbar, #wpfooter, #screen-meta-links, .notice, #wp-auth-check-wrap, #wp-admin-bar-root-default { display: none !important; }
			#wpcontent { margin-left: 0 !important; padding-top: 0 !important; padding-left: 0 !important; }
			html.wp-toolbar { padding-top: 0 !important; }
			#wpbody-content { padding-bottom: 0 !important; }
			.update-nag, .settings-error, .is-dismissible { display: none !important; }
			#wpbody { padding-top: 0 !important; }
			
			/* Specifically hide dashboard widgets if we are on the dashboard */
			.index-php #dashboard-widgets-wrap, .index-network-php #dashboard-widgets-wrap { display: none !important; }
			.index-php #slipstream-content, .index-network-php #slipstream-content { display: none !important; }

			/* Ensure the layout is clean when hijacking */
			.index-php #wpbody-content, .index-network-php #wpbody-content { padding: 0 !important; }

			/* Ensure our shell is visible and looks right */
			#slipstream-shell {
				display: flex !important;
				visibility: visible !important;
			}
		</style>
		<script src="https://cdn.tailwindcss.com?plugins=forms,typography,aspect-ratio,line-clamp"></script>
		<?php
	}

	/**
	 * Enqueue Tailwind and custom styles
	 */
	public function enqueue_styles() {

        wp_enqueue_style( 'slipstream-shell', SLIPSTREAM_URL . 'assets/css/shell.css', array(), SLIPSTREAM_VERSION );
        wp_enqueue_script( 'slipstream-core', SLIPSTREAM_URL . 'assets/js/all-admin.js', array( 'jquery' ), SLIPSTREAM_VERSION, true );

		if ( ! function_exists( 'get_current_screen' ) ) {
			return;
		}

		$screen = get_current_screen();

		// Ensure Tailwind is loaded for specific pages even for non-Owners (Admins)
		if ( $screen && in_array( $screen->id, array( 'slipstream_page_slipstream-merge-tags', 'slipstream_page_slipstream-developer-config', 'toplevel_page_slipstream-packages' ) ) ) {
			add_action( 'admin_head', function() {
				echo '<script src="https://cdn.tailwindcss.com?plugins=forms,typography,aspect-ratio,line-clamp"></script>';
			} );
		}

		if ( ! $this->is_owner() ) {
			return;
		}
		// Tailwind is also loaded in admin_head for immediate effect

		// Load custom overrides
		wp_enqueue_style( 'slipstream-overrides', SLIPSTREAM_URL . 'assets/css/owner-overrides.css', array(), SLIPSTREAM_VERSION );
		wp_enqueue_style( 'slipstream-acf', SLIPSTREAM_URL . 'assets/css/acf-owner.css', array(), SLIPSTREAM_VERSION );

		// Load post editor specific JS
		if ( function_exists( 'get_current_screen' ) ) {
			$screen = get_current_screen();
			if ( $screen && ( $screen->base === 'post' ) ) {
				wp_enqueue_script( 'slipstream-post-editor', SLIPSTREAM_URL . 'assets/js/post-editor.js', array( 'jquery' ), SLIPSTREAM_VERSION, true );
			}
		}
	}

	/**
	 * Hide Admin Bar for Owner role
	 */
	public function hide_admin_bar( $show ) {
		if ( $this->is_owner() ) {
			return false;
		}
		return $show;
	}

	/**
	 * Hide configured metaboxes for Owner
	 */
	public function hide_metaboxes() {
		if ( ! $this->is_owner() ) {
			return;
		}

		$hidden_metaboxes = get_option( 'slipstream_hidden_metaboxes', '' );
		if ( empty( $hidden_metaboxes ) ) {
			return;
		}

		$metaboxes_array = array_filter( array_map( 'trim', explode( "\n", $hidden_metaboxes ) ) );
		
		foreach ( $metaboxes_array as $metabox_id ) {
			// We don't know the screen or context easily here, so we remove from all
			// Or we could try to get current screen
			if ( function_exists( 'get_current_screen' ) ) {
				$screen = get_current_screen();
				if ( $screen ) {
					remove_meta_box( $metabox_id, $screen->id, 'normal' );
					remove_meta_box( $metabox_id, $screen->id, 'advanced' );
					remove_meta_box( $metabox_id, $screen->id, 'side' );
				}
			}
		}
	}

	/**
	 * Redirect Owner role from restricted areas
	 */
	public function lockdown() {
		if ( ! $this->is_owner() ) {
			return;
		}

		// Use $_SERVER['PHP_SELF'] as a fallback if get_current_screen() is not ready
		$current_page = basename( $_SERVER['PHP_SELF'] );
		
		// List of restricted pages
		$restricted = array( 
			'plugins.php', 
			'themes.php', 
			'options-general.php', 
			'tools.php', 
			'update-core.php',
			'options.php',
			'import.php',
			'export.php',
			'profile.php',
			'user-edit.php'
		);
		
		if ( in_array( $current_page, $restricted ) ) {
			wp_safe_redirect( admin_url() );
			exit;
		}
	}

	/**
	 * Start buffering the entire admin page to capture content
	 */
	public function start_skin_buffering() {
		if ( ! $this->is_owner() ) {
			return;
		}
		ob_start();
	}

	/**
	 * Inject custom Tailwind Shell and render captured content
	 */
	public function inject_skin_shell() {
		if ( ! $this->is_owner() ) {
			return;
		}

		// Get the captured content
		$wp_content = '';
		if ( ob_get_level() > 0 ) {
			$wp_content = ob_get_clean();
		}

		// If on dashboard, we might want to discard the captured content anyway
		if ( $this->should_hijack_dashboard() ) {
			$wp_content = '';
		}

		// Include the shell which will use $wp_content
		include SLIPSTREAM_PATH . 'templates/shell.php';
		
		// The shell now handles its own footer part or we include it
 	include SLIPSTREAM_PATH . 'templates/parts/footer.php';
	}

	/**
	 * Adjust the brightness of a hex color
	 */
	private function adjust_brightness( $hex, $steps ) {
		$steps = max( -255, min( 255, $steps ) );
		$hex = str_replace( '#', '', $hex );
		if ( strlen( $hex ) == 3 ) {
			$hex = str_repeat( substr( $hex, 0, 1 ), 2 ) . str_repeat( substr( $hex, 1, 1 ), 2 ) . str_repeat( substr( $hex, 2, 1 ), 2 );
		}
		$color_parts = str_split( $hex, 2 );
		$return = '#';
		foreach ( $color_parts as $color ) {
			$color = hexdec( $color );
			$color = max( 0, min( 255, $color + $steps ) );
			$return .= str_pad( dechex( $color ), 2, '0', STR_PAD_LEFT );
		}
		return $return;
	}

	/**
	 * Convert hex color to RGB
	 */
	private function hex_to_rgb( $hex ) {
		$hex = str_replace( '#', '', $hex );
		if ( strlen( $hex ) == 3 ) {
			$r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
			$g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
			$b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
		} else {
			$r = hexdec( substr( $hex, 0, 2 ) );
			$g = hexdec( substr( $hex, 2, 2 ) );
			$b = hexdec( substr( $hex, 4, 2 ) );
		}
		return array( 'r' => $r, 'g' => $g, 'b' => $b );
	}
}
