<?php
/**
 * AJAX Search Handler for Slipstream
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class Slipstream_Search {

	private static $instance;

	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'wp_ajax_slipstream_search', array( $this, 'ajax_search' ) );
		add_action( 'wp_ajax_slipstream_panel_toggle', array( $this, 'ajax_panel_toggle' ) );
		add_action( 'wp_ajax_slipstream_address_autocomplete', array( $this, 'ajax_address_autocomplete' ) );
		add_action( 'wp_ajax_nopriv_slipstream_address_autocomplete', array( $this, 'ajax_address_autocomplete' ) );
	}

	/**
	 * AJAX Address Autocomplete Handler
	 */
	public function ajax_address_autocomplete() {
		$term = isset( $_GET['term'] ) ? sanitize_text_field( $_GET['term'] ) : '';
		
		if ( empty( $term ) ) {
			wp_send_json( array() );
		}

		$middleware_url = "https://api.makedev.com.au/google_mw/autocomplete.php?input=" . urlencode( $term );

		$response = wp_remote_get( $middleware_url, array( 'timeout' => 5 ) );

		if ( is_wp_error( $response ) ) {
			wp_send_json( array( 'error' => $response->get_error_message() ), 500 );
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		status_header( $code );
		header( 'Content-Type: application/json' );
		echo $body;
		wp_die();
	}

	/**
	 * AJAX Panel Toggle Handler
	 */
	public function ajax_panel_toggle() {
		$panel_slug = isset( $_POST['panel_slug'] ) ? sanitize_text_field( $_POST['panel_slug'] ) : '';
		
		check_ajax_referer( 'slipstream_toggle_' . $panel_slug );

		if ( ! current_user_can( 'is_slipstream_owner' ) && ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}

		$state = isset( $_POST['state'] ) ? sanitize_text_field( $_POST['state'] ) : 'off';
		$toggle_action = isset( $_POST['toggle_action'] ) ? sanitize_text_field( $_POST['toggle_action'] ) : '';

		if ( empty( $toggle_action ) ) {
			wp_send_json_error( 'No toggle action defined for this panel.' );
		}

		// Fire the action that packages should hook into
		// Example: do_action( 'slipstream_toggle_maintenance_mode', 'on' );
		do_action( $toggle_action, $state, $panel_slug );

		wp_send_json_success();
	}

	/**
	 * AJAX Search Handler
	 */
	public function ajax_search() {
		// Try 'nonce' or '_ajax_nonce'
		$nonce = isset( $_REQUEST['nonce'] ) ? $_REQUEST['nonce'] : ( isset( $_REQUEST['_ajax_nonce'] ) ? $_REQUEST['_ajax_nonce'] : '' );
		check_ajax_referer( 'slipstream_search', $nonce ? null : 'nonce' );

		$query = isset( $_GET['term'] ) ? sanitize_text_field( $_GET['term'] ) : ( isset( $_GET['q'] ) ? sanitize_text_field( $_GET['q'] ) : '' );

		if ( empty( $query ) ) {
			wp_send_json_success( array( 'results' => array() ) );
		}

		$results = array();

		// 1. Search Panels
		$registry = Slipstream_Registry::get_instance();
		$panels = $registry->get_data( 'panels' );

		foreach ( $panels as $slug => $panel ) {
			if ( stripos( $panel['name'], $query ) !== false || ( isset( $panel['description'] ) && stripos( $panel['description'], $query ) !== false ) ) {
				$results[] = array(
					'type' => 'Setting',
					'title' => $panel['name'],
					'description' => isset( $panel['description'] ) ? $panel['description'] : '',
					'url' => $panel['url'],
					'icon' => isset( $panel['icon'] ) ? $panel['icon'] : '⚙️',
					'panel_slug' => $slug,
				);
			}
		}

		// 2. Search Post Titles
		$posts_query = new WP_Query( array(
			's' => $query,
			'post_type' => 'any',
			'posts_per_page' => 10,
			'post_status' => 'publish',
		) );

		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$post_type = get_post_type();
				$post_type_obj = get_post_type_object( $post_type );
				
				// Find matching panel slug for this post type
				$panel_slug = '';
				foreach ( $panels as $p_slug => $p_args ) {
					// Post types usually have URLs like edit.php?post_type=XXX
					if ( isset( $p_args['url'] ) && ( strpos( $p_args['url'], 'post_type=' . $post_type ) !== false || ( $post_type === 'post' && strpos( $p_args['url'], 'edit.php' ) !== false && strpos( $p_args['url'], 'post_type=' ) === false ) ) ) {
						$panel_slug = $p_slug;
						break;
					}
				}

				$results[] = array(
					'type' => 'post',
					'id' => get_the_ID(),
					'post_label' => $post_type_obj->labels->singular_name,
					'title' => get_the_title(),
					'description' => get_the_excerpt(),
					'url' => html_entity_decode( get_edit_post_link( get_the_ID() ) ),
					'icon' => '📄',
					'panel_slug' => $panel_slug,
				);
			}
			wp_reset_postdata();
		}

		wp_send_json_success( array( 'results' => $results ) );
	}
}
