<?php
/**
 * Plugin Name: Slipstream
 * Description: A "Headless-Lite" admin interface and package manager for WordPress.
 * Version: 2.3.29
 * Author: Mitchell Bray
 * License: GPL-2.0+
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load constants
$plugin_data = get_file_data( __FILE__, array( 'Version' => 'Version' ) );
 define( 'SLIPSTREAM_VERSION', '2.3.29' );

define( 'SLIPSTREAM_PATH', plugin_dir_path( __FILE__ ) );
define( 'SLIPSTREAM_URL', plugin_dir_url( __FILE__ ) );

// Load Composer autoloader if it exists
if ( file_exists( SLIPSTREAM_PATH . 'vendor/autoload.php' ) ) {
	require_once SLIPSTREAM_PATH . 'vendor/autoload.php';
}

// Use WP constants if defined, otherwise use fallbacks for static analysis
$wp_content_dir = defined( 'WP_CONTENT_DIR' ) ? WP_CONTENT_DIR : ABSPATH . 'wp-content';
define( 'SLIPSTREAM_PACKAGES_DIR', $wp_content_dir . '/slipstream-packages' );
define( 'SLIPSTREAM_PACKAGES_URL', content_url( '/slipstream-packages' ) );

/**
 * Get the Slipstream packages directory path.
 */
function slipstream_get_packages_dir() {
	return defined( 'SLIPSTREAM_PACKAGES_DIR' ) ? SLIPSTREAM_PACKAGES_DIR : ABSPATH . 'wp-content/slipstream-packages';
}

/**
 * Get the Slipstream packages directory URL.
 */
function slipstream_get_packages_url() {
	return defined( 'SLIPSTREAM_PACKAGES_URL' ) ? SLIPSTREAM_PACKAGES_URL : content_url( '/slipstream-packages' );
}

/**
 * Convert a file path within a Slipstream package to a URL.
 */
function slipstream_package_url( $path ) {
	$packages_dir = slipstream_get_packages_dir();
	$packages_url = slipstream_get_packages_url();

	// Normalize paths
	$path = wp_normalize_path( $path );
	$packages_dir = wp_normalize_path( $packages_dir );
	$slipstream_path = wp_normalize_path( SLIPSTREAM_PATH );

	if ( strpos( $path, $packages_dir ) !== false ) {
		return str_replace( $packages_dir, $packages_url, $path );
	}

	// Handle development packages in SLIPSTREAM_PATH/.server/ or examples/
	if ( strpos( $path, $slipstream_path ) !== false ) {
		return str_replace( $slipstream_path, SLIPSTREAM_URL, $path );
	}

	return $path;
}

/**
 * Main Slipstream Class
 */
final class Slipstream_Core {

	/**
	 * Instance of this class
	 * @var Slipstream_Core
	 */
	private static $instance;

	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->includes();
		$this->init_hooks();

		// Immediately initialize core handlers that need to catch early hooks
		Slipstream_Menu_Handler::get_instance();

		add_filter( 'edit_posts_per_page', array( $this, 'set_posts_per_page' ) );
	}

	public function set_posts_per_page( $posts_per_page ) {
		return 100;
	}

	/**
	 * Include required files
	 */
	private function includes() {
		require_once SLIPSTREAM_PATH . 'includes/registry.php';
		require_once SLIPSTREAM_PATH . 'includes/router.php';
		require_once SLIPSTREAM_PATH . 'includes/package-loader.php';
		require_once SLIPSTREAM_PATH . 'includes/package-manager.php';
		require_once SLIPSTREAM_PATH . 'includes/acf-handler.php';
		require_once SLIPSTREAM_PATH . 'includes/admin-ui.php';
		require_once SLIPSTREAM_PATH . 'includes/assets.php';
		require_once SLIPSTREAM_PATH . 'includes/profile-ui.php';
		require_once SLIPSTREAM_PATH . 'includes/search.php';
		require_once SLIPSTREAM_PATH . 'includes/merge-tags.php';
		require_once SLIPSTREAM_PATH . 'includes/widgets.php';
		require_once SLIPSTREAM_PATH . 'includes/menu-handler.php';
		require_once SLIPSTREAM_PATH . 'includes/activity-log.php';
		require_once SLIPSTREAM_PATH . 'includes/api.php';
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

 	add_action( 'plugins_loaded', array( $this, 'boot' ) );
		add_action( 'init', array( $this, 'grant_owner_capabilities' ), 999 );
		add_filter( 'user_has_cap', array( $this, 'filter_owner_capabilities' ), 10, 4 );

		// Hook for 3rd party plugins to register their components
		add_action( 'init', function() {
			/**
			 * Triggered when Slipstream is ready for registrations.
			 * Use this to register panels, widgets, merge tags, etc.
			 * 
			 * @param Slipstream_Registry $registry The registry instance.
			 */
			do_action( 'slipstream_init', Slipstream_Registry::get_instance() );
		}, 15 );
	}

	/**
	 * Filter capabilities for the Owner role dynamically
	 */
	public function filter_owner_capabilities( $allcaps, $caps, $args, $user ) {
		if ( ! $user || ! in_array( 'owner', (array) $user->roles ) ) {
			return $allcaps;
		}

		$registry = Slipstream_Registry::get_instance();
		$dynamic_caps = $registry->get_owner_capabilities();

		foreach ( $dynamic_caps as $cap ) {
			$allcaps[$cap] = true;
		}

		return $allcaps;
	}

	/**
	 * Boot the plugin components
	 */
	public function boot() {
		// Initialize Singleton classes
		$registry = Slipstream_Registry::get_instance();
		Slipstream_Router::get_instance();
		Slipstream_Package_Loader::get_instance();
		Slipstream_Package_Manager::get_instance();
		Slipstream_ACF_Handler::get_instance();
		Slipstream_Admin_UI::get_instance();
		Slipstream_Search::get_instance();
		Slipstream_Merge_Tags::get_instance();
		Slipstream_Widgets::get_instance();
		Slipstream_Activity_Log::get_instance();
		Slipstream_API::get_instance();

		// Ensure Owner role has the basic capabilities
		$role = get_role( 'owner' );
		if ( $role ) {
			if ( ! $role->has_cap( 'is_slipstream_owner' ) ) {
				$role->add_cap( 'is_slipstream_owner' );
			}
			// Ensure they can at least read (required for some admin areas)
			if ( ! $role->has_cap( 'read' ) ) {
				$role->add_cap( 'read' );
			}
		}
	}

	/**
	 * Grant dynamic capabilities to the Owner role
	 * Runs on 'init' priority 999 to ensure all packages have registered their needs
	 */
	public function grant_owner_capabilities() {
		$registry = Slipstream_Registry::get_instance();
		$role = get_role( 'owner' );
		
		if ( ! $role ) {
			return;
		}

		// Also grant basic post capabilities for standard posts/pages if needed
		$this->add_base_post_caps( $role );

		// Add dynamic capabilities requested by packages
		$dynamic_caps = $registry->get_owner_capabilities();
		foreach ( $dynamic_caps as $cap ) {
			if ( ! $role->has_cap( $cap ) ) {
				$role->add_cap( $cap );
			}
		}
	}

	/**
	 * Ensure the Owner role has basic capabilities for posts and pages
	 */
	private function add_base_post_caps( $role ) {
		$base_caps = array(
			'edit_posts', 'publish_posts', 'delete_posts', 'edit_published_posts', 'delete_published_posts',
			'edit_pages', 'publish_pages', 'delete_pages', 'edit_published_pages', 'delete_published_pages'
		);
		foreach ( $base_caps as $cap ) {
			if ( ! $role->has_cap( $cap ) ) {
				$role->add_cap( $cap );
			}
		}
	}

	/**
	 * Activation logic
	 */
	public function activate() {
		// Create Owner role
		add_role( 'owner', __( 'Owner' ), array(
			'read'           => true,
			'upload_files'   => true,
			'is_slipstream_owner' => true,
		) );

		// Create packages directory
		if ( ! file_exists( SLIPSTREAM_PACKAGES_DIR ) ) {
			wp_mkdir_p( SLIPSTREAM_PACKAGES_DIR );
			
			// Set permissions
			@chmod( SLIPSTREAM_PACKAGES_DIR, 0755 );

			// Create .htaccess to prevent direct PHP execution
			$htaccess_content = "Options -Indexes\n<Files *.php>\ndeny from all\n</Files>";
			file_put_contents( SLIPSTREAM_PACKAGES_DIR . '/.htaccess', $htaccess_content );
		}
	}

	/**
	 * Deactivation logic
	 */
	public function deactivate() {
		// Remove Owner role if needed? (Decided in brief to remove on deactivation)
		remove_role( 'owner' );
	}
}

/**
 * Initialize the plugin
 */
function slipstream() {
	return Slipstream_Core::get_instance();
}

slipstream();


////////////////////////// UPDATES //////////////////////////
// ID, Key, __FILE__
if ( class_exists( 'makeUpdate' ) && get_option( 'slipstream_locked_down' ) !== 'yes' ) {
    try {
        $updater = new makeUpdate("9555", "xjr5waw5dwe4fae_ZJR", __FILE__);
    } catch (Exception $e) {
        // Silently ignore updater initialization errors to avoid breaking the plugin
    }
}